package com.biz.crm.mdm.business.customer.local.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.mdm.business.customer.sdk.dto.CustomerQueryDto;
import com.biz.crm.mdm.business.customer.sdk.dto.CustomerSelectDto;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 客户信息管理
 *
 * @author ning.zhang
 * @date 2021/10/15
 */
@Slf4j
@RestController
@RequestMapping("/v1/customer/customer")
@Api(tags = "客户信息: CustomerVo: 客户管理")
public class CustomerVoController {

  @Autowired(required = false)
  private CustomerVoService customerVoService;

  @ApiOperation(value = "通过客户ID或者客户编码查询客户信息详情(包含客户信息所有关联数据)")
  @GetMapping("/findDetailsByIdOrCode")
  public Result<CustomerVo> findDetailsByIdOrCode(
      @RequestParam(value = "id", required = false) @ApiParam(name = "id", value = "主键ID") String id
      ,
      @RequestParam(value = "customerCode", required = false) @ApiParam(name = "customerCode", value = "客户编码") String customerCode) {
    try {
      return Result.ok(customerVoService.findDetailsByIdOrCode(id, customerCode));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  @ApiOperation(value = "客户信息下拉框分页列表")
  @GetMapping("/findByCustomerSelectDto")
  public Result<Page<CustomerVo>> findByCustomerSelectDto(@PageableDefault(50) Pageable pageable,
      @ApiParam(name = "CustomerSelectDto", value = "下拉分页Dto") CustomerSelectDto dto) {
    try {
      return Result.ok(customerVoService.findByCustomerSelectDto(pageable, dto));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据企业组织编码获取匹配的客户信息
   *
   * @param orgCodeList 组织编码集合
   * @return 客户信息集合
   */
  @ApiOperation(value = "根据企业组织编码获取匹配的客户信息")
  @GetMapping("/findByOrgCodes")
  public Result<List<CustomerVo>> findByOrgCodes(
      @RequestParam("orgCodeList") List<String> orgCodeList) {
    try {
      return Result.ok(customerVoService.findByOrgCodes(orgCodeList));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 判断客户是否在组织内
   *
   * @param customerCode 客户代码
   * @param orgCodeIn    组织代码
   * @param orgCodeNotIn org代码不
   * @return {@link Result}<{@link Boolean}>
   */
  @ApiOperation(value = "判断客户是否在组织内")
  @GetMapping("/existByCustomer7OrgIn7OrgNotIn")
  public Result<Boolean> existByCustomer7OrgIn7OrgNotIn(
      @RequestParam("customerCode") String customerCode,
      @RequestParam("orgCodeIn") List<String> orgCodeIn,
      @RequestParam("orgCodeNotIn") List<String> orgCodeNotIn) {
    try {
      Boolean isExist = customerVoService.existByCustomer7OrgIn7OrgNotIn(customerCode, orgCodeIn, orgCodeNotIn);
      return Result.ok(isExist);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据客户编码集合获取对应的客户信息
   *
   * @param customerCodeList 客户编码集合
   * @return 客户信息集合
   */
  @ApiOperation(value = "根据客户编码集合获取对应的客户信息")
  @GetMapping("/findByCustomerCodes")
  public Result<List<CustomerVo>> findByCustomerCodes(
      @RequestParam("customerCodeList") List<String> customerCodeList) {
    try {
      return Result.ok(customerVoService.findByCustomerCodes(customerCodeList));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 判断客户是否存在
   *
   * @param customerCode 客户编码
   * @return 客户信息集合
   */
  @ApiOperation(value = "判断客户是否存在")
  @GetMapping("/existByCustomerCode")
  public Result<Boolean> existByCustomerCode(
      @RequestParam("customerCode") String customerCode) {
    try {
      return Result.ok(customerVoService.existByCustomerCode(customerCode));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据客户编码集合获取对应的客户信息-包含主信息+组织信息
   *
   * @param customerCodeSet 客户编码集合
   * @return 客户信息集合
   */
  @ApiOperation(value = "根据客户编码集合获取对应的客户信息-包含主信息+组织信息")
  @GetMapping("/findForPriceByCustomerCodes")
  public Result<List<CustomerVo>> findForPriceByCustomerCodes(
      @RequestParam("customerCodeSet") Set<String> customerCodeSet) {
    try {
      return Result.ok(customerVoService.findForPriceByCustomerCodes(customerCodeSet));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据渠道编码集合获取对应的客户信息
   *
   * @param channelList 渠道集合
   * @return 客户信息集合
   */
  @ApiOperation(value = "根据渠道编码集合获取对应的客户信息")
  @GetMapping("/findByChannels")
  public Result<List<CustomerVo>> findByChannels(@RequestParam("channelList") List<String> channelList) {
    try {
      return Result.ok(customerVoService.findByChannels(channelList));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 判断客户是否在渠道内
   *
   * @param channelList 渠道集合
   * @return 客户信息集合
   */
  @ApiOperation(value = "判断客户是否在渠道内")
  @GetMapping("/existByCustomerCodeAndChannels")
  public Result<Boolean> existByCustomerCodeAndChannels(@RequestParam("channelList") List<String> channelList, @RequestParam("customerCode") String customerCode) {
    try {
      return Result.ok(customerVoService.existByCustomerCodeAndChannels(channelList, customerCode));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据客户类型集合获取对应的客户信息
   *
   * @param typeList 客户类型集合
   * @return 客户信息集合
   */
  @ApiOperation(value = "根据渠道编码集合获取对应的客户信息")
  @GetMapping("/findByTypes")
  public Result<List<CustomerVo>> findByTypes(@RequestParam("typeList") List<String> typeList) {
    try {
      return Result.ok(customerVoService.findByTypes(typeList));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据组织集合、渠道集合、标签集合查询客户编码集合
   *
   * @param orgCodes 组织集合
   * @param channels 渠道集合
   * @param tags 标签集合
   * @return 客户编码集合
   */
  @ApiOperation(value = "根据渠道编码集合获取对应的客户信息")
  @GetMapping("/findCustomerCodesByOrgCodesAndChannelsAndTags")
  public Result<Set<String>> findCustomerCodesByOrgCodesAndChannelsAndTags(@RequestParam(value = "orgCodes", required=false) List<String> orgCodes,
                                                                           @RequestParam(value = "channels", required=false) List<String> channels,
                                                                           @RequestParam(value = "tags", required=false) List<String> tags) {
    try {
      return Result.ok(customerVoService.findCustomerCodesByOrgCodesAndChannelsAndTags(orgCodes, channels, tags));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据CustomerQueryDto获取对应的客户编码集合
   *
   * @param dto
   * @return
   */
  @ApiOperation(value = "根据CustomerQueryDto获取对应的客户编码集合")
  @PostMapping("/findByCustomerQueryDto")
  public Result<Set<String>> findByCustomerQueryDto(@RequestBody CustomerQueryDto dto) {
    try {
      return Result.ok(this.customerVoService.findByCustomerQueryDto(dto));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据组织获取组织及下级所有的审核通过且未删除的经销商信息
   *
   * @param orgCodes
   * @return
   */
  @ApiOperation(value = "根据组织获取组织及下级所有的审核通过且未删除的经销商信息")
  @GetMapping("/findAllowSaleCustomerByOrgCodes")
  public Result<Map<String, Set<String>>> findAllowSaleCustomerByOrgCodes(
      @RequestParam("orgCodes") Set<String> orgCodes) {
    try {
      return Result.ok(this.customerVoService.findAllowSaleCustomerByOrgCodes(orgCodes));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }
}
