package com.biz.crm.mdm.business.customer.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.LoginUserDetailsForEMS;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mdm.business.customer.local.entity.CustomerAddressEntity;
import com.biz.crm.mdm.business.customer.local.repository.CustomerAddressRepository;
import com.biz.crm.mdm.business.customer.local.service.CustomerAddressService;
import com.biz.crm.mdm.business.customer.sdk.dto.CustomerAddressPageDto;
import com.biz.crm.mdm.business.customer.sdk.dto.CustomerAddressQueryDto;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerAddressVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerAddressVo;
import com.biz.crm.mdm.business.region.sdk.service.RegionVoService;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * 客户收货地址信息VO服务接口实现类
 *
 * @author ning.zhang
 * @date 2021/10/28
 */
@Service
public class CustomerAddressVoServiceImpl implements CustomerAddressVoService {

  @Autowired(required = false)
  private CustomerAddressService customerAddressService;
  @Autowired(required = false)
  private RegionVoService regionVoService;
  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private CustomerAddressRepository customerAddressRepository;

  @Autowired(required = false)
  private LoginUserService loginUserService;

  @Override
  public Page<CustomerAddressVo> findByConditions(Pageable pageable, CustomerAddressPageDto dto) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    dto = ObjectUtils.defaultIfNull(dto, new CustomerAddressPageDto());
    dto.setTenantCode(TenantUtils.getTenantCode());
    Page<CustomerAddressEntity> entityPage = this.customerAddressService.findByConditions(pageable, dto);
    List<CustomerAddressEntity> entities = entityPage.getRecords();
    Page<CustomerAddressVo> pageResult = new Page<>(entityPage.getCurrent(), entityPage.getSize(), entityPage.getTotal());
    pageResult.setRecords(covertEntityToVo(entities));
    return pageResult;
  }

  @Override
  public CustomerAddressVo findDetailsById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    CustomerAddressEntity entity = this.customerAddressService.findById(id);
    if (Objects.isNull(entity)) {
      return null;
    }
    return covertEntityToVo(Lists.newArrayList(entity)).get(0);
  }

  @Override
  public List<CustomerAddressVo> findByCustomerAddressQueryDto(CustomerAddressQueryDto dto) {
    if (Objects.isNull(dto) || StringUtils.isBlank(dto.getCustomerCode())) {
      return Lists.newLinkedList();
    }
    dto.setTenantCode(TenantUtils.getTenantCode());
    List<CustomerAddressEntity> entities = this.customerAddressRepository.findByCustomerAddressQueryDto(dto);
    return this.covertEntityToVo(entities);
  }

  /**
   * 客户收货地址entity列表转vo列表
   *
   * @param entities 客户收货地址entity列表
   * @return 客户收货地址vo列表
   */
  private List<CustomerAddressVo> covertEntityToVo(List<CustomerAddressEntity> entities) {
    if (CollectionUtils.isEmpty(entities)) {
      return Lists.newArrayList();
    }
    List<CustomerAddressVo> list = Lists.newArrayList(this.nebulaToolkitService.copyCollectionByWhiteList(entities, CustomerAddressEntity.class
        , CustomerAddressVo.class, HashSet.class, ArrayList.class));
    List<String> regionCodes = Lists.newArrayList();
    list.forEach(vo -> {
      if (StringUtils.isNotBlank(vo.getCityCode())) {
        regionCodes.add(vo.getCityCode());
      }
      if (StringUtils.isNotBlank(vo.getDistrictCode())) {
        regionCodes.add(vo.getDistrictCode());
      }
      if (StringUtils.isNotBlank(vo.getProvinceCode())) {
        regionCodes.add(vo.getProvinceCode());
      }
    });
    if (CollectionUtils.isEmpty(regionCodes)) {
      return Lists.newArrayList();
    }
    Map<String, String> regionMap = this.regionVoService.findRegionCodeNameMap(regionCodes);
    list.forEach(vo -> {
      vo.setCityName(regionMap.get(vo.getCityCode()));
      vo.setDistrictName(regionMap.get(vo.getDistrictCode()));
      vo.setProvinceName(regionMap.get(vo.getProvinceCode()));
    });
    return list;
  }

  /**
   * 商城-客户收货地址分页查询
   * @param pageable
   * @param dto
   * @return {@link Page< CustomerAddressVo>}
   */
  @Override
  public Page<CustomerAddressVo> findCustomerAddressByCurrentCustomer(Pageable pageable, CustomerAddressPageDto dto) {
    LoginUserDetailsForEMS loginUserDetails = this.loginUserService.getLoginDetails(LoginUserDetailsForEMS.class);
    if (StringUtils.isBlank(loginUserDetails.getCustomerCode())) {
      Page<CustomerAddressVo> pageResult = new Page<>(0L, 0L, 0L);
      return pageResult;
    }
    dto.setCustomerCode(loginUserDetails.getCustomerCode());
    dto.setTenantCode(TenantUtils.getTenantCode());
    return this.findByConditions(pageable,dto);
  }
}
