package com.biz.crm.mdm.business.customer.local.service.internal;

import com.biz.crm.mdm.business.customer.local.entity.CustomerEntity;
import com.biz.crm.mdm.business.customer.local.entity.CustomerROrgEntity;
import com.biz.crm.mdm.business.customer.local.repository.CustomerROrgRepository;
import com.biz.crm.mdm.business.customer.local.repository.CustomerRepository;
import com.biz.crm.mdm.business.customer.local.service.CustomerROrgService;
import com.biz.crm.mdm.business.customer.sdk.dto.CustomerRelateOrgEventDto;
import com.biz.crm.mdm.business.customer.sdk.event.CustomerRelateOrgEventListener;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Set;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 客户与企业组织关联表服务实现类
 *
 * @author ning.zhang
 * @date 2021-11-10 10:19:10
 */
@Slf4j
@Service("customerROrgService")
public class CustomerROrgServiceImpl implements CustomerROrgService {

  @Autowired(required = false)
  private CustomerROrgRepository customerROrgRepository;

  @Autowired(required = false) private CustomerRepository customerRepository;

  @Autowired(required = false) private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false) private NebulaNetEventClient nebulaNetEventClient;

  @Override
  @Transactional
  public void rebindCustomerCode(String orgCode, String customerCode) {
    Validate.notBlank(customerCode, "客户编码不能为空");
    String tenantCode = TenantUtils.getTenantCode();
    this.customerROrgRepository.deleteByCustomerCode(customerCode, tenantCode);
    if (StringUtils.isBlank(orgCode)) {
      return;
    }
    //过滤掉企业组织编码为空的数据
    List<CustomerROrgEntity> entities = Arrays.stream(orgCode.split(","))
        .filter(StringUtils::isNotBlank)
        .map(code -> {
          CustomerROrgEntity entity = new CustomerROrgEntity();
          entity.setOrgCode(code);
          entity.setCustomerCode(customerCode);
          entity.setTenantCode(tenantCode);
          return entity;
        }).collect(Collectors.toList());
    if (CollectionUtils.isEmpty(entities)) {
      return;
    }
    this.customerROrgRepository.saveBatch(entities);
  }

  @Override
  @Transactional
  public void rebindOrgCode(String orgCode, String originOrgCode, List<String> customerCodeList) {
    this.customerROrgRepository.rebindOrgCode(orgCode, originOrgCode, customerCodeList, TenantUtils.getTenantCode());
    final List<CustomerEntity> customerEntityList =
        this.customerRepository.findByCodes(customerCodeList, TenantUtils.getTenantCode());
    if (CollectionUtils.isEmpty(customerEntityList)) {
      return;
    }
    List<CustomerVo> voList =
        (List<CustomerVo>)
            this.nebulaToolkitService.copyCollectionByBlankList(
                customerEntityList,
                CustomerEntity.class,
                CustomerVo.class,
                HashSet.class,
                ArrayList.class);
    for (CustomerVo item : voList) {
      CustomerRelateOrgEventDto eventDto = new CustomerRelateOrgEventDto();
      eventDto.setOldOrgCode(originOrgCode);
      eventDto.setNewOrgCode(orgCode);
      eventDto.setCustomerVo(item);
      SerializableBiConsumer<CustomerRelateOrgEventListener, CustomerRelateOrgEventDto> onRebind =
          CustomerRelateOrgEventListener::onRebind;
      this.nebulaNetEventClient.publish(eventDto, CustomerRelateOrgEventListener.class, onRebind);
    }
  }

  @Override
  public List<CustomerROrgEntity> findByCustomerCodes(Set<String> customerCodes) {
    if (CollectionUtils.isEmpty(customerCodes)) {
      return Lists.newLinkedList();
    }
    return this.customerROrgRepository.findByCustomerCodes(
        Lists.newArrayList(customerCodes), TenantUtils.getTenantCode());
  }
}
