package com.biz.crm.mdm.business.customer.local.repository;

import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.mdm.business.customer.local.entity.CustomerROrgEntity;
import com.biz.crm.mdm.business.customer.local.mapper.CustomerROrgMapper;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Set;

/**
 * 客户与企业组织关联表的数据库访问类 {@link CustomerROrgEntity}
 *
 * @author ning.zhang
 * @date 2021-11-10 10:19:10
 */
@Component
public class CustomerROrgRepository extends ServiceImpl<CustomerROrgMapper, CustomerROrgEntity> {

  private static final Integer SIZE=1000;

  /**
   * 通过客户编码删除客户关联企业组织关系
   *
   * @param customerCode 客户编码
   * @param tenantCode 租户编码
   */
  public void deleteByCustomerCode(String customerCode, String tenantCode) {
    this.lambdaUpdate()
        .eq(CustomerROrgEntity::getTenantCode, tenantCode)
        .eq(CustomerROrgEntity::getCustomerCode, customerCode)
        .remove();
  }

  /**
   * 重新绑定企业组织编码
   *
   * @param orgCode 企业组织编码
   * @param originOrgCode 原企业组织编码
   * @param customerCodeList 客户编码集合
   * @param tenantCode 租户编码
   */
  public void rebindOrgCode(
      String orgCode, String originOrgCode, List<String> customerCodeList, String tenantCode) {
    this.lambdaUpdate()
        .eq(CustomerROrgEntity::getOrgCode, originOrgCode)
        .eq(CustomerROrgEntity::getTenantCode, tenantCode)
        .in(CustomerROrgEntity::getCustomerCode, customerCodeList)
        .set(CustomerROrgEntity::getOrgCode, orgCode)
        .update();
  }

  /**
   * 根据客户编码获取组织绑定信息
   *
   * @param customerCodeList
   * @param tenantCode
   * @return
   */
  public List<CustomerROrgEntity> findByCustomerCodes(
      List<String> customerCodeList, String tenantCode) {
    return this.lambdaQuery()
        .eq(CustomerROrgEntity::getTenantCode, tenantCode)
        .in(CustomerROrgEntity::getCustomerCode, customerCodeList)
        .list();
  }

  /**
   * 获取审批通过且未删除的经销商对应的组织关系明细
   *
   * @param orgCodes
   * @return
   */
  public List<CustomerROrgEntity> findAllowSaleCustomerByOrgCodes(Set<String> orgCodes) {
    List<CustomerROrgEntity> list = Lists.newArrayList();
    String processStatus = ProcessStatusEnum.PASS.getDictCode();
    String delFlag = DelFlagStatusEnum.NORMAL.getCode();
    String tenantCode = TenantUtils.getTenantCode();
    for (List<String> item : Lists.partition(Lists.newArrayList(orgCodes), SIZE)) {
      List<CustomerROrgEntity> cur = this.baseMapper.findAllowSaleCustomerByOrgCodes(item, processStatus, delFlag, tenantCode);
      if (CollectionUtils.isEmpty(cur)) {
        continue;
      }
      list.addAll(cur);
    }
    return list;
  }
}
