package com.biz.crm.mdm.business.customer.local.repository;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.enums.FreezeStatusEnum;
import com.biz.crm.mdm.business.customer.local.entity.CustomerEntity;
import com.biz.crm.mdm.business.customer.local.mapper.CustomerMapper;
import com.biz.crm.mdm.business.customer.local.model.MultipleConditionModel;
import com.biz.crm.mdm.business.customer.sdk.dto.CustomerQueryDto;
import com.biz.crm.mdm.business.customer.sdk.dto.CustomerSelectDto;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Set;

/**
 * 客户的数据库访问类 {@link CustomerEntity}
 *
 * @author ning.zhang
 * @date 2021/9/28
 */
@Component
public class CustomerRepository extends ServiceImpl<CustomerMapper, CustomerEntity> {

  private static final Integer SIZE = 1000;

  /**
   * 根据客户编码列表获取客户列表
   *
   * @param customerCodes 客户编码集合
   * @param tenantCode    租户编码
   * @return 客户列表
   */
  public List<CustomerEntity> findByCodes(List<String> customerCodes, String tenantCode) {
    return this.lambdaQuery()
        .eq(CustomerEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(CustomerEntity::getTenantCode, tenantCode)
        .in(CustomerEntity::getCustomerCode, customerCodes)
        .list();
  }

  /**
   * 根据流程编码获取客户实体
   *
   * @param processNumber 流程编码
   * @return 客户实体
   */
  public CustomerEntity findByProcessNumber(String processNumber) {
    return this.lambdaQuery().eq(CustomerEntity::getProcessNumber, processNumber).one();
  }

  /**
   * 批量删除
   *
   * @param ids ID集合
   */
  public void updateDelFlagByIds(List<String> ids) {
    this.lambdaUpdate()
        .in(CustomerEntity::getId, ids)
        .set(CustomerEntity::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
        .update();
  }

  /**
   * 批量更新启用禁用状态
   *
   * @param ids          ID集合
   * @param enableStatus 启用禁用状态
   */
  public void updateEnableStatusByIds(List<String> ids, EnableStatusEnum enableStatus) {
    this.lambdaUpdate()
        .in(CustomerEntity::getId, ids)
        .set(CustomerEntity::getEnableStatus, enableStatus.getCode())
        .update();
  }

  /**
   * 批量更新冻结状态
   *
   * @param ids          ID集合
   * @param freezeStatus 冻结状态
   */
  public void freezeBatch(List<String> ids, FreezeStatusEnum freezeStatus) {
    this.lambdaUpdate()
        .in(CustomerEntity::getId, ids)
        .set(CustomerEntity::getLockState, freezeStatus.getCode())
        .update();
  }

  /**
   * 客户信息下拉框分页列表
   *
   * @param pageable 分页信息
   * @param dto      分页参数dto
   * @return 客户信息下拉框分页列表
   */
  public Page<CustomerEntity> findByCustomerSelectDto(Pageable pageable, CustomerSelectDto dto) {
    Page<CustomerEntity> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    dto.setProcessStatus(ProcessStatusEnum.PASS.getDictCode());
    return this.baseMapper.findByCustomerSelectDto(page, dto);
  }

  /**
   * 下级客户信息分页列表
   *
   * @param pageable
   * @param tenantCode
   * @param customerCode
   * @return
   */
  public Page<CustomerEntity> findChildrenByCustomerCode(
      Pageable pageable, String tenantCode, String customerCode) {
    Page<CustomerEntity> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return this.baseMapper.findChildrenByCustomerCode(page, tenantCode, customerCode);
  }

  /**
   * 通过ID或者编码获取客户信息详情(包含关联数据)
   *
   * @param id           客户ID
   * @param customerCode 客户编码
   * @param tenantCode   租户编码
   * @return 客户信息详情(包含关联数据)
   */
  public CustomerEntity findDetailsByIdOrCode(String id, String customerCode, String tenantCode) {
    if (StringUtils.isAllBlank(id, customerCode)) {
      return null;
    }
    return this.baseMapper.findDetailsByIdOrCode(id, customerCode, tenantCode);
  }

  /**
   * 重新绑定客户组织编码
   *
   * @param orgCode          客户组织编码
   * @param originOrgCode    原企业组织编码
   * @param customerCodeList 客户编码集合
   * @param tenantCode       租户编码
   */
  public void rebindCustomerOrgCode(
      String orgCode, String originOrgCode, List<String> customerCodeList, String tenantCode) {
    this.lambdaUpdate()
        .eq(CustomerEntity::getCustomerOrgCode, originOrgCode)
        .eq(CustomerEntity::getTenantCode, tenantCode)
        .in(CustomerEntity::getCustomerCode, customerCodeList)
        .set(CustomerEntity::getCustomerOrgCode, orgCode)
        .update();
  }

  /**
   * 多条件查询客户列表
   *
   * @param model 条件model
   * @return 客户列表
   */
  public List<CustomerEntity> findByMultipleConditionModel(MultipleConditionModel model) {
    model.setProcessStatus(ProcessStatusEnum.PASS.getDictCode());
    return this.baseMapper.findByMultipleConditionModel(model);
  }

  /**
   * 根据企业组织获取对应的客户信息
   *
   * @param orgCodeList 组织编码集合
   * @return 客户信息集合
   */
  public List<CustomerEntity> findByOrgCodes(List<String> orgCodeList) {
    return this.baseMapper.findByOrgCodes(
        orgCodeList, DelFlagStatusEnum.NORMAL.getCode(), TenantUtils.getTenantCode());
  }

  /**
   * 根据渠道编码集合获取对应的客户信息
   *
   * @param channelList 渠道集合
   * @param tenantCode  租户编码
   * @return 客户信息集合
   */
  public List<CustomerEntity> findByChannels(List<String> channelList, String tenantCode) {
    return this.lambdaQuery()
        .eq(CustomerEntity::getTenantCode, tenantCode)
        .in(CustomerEntity::getChannel, channelList)
        .list();
  }

  /**
   * 上级客户信息下拉框分页列表
   *
   * @param pageable
   * @return
   */
  public Page<CustomerEntity> findByParentCustomerIsNull(Pageable pageable, String status) {
    Page<CustomerEntity> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return this.baseMapper.findByParentCustomerIsNull(page, status);
  }

  /**
   * 根据标签id查询
   *
   * @param tagId
   * @return
   */
  public Page<CustomerEntity> findByTagId(Pageable pageable, String tagId) {
    Page<CustomerEntity> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return this.baseMapper.findByTagId(page, tagId);
  }

  /**
   * 根据客户编码或客户名称查询
   *
   * @param
   * @return
   */
  public List<CustomerEntity> findByCustomerCodeLikeOrCustomerNameLike(
      String customerCodeLikeOrNameLike) {
    if (StringUtils.isEmpty(customerCodeLikeOrNameLike)) {
      return lambdaQuery().eq(CustomerEntity::getTenantCode, TenantUtils.getTenantCode()).list();
    }
    return this.lambdaQuery()
        .eq(CustomerEntity::getTenantCode, TenantUtils.getTenantCode())
        .and(
            i ->
                i.like(
                    StringUtils.isNotEmpty(customerCodeLikeOrNameLike),
                    CustomerEntity::getCustomerCode,
                    customerCodeLikeOrNameLike)
                    .or()
                    .like(
                        StringUtils.isNotEmpty(customerCodeLikeOrNameLike),
                        CustomerEntity::getCustomerName,
                        customerCodeLikeOrNameLike))
        .list();
  }

  /**
   * 根据状态查询
   *
   * @param status
   * @return
   */
  public List<CustomerEntity> findByStatus(String status) {
    return this.lambdaQuery()
        .eq(CustomerEntity::getEnableStatus, status)
        .eq(CustomerEntity::getDelFlag, status)
        .list();
  }

  /**
   * 根据渠道编码集合获取对应的客户信息
   *
   * @param typeList   渠道集合
   * @param tenantCode 租户编码
   * @return 客户信息集合
   */
  public List<CustomerEntity> findByTypes(List<String> typeList, String tenantCode) {
    return this.lambdaQuery()
        .eq(CustomerEntity::getTenantCode, tenantCode)
        .in(CustomerEntity::getCustomerType, typeList)
        .list();
  }

  /**
   * 根据组织集合、渠道集合、标签集合查询客户编码集合 参数非必填项所以不需要进行参数校验
   *
   * @param orgCodes 组织集合
   * @param channels 渠道集合
   * @param tags     标签集合
   * @return 客户编码集合
   */
  public Set<String> findCustomerCodesByOrgCodesAndChannelsAndTags(
      List<String> orgCodes, List<String> channels, List<String> tags) {
    return this.baseMapper.findCustomerCodesByOrgCodesAndChannelsAndTags(orgCodes, channels, tags);
  }

  /**
   * 根据高德id查询数据集合
   *
   * @param amapIds
   * @return
   */
  public List<CustomerEntity> findByAmapIds(Set<String> amapIds) {
    return this.lambdaQuery().in(CustomerEntity::getAmapId, amapIds).list();
  }

  /**
   * 根据CustomerQueryDto获取匹配的客户编码集合
   *
   * @param dto
   * @return
   */
  public Set<String> findByCustomerQueryDto(CustomerQueryDto dto) {
    if (CollectionUtils.isEmpty(dto.getOrgCodeSet())) {
      return this.findItemCustomerQueryDto(dto);
    }
    Set<String> set = Sets.newHashSet();
    for (List<String> item : Lists.partition(Lists.newArrayList(dto.getOrgCodeSet()), SIZE)) {
      dto.setOrgCodeSet(Sets.newHashSet(item));
      final Set<String> cur = this.findItemCustomerQueryDto(dto);
      if (CollectionUtils.isEmpty(cur)) {
        continue;
      }
      set.addAll(cur);
    }
    return set;
  }

  private Set<String> findItemCustomerQueryDto(CustomerQueryDto dto) {
    if (CollectionUtils.isEmpty(dto.getCustomerCodeSet())) {
      return this.baseMapper.findByCustomerQueryDto(dto, TenantUtils.getTenantCode());
    }
    Set<String> set = Sets.newHashSet();
    for (List<String> item : Lists.partition(Lists.newArrayList(dto.getCustomerCodeSet()), SIZE)) {
      dto.setCustomerCodeSet(Sets.newHashSet(item));
      final Set<String> cur =
          this.baseMapper.findByCustomerQueryDto(dto, TenantUtils.getTenantCode());
      if (CollectionUtils.isNotEmpty(cur)) {
        set.addAll(cur);
      }
    }
    return set;
  }

  /**
   * 根据创建人账号，年月日期，业务系统类型查询数据
   *
   * @param createAccount
   * @param fromType
   * @return
   */
  public List<CustomerEntity> findCountByCreateAccountAndFromTypeAndCreateTimeScope(String createAccount, String fromType, String startDate, String endDate) {
    return this.lambdaQuery().eq(CustomerEntity::getCreateAccount, createAccount)
        .eq(CustomerEntity::getFromType, fromType)
        .between(CustomerEntity::getCreateTime, startDate, endDate)
        .list();
  }

  /**
   * 根据渠道和客户编码查询
   *
   * @param customerCode 客户代码
   * @param channelList  频道列表
   * @return {@link CustomerEntity}
   */
  public CustomerEntity findByCustomerCodeAndChannels(String customerCode, List<String> channelList) {
    return this.lambdaQuery()
        .eq(CustomerEntity::getTenantCode, TenantUtils.getTenantCode())
        .eq(CustomerEntity::getCustomerCode, customerCode)
        .in(CustomerEntity::getChannel, channelList)
        .one();
  }

  /**
   * 找到客户代码
   *
   * @param customerCode 客户代码
   * @return {@link CustomerEntity}
   */
  public CustomerEntity findByCustomerCode(String customerCode) {
    return this.lambdaQuery()
        .eq(CustomerEntity::getTenantCode, TenantUtils.getTenantCode())
        .eq(CustomerEntity::getCustomerCode, customerCode)
        .one();
  }

  /**
   * 找到customer7 org并不org不在
   *
   * @param customerCode 客户代码
   * @param orgCodeIn    组织代码
   * @param orgCodeNotIn org代码不
   * @return {@link List}<{@link CustomerEntity}>
   */
  public List<CustomerEntity> findByCustomer7OrgIn7OrgNotIn(String customerCode, List<String> orgCodeIn, List<String> orgCodeNotIn) {
    return this.baseMapper.findByCustomer7OrgIn7OrgNotIn(
        customerCode, orgCodeIn, orgCodeNotIn,
        DelFlagStatusEnum.NORMAL.getCode(), TenantUtils.getTenantCode()
    );
  }

}
