package com.biz.crm.mdm.business.customer.local.service.notifier;

import cn.hutool.core.collection.CollectionUtil;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.common.log.sdk.dto.CrmBusinessLogDto;
import com.biz.crm.common.log.sdk.enums.OperationTypeEunm;
import com.biz.crm.common.log.sdk.service.CrmBusinessLogVoService;
import com.biz.crm.mdm.business.customer.local.entity.CustomerEntity;
import com.biz.crm.mdm.business.customer.local.model.MultipleConditionModel;
import com.biz.crm.mdm.business.customer.local.service.CustomerService;
import com.biz.crm.mdm.business.customer.org.sdk.event.CustomerOrgEventListener;
import com.biz.crm.mdm.business.customer.org.sdk.vo.CustomerOrgVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.stream.Collectors;

/**
 * 客户信息客户组织事件监听实现
 *
 * @author ning.zhang
 * @date 2021/11/12
 */
@Component
public class CusCustomerOrgEventListener implements CustomerOrgEventListener {

  @Autowired(required = false)
  private CustomerService customerService;

  @Autowired(required = false)
  private CrmBusinessLogVoService crmBusinessLogVoService;

  @Override
  public void onBatchCreate(List<CustomerOrgVo> customerOrgVos) {
    //保存日志
    if (CollectionUtils.isNotEmpty(customerOrgVos)) {
      CustomerOrgVo customerOrgVo = customerOrgVos.get(0);
      String onlyKey = customerOrgVo.getId();
      CrmBusinessLogDto crmBusinessLogDto = new CrmBusinessLogDto();
      crmBusinessLogDto.setOperationType(OperationTypeEunm.CREATE.getDictCode());
      crmBusinessLogDto.setOnlyKey(onlyKey);
      crmBusinessLogDto.setAppCode(TenantUtils.getTenantCode());
      crmBusinessLogDto.setTenantCode(TenantUtils.getTenantCode());
      crmBusinessLogDto.setOldObject(null);
      crmBusinessLogDto.setNewObject(customerOrgVo);
      crmBusinessLogVoService.handleSave(crmBusinessLogDto);
    }
  }

  @Override
  public void onBatchUpdate(List<CustomerOrgVo> customerOrgVos) {
    //修改数据日志记录
    if (CollectionUtils.isNotEmpty(customerOrgVos) && customerOrgVos.size() > 1) {
      //0 新数据，1 旧数据
      CustomerOrgVo newCustomerOrgVo = customerOrgVos.get(0);
      CustomerOrgVo oldCustomerOrgVo = customerOrgVos.get(1);
      String onlyKey = newCustomerOrgVo.getId();
      CrmBusinessLogDto crmBusinessLogDto = new CrmBusinessLogDto();
      crmBusinessLogDto.setOperationType(OperationTypeEunm.UPDATE.getDictCode());
      crmBusinessLogDto.setOnlyKey(onlyKey);
      crmBusinessLogDto.setAppCode(TenantUtils.getTenantCode());
      crmBusinessLogDto.setTenantCode(TenantUtils.getTenantCode());
      crmBusinessLogDto.setNewObject(newCustomerOrgVo);
      crmBusinessLogDto.setOldObject(oldCustomerOrgVo);
      crmBusinessLogVoService.handleSave(crmBusinessLogDto);
    }
    //do nothing
  }

  @Override
  public void onBatchDisable(List<CustomerOrgVo> customerOrgVos) {
    if (CollectionUtils.isEmpty(customerOrgVos)) {
      return;
    }
    //禁用数据写入日志
    customerOrgVos.stream().forEach(customerOrgVo -> {
      CustomerOrgVo newCustomerOrgVo = new CustomerOrgVo();
      CustomerOrgVo oldCustomerOrgVo = new CustomerOrgVo();
      newCustomerOrgVo.setEnableStatus(customerOrgVo.getEnableStatus());
      newCustomerOrgVo.setId(customerOrgVo.getId());
      oldCustomerOrgVo.setId(customerOrgVo.getId());
      oldCustomerOrgVo.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
      String onlyKey = customerOrgVo.getId();
      CrmBusinessLogDto crmBusinessLogDto = new CrmBusinessLogDto();
      crmBusinessLogDto.setOperationType(OperationTypeEunm.UPDATE.getDictCode());
      crmBusinessLogDto.setOnlyKey(onlyKey);
      crmBusinessLogDto.setAppCode(TenantUtils.getTenantCode());
      crmBusinessLogDto.setTenantCode(TenantUtils.getTenantCode());
      crmBusinessLogDto.setOldObject(oldCustomerOrgVo);
      crmBusinessLogDto.setNewObject(newCustomerOrgVo);
      crmBusinessLogVoService.handleSave(crmBusinessLogDto);
    });
    //do nothing
  }

  @Override
  public void onBatchEnable(List<CustomerOrgVo> customerOrgVos) {
    if (CollectionUtils.isEmpty(customerOrgVos)) {
      return;
    }
    //启用数据写入日志
    customerOrgVos.stream().forEach(customerOrgVo -> {
      CustomerOrgVo newCustomerOrgVo = new CustomerOrgVo();
      CustomerOrgVo oldCustomerOrgVo = new CustomerOrgVo();
      newCustomerOrgVo.setEnableStatus(customerOrgVo.getEnableStatus());
      newCustomerOrgVo.setId(customerOrgVo.getId());
      oldCustomerOrgVo.setId(customerOrgVo.getId());
      oldCustomerOrgVo.setEnableStatus(EnableStatusEnum.DISABLE.getCode());
      String onlyKey = customerOrgVo.getId();
      CrmBusinessLogDto crmBusinessLogDto = new CrmBusinessLogDto();
      crmBusinessLogDto.setOperationType(OperationTypeEunm.UPDATE.getDictCode());
      crmBusinessLogDto.setOnlyKey(onlyKey);
      crmBusinessLogDto.setAppCode(TenantUtils.getTenantCode());
      crmBusinessLogDto.setTenantCode(TenantUtils.getTenantCode());
      crmBusinessLogDto.setOldObject(oldCustomerOrgVo);
      crmBusinessLogDto.setNewObject(newCustomerOrgVo);
      crmBusinessLogVoService.handleSave(crmBusinessLogDto);
    });
    //do nothing
  }

  /**
   * 当客户组织发生删除事件时，根据触发事件的客户组织编码,验证客户组织是否存在与客户的关联关系。
   * <p>
   * 1.存在与客户的关联关系,则抛出组织已关联客户的异常,阻断当前客户组织的删除操作
   * 2.不存在与客户的关联关系,则什么也不做.
   */
  @Override
  public void onBatchDelete(List<CustomerOrgVo> eventDtos) {
    if (CollectionUtils.isEmpty(eventDtos)) {
      return;
    }
    List<String> customerOrgCodes = eventDtos.stream().map(CustomerOrgVo::getCustomerOrgCode).collect(Collectors.toList());
    MultipleConditionModel model = new MultipleConditionModel();
    model.setCustomerOrgCodes(customerOrgCodes);
    List<CustomerEntity> list = customerService.findByMultipleConditionModel(model);
    Validate.isTrue(CollectionUtils.isEmpty(list), "存在与客户的绑定关系不能删除");
    //记录删除日志
    eventDtos.stream().forEach(customerOrgVo -> {
      String onlyKey = customerOrgVo.getId();
      CrmBusinessLogDto crmBusinessLogDto = new CrmBusinessLogDto();
      crmBusinessLogDto.setOperationType(OperationTypeEunm.DELETE.getDictCode());
      crmBusinessLogDto.setOnlyKey(onlyKey);
      crmBusinessLogDto.setAppCode(TenantUtils.getTenantCode());
      crmBusinessLogDto.setTenantCode(TenantUtils.getTenantCode());
      crmBusinessLogDto.setOldObject(customerOrgVo);
      crmBusinessLogDto.setNewObject(null);
      crmBusinessLogVoService.handleSave(crmBusinessLogDto);
    });
  }

}
