package com.biz.crm.mdm.business.customer.local.repository;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.mdm.business.customer.local.entity.CustomerAddressEntity;
import com.biz.crm.mdm.business.customer.local.mapper.CustomerAddressMapper;
import com.biz.crm.mdm.business.customer.sdk.dto.CustomerAddressPageDto;
import com.biz.crm.mdm.business.customer.sdk.dto.CustomerAddressQueryDto;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Objects;

/**
 * 客户收货地址表的数据库访问类 {@link CustomerAddressEntity}
 *
 * @author ning.zhang
 * @date 2021-10-28 17:48:58
 */
@Component
public class CustomerAddressRepository extends ServiceImpl<CustomerAddressMapper, CustomerAddressEntity> {

  /**
   * 通过客户编码解绑默认地址
   *
   * @param customerCode 客户编码
   * @param tenantCode   租户编码
   */
  public void unbindDefaultAddressByCustomerCode(String customerCode, String tenantCode) {
    this.lambdaUpdate()
        .eq(CustomerAddressEntity::getTenantCode, tenantCode)
        .eq(CustomerAddressEntity::getCustomerCode, customerCode)
        .eq(CustomerAddressEntity::getDefaultAddress, Boolean.TRUE)
        .set(CustomerAddressEntity::getDefaultAddress, Boolean.FALSE)
        .update();
  }

  /**
   * 批量删除
   *
   * @param ids ID集合
   */
  public void updateDelFlagByIds(List<String> ids) {
    this.lambdaUpdate()
        .eq(CustomerAddressEntity::getTenantCode, TenantUtils.getTenantCode())    //新增租户编号判断条件
        .in(CustomerAddressEntity::getId, ids)
        .set(CustomerAddressEntity::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
        .update();
  }

  /**
   * 客户收货地址分页列表
   *
   * @param dto      请求参数dto
   * @param pageable 分页信息
   * @return Page<CustomerAddressEntity> 客户收货地址分页信息
   */
  public Page<CustomerAddressEntity> findByConditions(Pageable pageable, CustomerAddressPageDto dto) {
    Page<CustomerAddressEntity> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return this.baseMapper.findByConditions(page, dto);
  }

  /**
   * 通过ID获取客户收货地址信息
   *
   * @param id 收货地址ID
   * @return 客户收货地址信息
   */
  public CustomerAddressEntity findById(String id) {
    return this.lambdaQuery()
        .eq(CustomerAddressEntity::getId, id)
        .eq(CustomerAddressEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .one();
  }

  /**
   * 查询客户收货地址列表
   *
   * @param dto 查询dto
   * @return 客户收货地址信息列表
   */
  public List<CustomerAddressEntity> findByCustomerAddressQueryDto(CustomerAddressQueryDto dto) {
    return this.lambdaQuery()
        .eq(CustomerAddressEntity::getTenantCode, dto.getTenantCode())
        .eq(CustomerAddressEntity::getCustomerCode, dto.getCustomerCode())
        .eq(CustomerAddressEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(Objects.nonNull(dto.getDefaultAddress()), CustomerAddressEntity::getDefaultAddress, dto.getDefaultAddress())
        .list();
  }
}
