package com.biz.crm.mdm.business.customer.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.mdm.business.customer.local.entity.CustomerAddressEntity;
import com.biz.crm.mdm.business.customer.local.repository.CustomerAddressRepository;
import com.biz.crm.mdm.business.customer.local.service.CustomerAddressService;
import com.biz.crm.mdm.business.customer.sdk.dto.CustomerAddressDto;
import com.biz.crm.mdm.business.customer.sdk.dto.CustomerAddressPageDto;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;

/**
 * 客户收货地址表服务实现类
 *
 * @author ning.zhang
 * @date 2021-10-28 17:48:59
 */
@Service("customerAddressService")
public class CustomerAddressServiceImpl implements CustomerAddressService {

  @Autowired(required = false)
  private CustomerAddressRepository customerAddressRepository;
  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Override
  @Transactional
  public CustomerAddressEntity create(CustomerAddressDto dto) {
    this.createValidation(dto);
    CustomerAddressEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, CustomerAddressEntity.class, HashSet.class, ArrayList.class);
    //将null值也视为非默认地址
    entity.setDefaultAddress(Boolean.TRUE.equals(entity.getDefaultAddress()));
    if (Boolean.TRUE.equals(entity.getDefaultAddress())) {
      dto.setTenantCode(TenantUtils.getTenantCode());    //新增租户编号判断条件
      this.customerAddressRepository.unbindDefaultAddressByCustomerCode(dto.getCustomerCode(), dto.getTenantCode());
    }
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(DelFlagStatusEnum.NORMAL.getCode());
    //新增租户编号
    entity.setTenantCode(TenantUtils.getTenantCode());
    this.customerAddressRepository.save(entity);
    return entity;
  }

  @Override
  @Transactional
  public CustomerAddressEntity update(CustomerAddressDto dto) {
    this.updateValidation(dto);
    CustomerAddressEntity entity = this.customerAddressRepository.findByIdAndTenantCode(dto.getId(),TenantUtils.getTenantCode());
    Validate.notNull(entity, "客户收货地址信息不存在");
    Validate.isTrue(entity.getCustomerCode().equals(dto.getCustomerCode()), "所属客户不能修改");
    CustomerAddressEntity updateEntity = this.nebulaToolkitService.copyObjectByWhiteList(dto, CustomerAddressEntity.class, HashSet.class, ArrayList.class);
    //将null值也视为非默认地址
    updateEntity.setDefaultAddress(Boolean.TRUE.equals(updateEntity.getDefaultAddress()));
    if (Boolean.TRUE.equals(updateEntity.getDefaultAddress())) {
      dto.setTenantCode(TenantUtils.getTenantCode());    //新增租户编号判断条件
      this.customerAddressRepository.unbindDefaultAddressByCustomerCode(dto.getCustomerCode(), dto.getTenantCode());
    }
    //新增租户编号
    updateEntity.setTenantCode(TenantUtils.getTenantCode());
    //重构修改方法
    this.customerAddressRepository.updateByIdAndTenantCode(updateEntity,TenantUtils.getTenantCode());
    return updateEntity;
  }

  @Override
  @Transactional
  public void deleteBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "缺失id");
    List<CustomerAddressEntity> entities = this.customerAddressRepository.listByIdsAndTenantCode(ids,TenantUtils.getTenantCode());
    Validate.isTrue(CollectionUtils.isNotEmpty(entities) && entities.size() == ids.size(), "数据删除个数不匹配");
    this.customerAddressRepository.updateDelFlagByIds(ids);
  }

  @Override
  public Page<CustomerAddressEntity> findByConditions(Pageable pageable, CustomerAddressPageDto dto) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    dto = ObjectUtils.defaultIfNull(dto, new CustomerAddressPageDto());
    dto.setTenantCode(TenantUtils.getTenantCode());
    return this.customerAddressRepository.findByConditions(pageable, dto);
  }

  @Override
  public CustomerAddressEntity findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.customerAddressRepository.findById(id);
  }

  /**
   * 在创建customerAddress模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void createValidation(CustomerAddressDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setTenantCode(TenantUtils.getTenantCode());
    dto.setId(null);
    Validate.notBlank(dto.getCustomerCode(), "缺失客户编码");
    Validate.notBlank(dto.getCityCode(), "缺失市级编码");
    Validate.notBlank(dto.getDistrictCode(), "缺失区级编码");
    Validate.notBlank(dto.getProvinceCode(), "缺失省级编码");
    Validate.notBlank(dto.getDetailedAddress(), "缺失详细地址");
    Validate.notBlank(dto.getContactName(), "缺失联系人");
    Validate.isTrue(dto.getDetailedAddress().length() < 128, "详细地址，在进行添加时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(dto.getContactName().length() < 64, "联系人，在进行添加时填入值超过了限定长度(64)，请检查!");
  }

  /**
   * 在修改customerAddress模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void updateValidation(CustomerAddressDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getId(), "修改信息时，id不能为空！");
    Validate.notBlank(dto.getCustomerCode(), "缺失客户编码");
    Validate.notBlank(dto.getCityCode(), "缺失市级编码");
    Validate.notBlank(dto.getDistrictCode(), "缺失区级编码");
    Validate.notBlank(dto.getProvinceCode(), "缺失省级编码");
    Validate.notBlank(dto.getDetailedAddress(), "缺失详细地址");
    Validate.notBlank(dto.getContactName(), "缺失联系人");
    Validate.isTrue(dto.getDetailedAddress().length() < 128, "详细地址，在进行编辑时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(dto.getContactName().length() < 64, "联系人，在进行编辑时填入值超过了限定长度(64)，请检查!");
  }
}
