package com.biz.crm.mdm.business.customer.local.service.internal;

import com.biz.crm.mdm.business.customer.local.entity.CustomerBillEntity;
import com.biz.crm.mdm.business.customer.local.repository.CustomerBillRepository;
import com.biz.crm.mdm.business.customer.local.service.CustomerBillService;
import com.biz.crm.mdm.business.customer.sdk.dto.CustomerBillDto;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 客户开票信息实表服务实现类
 *
 * @author ning.zhang
 * @date 2021-10-26 16:18:14
 */
@Service("customerBillService")
public class CustomerBillServiceImpl implements CustomerBillService {

  @Autowired(required = false)
  private CustomerBillRepository customerBillRepository;
  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Override
  @Transactional
  public void rebindCustomerCode(List<CustomerBillDto> dtoList, String customerCode) {
    Validate.notBlank(customerCode, "客户编码不能为空");
    String tenantCode = TenantUtils.getTenantCode();
    this.customerBillRepository.deleteByCustomerCode(customerCode, tenantCode);
    if (CollectionUtils.isEmpty(dtoList)) {
      return;
    }
    //过滤掉开票单位为空的数据
    List<CustomerBillEntity> entities = dtoList.stream()
        .filter(dto -> Objects.nonNull(dto) && StringUtils.isNotBlank(dto.getBillCompany()))
        .map(dto -> {
          CustomerBillEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, CustomerBillEntity.class, HashSet.class, ArrayList.class);
          //默认开票信息标识为空表示默认开票
          entity.setBillMain(Objects.isNull(entity.getBillMain()) ? Boolean.FALSE : entity.getBillMain());
          entity.setCustomerCode(customerCode);
          entity.setTenantCode(tenantCode);
          return entity;
        }).collect(Collectors.toList());
    if (CollectionUtils.isEmpty(entities)) {
      return;
    }
    long count = entities.stream().filter(CustomerBillEntity::getBillMain).count();
    Validate.isTrue(count <= 1, "客户信息：默认开票信息只能有一个");
    Validate.isTrue(count == 1, "客户信息：默认开票信息必须有一个");
    this.customerBillRepository.saveBatch(entities);
  }
}
