package com.biz.crm.mdm.business.customer.local.service.internal;

import com.biz.crm.mdm.business.customer.local.entity.CustomerContactEntity;
import com.biz.crm.mdm.business.customer.local.repository.CustomerContactRepository;
import com.biz.crm.mdm.business.customer.local.service.CustomerContactService;
import com.biz.crm.mdm.business.customer.sdk.dto.CustomerContactDto;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerContactVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 客户联系人信息表服务实现类
 *
 * @author ning.zhang
 * @date 2021-10-26 16:23:48
 */
@Service("customerContactService")
public class CustomerContactServiceImpl implements CustomerContactService {

  @Autowired(required = false)
  private CustomerContactRepository customerContactRepository;
  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

    /**
     * 基于数据库执行的数据视图执行内容缓存（最多500毫秒）
     */
    private static volatile Cache<String, List<CustomerContactVo>> cache = null;

    public CustomerContactServiceImpl(){
        if(cache == null) {
            synchronized (CustomerContactServiceImpl.class) {
                while(cache == null) {
                    cache = CacheBuilder.newBuilder()
                            .initialCapacity(10000)
                            .expireAfterWrite(500, TimeUnit.MILLISECONDS)
                            .maximumSize(100000)
                            .build();
                }
            }
        }
    }

  @Override
  @Transactional
  public void rebindCustomerCode(List<CustomerContactDto> dtoList, String customerCode) {
    Validate.notBlank(customerCode, "客户编码不能为空");
    String tenantCode = TenantUtils.getTenantCode();
    this.customerContactRepository.deleteByCustomerCode(customerCode, tenantCode);
    if (CollectionUtils.isEmpty(dtoList)) {
      return;
    }
    //过滤联系人姓名为空的数据
    List<CustomerContactEntity> entities = dtoList.stream()
        .filter(dto -> Objects.nonNull(dto) && StringUtils.isNotEmpty(dto.getContactName()))
        .map(dto -> {
          CustomerContactEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, CustomerContactEntity.class, HashSet.class, ArrayList.class);
          //主联系人标识为空表示非主联系人
          entity.setContactMain(Objects.isNull(entity.getContactMain()) ? Boolean.FALSE : entity.getContactMain());
          entity.setCustomerCode(customerCode);
          entity.setTenantCode(tenantCode);
          return entity;
        }).collect(Collectors.toList());
    if (CollectionUtils.isEmpty(entities)) {
      return;
    }
    long count = entities.stream().filter(CustomerContactEntity::getContactMain).count();
    Validate.isTrue(count <= 1, "客户信息：主联系人只能有一个");
    Validate.isTrue(count == 1, "客户信息：主联系人必须有一个");
    this.customerContactRepository.saveBatch(entities);
  }

    @Override
    public List<CustomerContactVo> findByCustomerCodes(List<String> customerCodes) {
        if (CollectionUtils.isEmpty(customerCodes)) {
            return Lists.newArrayList();
        }
        String cacheKey = StringUtils.join(TenantUtils.getTenantCode(), customerCodes);
        List<CustomerContactVo> graph = cache.getIfPresent(cacheKey);
        if (graph != null) {
            return graph;
        }
        List<CustomerContactEntity> contactEntities = this.customerContactRepository.findByCustomerCodes(customerCodes, TenantUtils.getTenantCode());
        if (CollectionUtils.isEmpty(contactEntities)) {
            return Lists.newArrayList();
        }
        graph = (List<CustomerContactVo>) this.nebulaToolkitService.copyCollectionByBlankList(
                contactEntities, CustomerContactEntity.class, CustomerContactVo.class, HashSet.class, ArrayList.class);
        cache.put(cacheKey, graph);
        return graph;
    }

}
