package com.biz.crm.mdm.business.customer.local.service.internal;

import com.biz.crm.mdm.business.customer.local.entity.CustomerDockingEntity;
import com.biz.crm.mdm.business.customer.local.repository.CustomerDockingRepository;
import com.biz.crm.mdm.business.customer.local.service.CustomerDockingService;
import com.biz.crm.mdm.business.customer.local.service.CustomerDockingSupplyService;
import com.biz.crm.mdm.business.customer.sdk.dto.BindCustomerDockingDto;
import com.biz.crm.mdm.business.customer.sdk.dto.CustomerDockingDto;
import com.biz.crm.mdm.business.customer.sdk.dto.UnbindCustomerDockingDto;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.bouncycastle.crypto.engines.TEAEngine;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 客户对接人表服务实现类
 *
 * @author ning.zhang
 * @date 2021-10-26 16:25:01
 */
@Service("customerDockingService")
public class CustomerDockingServiceImpl implements CustomerDockingService {

  @Autowired(required = false)
  private CustomerDockingRepository customerDockingRepository;
  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private CustomerDockingSupplyService customerDockingSupplyService;

  @Override
  @Transactional
  public void rebindCustomerCode(List<CustomerDockingDto> dtoList, String customerCode) {
    Validate.notBlank(customerCode, "客户编码不能为空");
    String tenantCode = TenantUtils.getTenantCode();
    //清除已有对接人和对接人供货信息
    List<CustomerDockingEntity> list = this.customerDockingRepository.findByCustomerCode(customerCode, tenantCode);
    if (CollectionUtils.isNotEmpty(list)) {
      List<String> dockingIds = list.stream().map(CustomerDockingEntity::getId).collect(Collectors.toList());
      this.customerDockingRepository.removeByIdsAndTenantCode(dockingIds,TenantUtils.getTenantCode());
      this.customerDockingSupplyService.deleteByDockingIds(dockingIds);
    }
    if (CollectionUtils.isEmpty(dtoList)) {
      return;
    }
    //过滤职位编码为空的数据
    dtoList = dtoList.stream().filter(dto -> Objects.nonNull(dto) && StringUtils.isNotEmpty(dto.getPositionCode())).collect(Collectors.toList());
    if (CollectionUtils.isEmpty(dtoList)) {
      return;
    }
    //判断集合中是否存在重复
    Set<String> dtoSet = dtoList.stream().map(CustomerDockingDto::getPositionCode).collect(Collectors.toSet());
    Validate.isTrue(dtoSet.size() == dtoList.size(), "当前职位已经是对接人，请更换为其他职位");
    
    dtoList.forEach(dto -> {
      CustomerDockingEntity entity = this.nebulaToolkitService
          .copyObjectByWhiteList(dto, CustomerDockingEntity.class, HashSet.class, ArrayList.class);
      entity.setCustomerCode(customerCode);
      entity.setTenantCode(tenantCode);
      this.customerDockingRepository.save(entity);
      this.customerDockingSupplyService.bindDockingId(dto.getMdmCustomerSupplyDetailList(), entity.getId());
    });
  }

  @Override
  @Transactional
  public void bind(BindCustomerDockingDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    Validate.isTrue(CollectionUtils.isNotEmpty(dto.getCustomerCodeList()), "缺失客户编码");
    Validate.notBlank(dto.getPositionCode(), "缺失职位编码");
    String tenantCode = TenantUtils.getTenantCode();
    List<CustomerDockingEntity> list = this.customerDockingRepository
        .findByCustomerCodesAndPositionCode(dto.getCustomerCodeList(), dto.getPositionCode(), tenantCode);
    //过滤掉已经存在的关联
    if (CollectionUtils.isNotEmpty(list)) {
      List<String> existCustomerCodes = list.stream().map(CustomerDockingEntity::getCustomerCode).collect(Collectors.toList());
      dto.setCustomerCodeList(dto.getCustomerCodeList().stream().filter(s -> !existCustomerCodes.contains(s)).collect(Collectors.toList()));
    }
    //已经没有需要绑定的客户对接人信息,直接返回
    if (CollectionUtils.isEmpty(dto.getCustomerCodeList())) {
      return;
    }
    List<CustomerDockingEntity> entities = dto.getCustomerCodeList().stream().map(customerCode -> {
      CustomerDockingEntity entity = new CustomerDockingEntity();
      entity.setTenantCode(tenantCode);
      entity.setCustomerCode(customerCode);
      entity.setPositionCode(dto.getPositionCode());
      return entity;
    }).collect(Collectors.toList());
    this.customerDockingRepository.saveBatch(entities);
  }

  @Override
  @Transactional
  public void unbind(UnbindCustomerDockingDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    Validate.isTrue(CollectionUtils.isNotEmpty(dto.getCustomerCodeList()), "缺失客户编码");
    Validate.notBlank(dto.getPositionCode(), "缺失职位编码");
    List<CustomerDockingEntity> list = this.customerDockingRepository
        .findByCustomerCodesAndPositionCode(dto.getCustomerCodeList(), dto.getPositionCode(), TenantUtils.getTenantCode());
    //没有需要解绑的数据,直接返回
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    List<String> dockingIds = list.stream().map(CustomerDockingEntity::getId).collect(Collectors.toList());
    //重构删除方法
    this.customerDockingRepository.deleteByIdsAndTenantCode(dockingIds, TenantUtils.getTenantCode());
    this.customerDockingSupplyService.deleteByDockingIds(dockingIds);
  }

}
