package com.biz.crm.mdm.business.customer.local.service.internal;

import com.biz.crm.mdm.business.customer.local.entity.CustomerSaleAreaEntity;
import com.biz.crm.mdm.business.customer.local.repository.CustomerSaleAreaRepository;
import com.biz.crm.mdm.business.customer.local.service.CustomerSaleAreaService;
import com.biz.crm.mdm.business.customer.sdk.dto.CustomerSaleAreaDto;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerSaleAreaVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;

import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 客户销售区域表服务实现类
 *
 * @author ning.zhang
 * @date 2021-10-26 16:27:00
 */
@Service("customerSaleAreaService")
public class CustomerSaleAreaServiceImpl implements CustomerSaleAreaService {

  @Autowired(required = false)
  private CustomerSaleAreaRepository customerSaleAreaRepository;
  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Override
  @Transactional
  public void rebindCustomerCode(List<CustomerSaleAreaDto> dtoList, String customerCode) {
    Validate.notBlank(customerCode, "客户编码不能为空");
    String tenantCode = TenantUtils.getTenantCode();
    this.customerSaleAreaRepository.deleteByCustomerCode(customerCode, tenantCode);
    if (CollectionUtils.isEmpty(dtoList)) {
      return;
    }
    //过滤一级区域为空的数据
    List<CustomerSaleAreaEntity> entities = dtoList.stream()
        .filter(dto-> Objects.nonNull(dto) && StringUtils.isNotEmpty(dto.getAreaOne()))
        .map(dto -> {
          CustomerSaleAreaEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, CustomerSaleAreaEntity.class, HashSet.class, ArrayList.class);
          entity.setCustomerCode(customerCode);
          entity.setTenantCode(tenantCode);
          return entity;
        }).collect(Collectors.toList());
    if (CollectionUtils.isEmpty(entities)) {
      return;
    }
    this.customerSaleAreaRepository.saveBatch(entities);
  }

    @Override
    public List<CustomerSaleAreaEntity> findByCustomerCode(List<String> code) {
        if (CollectionUtils.isEmpty(code)){
            return Lists.newLinkedList();
        }
        List<CustomerSaleAreaEntity> list = this.customerSaleAreaRepository.lambdaQuery()
                .in(CustomerSaleAreaEntity::getCustomerCode, code)
                .eq(CustomerSaleAreaEntity::getTenantCode, TenantUtils.getTenantCode())
                .list();
        if (CollectionUtils.isEmpty(list)){
            return Lists.newLinkedList();
        }
        return list;
    }
}
