package com.biz.crm.mdm.business.customer.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.enums.FreezeStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mdm.business.customer.local.entity.*;
import com.biz.crm.mdm.business.customer.local.model.MultipleConditionModel;
import com.biz.crm.mdm.business.customer.local.repository.*;
import com.biz.crm.mdm.business.customer.local.service.CustomerBillService;
import com.biz.crm.mdm.business.customer.local.service.CustomerContactService;
import com.biz.crm.mdm.business.customer.local.service.CustomerDockingService;
import com.biz.crm.mdm.business.customer.local.service.CustomerMediaService;
import com.biz.crm.mdm.business.customer.local.service.CustomerROrgService;
import com.biz.crm.mdm.business.customer.local.service.CustomerSaleAreaService;
import com.biz.crm.mdm.business.customer.local.service.CustomerService;
import com.biz.crm.mdm.business.customer.sdk.constant.CustomerConstant;
import com.biz.crm.mdm.business.customer.sdk.dto.*;
import com.biz.crm.mdm.business.customer.sdk.enums.CustomerLevelEnum;
import com.biz.crm.mdm.business.customer.sdk.event.CustomerEventListener;
import com.biz.crm.mdm.business.customer.sdk.event.CustomerNebulaEventListener;
import com.biz.crm.mdm.business.customer.sdk.service.SupplyRelationshipSdkService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerPageVo;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerRelateOrgVo;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.workflow.sdk.dto.ProcessBusinessDto;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.biz.crm.workflow.sdk.service.ProcessBusinessService;
import com.biz.crm.workflow.sdk.vo.ProcessBusinessVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.JsonUtils;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import liquibase.pro.packaged.A;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 客户服务接口实现类
 *
 * @author ning.zhang
 * @date 2021/10/25
 */
@Service
public class CustomerServiceImpl implements CustomerService {

  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;
  @Autowired(required = false)
  private CustomerRepository customerRepository;
  @Autowired(required = false)
  private CustomerBillService customerBillService;
  @Autowired(required = false)
  private CustomerContactService customerContactService;
  @Autowired(required = false)
  private CustomerDockingService customerDockingService;
  @Autowired(required = false)
  private CustomerSaleAreaService customerSaleAreaService;
  @Autowired(required = false)
  private CustomerMediaService customerMediaService;
  @Autowired(required = false)
  private CustomerROrgService customerROrgService;
  @Autowired(required = false)
  private NebulaNetEventClient nebulaNetEventClient;

  @Autowired(required = false)
  private DealerOrgRelationshipRepository dealerOrgRelationshipRepository;

  @Autowired(required = false)
  private SupplyRelationshipSdkService supplyRelationshipSdkService;

  @Autowired(required = false)
  private DealerSalesRegionRepository dealerSalesRegionRepository;

  @Autowired(required = false)
  private CustomerROrgRepository customerROrgRepository;

  public static final String CUSTOMER_PROCESS_NAME= "经销商审批流程";

  /**
   * 基于数据库执行的数据视图执行内容缓存（最多500毫秒）
   */
  private static volatile Cache<String, List<CustomerEntity>> cache = null;

  public CustomerServiceImpl(){
    if(cache == null) {
      synchronized (CustomerServiceImpl.class) {
        while(cache == null) {
          cache = CacheBuilder.newBuilder()
                  .initialCapacity(10000)
                  .expireAfterWrite(500, TimeUnit.MILLISECONDS)
                  .maximumSize(100000)
                  .build();
        }
      }
    }
    
  }
  @Override
  @Transactional
  public CustomerEntity create(CustomerDto dto) {
    this.createValidation(dto);
    this.validateContacts(dto);
    if (StringUtils.isEmpty(dto.getCustomerCode())) {
      String ruleCode = StringUtils.join("KH", DateFormatUtils.format(new Date(), "yyyyMMdd"));
      String code = this.generateCodeService.generateCode(ruleCode, 1, 5, 2, TimeUnit.DAYS).get(0);
      dto.setCustomerCode(code);
    }
    dto.setTenantCode(TenantUtils.getTenantCode());    //新增租户编号判断条件
    List<CustomerEntity> list = this.customerRepository.findByCodes(Lists.newArrayList(dto.getCustomerCode()), dto.getTenantCode());
    CustomerVo oldVo = null;
    if (StringUtils.isEmpty(dto.getId())) {
      Validate.isTrue(CollectionUtils.isEmpty(list), "客户编码已经存在");
    } else {
      Validate.isTrue(list.size() == 1, "客户编码已经存在");
      oldVo = this.findByCustomerEntity(list.get(0));
    }
    CustomerEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, CustomerEntity.class, HashSet.class, ArrayList.class);
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    entity.setLockState(FreezeStatusEnum.NORMAL.getCode());
    entity.setProcessStatus(ProcessStatusEnum.PASS.getDictCode());
    //新增租户编号
    entity.setTenantCode(TenantUtils.getTenantCode());
    entity.setDealerId(dto.getCustomerCode());
    this.customerRepository.saveOrUpdate(entity);
    //dto.setDealerId(entity.getId());
    dto.setDealerId(entity.getDealerId());
    dto.setId(entity.getId());
    // 绑定客户关联数据
    this.rebindRelationData(dto);

    //保存经销商与组织关系集合
//    List<DealerOrgRelationshipDto> relationshipDtos = dto.getDealerOrgRelationshipDtos();
//    List<DealerOrgRelationshipEntity> relationshipEntities = (ArrayList<DealerOrgRelationshipEntity>)this.nebulaToolkitService.copyCollectionByBlankList(relationshipDtos, DealerOrgRelationshipDto.class, DealerOrgRelationshipEntity.class, HashSet.class, ArrayList.class);
//    this.dealerOrgRelationshipRepository.saveOrUpdateBatch(relationshipEntities);
    //保存经销商与组织关系
//    List<CustomerROrgDto> customerROrgDtos =dto.getCustomerROrgDtoList();
//    List<CustomerROrgEntity> orgEntities =(List<CustomerROrgEntity>) this.nebulaToolkitService.copyCollectionByBlankList(customerROrgDtos, CustomerROrgDto.class, CustomerROrgEntity.class, HashSet.class, ArrayList.class);
//    orgEntities.stream().forEach(o->{
//      o.setCustomerCode(dto.getCustomerCode());
//      o.setDealerId(dto.getId());
//      o.setTenantCode(TenantUtils.getTenantCode());
//    });
//    this.customerROrgRepository.saveOrUpdateBatch(orgEntities);
    //保存经销商销售区域
    List<DealerSalesRegionDto> dealerSalesRegionDtoList = dto.getDealerSalesRegionDtoList();
    List<DealerSalesRegionEntity> saleAreaEntities =(List<DealerSalesRegionEntity>) this.nebulaToolkitService.copyCollectionByBlankList(dealerSalesRegionDtoList, DealerSalesRegionDto.class, DealerSalesRegionEntity.class, HashSet.class, ArrayList.class);
    saleAreaEntities.stream().forEach(o->{
      o.setDealerId(entity.getDealerId());
      o.setDealerCode(entity.getCustomerCode());
      o.setDealerName(entity.getCustomerName());
      o.setTenantCode(TenantUtils.getTenantCode());
      o.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
      o.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    });
    this.dealerSalesRegionRepository.saveOrUpdateBatch(saleAreaEntities);

    //保存经销商采供关系集合
    List<SupplyRelationshipDto> supplyRelationshipDtos = dto.getSupplyRelationshipDtos();
    supplyRelationshipDtos.forEach(o -> o.setBuyerId(entity.getDealerId()));
    this.supplyRelationshipSdkService.savaOrUpdate(supplyRelationshipDtos,dto.getDealerId());

    // 业务日志新增
    CustomerEventDto customerEventDto = new CustomerEventDto();
    customerEventDto.setOriginal(null);
    CustomerVo customerVo =this.nebulaToolkitService.copyObjectByWhiteList(entity, CustomerVo.class, HashSet.class, ArrayList.class,"dockingList","contactList","saleAreaList","billList","fileList");
    customerEventDto.setNewest(customerVo);

    //返回组织
    entity.setOrgCode(dto.getOrgCode());
    entity.setDealerSalesRegionEntityList(saleAreaEntities);
    entity.setSupplyRelationshipEntityList((List<SupplyRelationshipEntity>) this.nebulaToolkitService.copyCollectionByBlankList(dto.getSupplyRelationshipDtos(), SupplyRelationshipDto.class,SupplyRelationshipEntity.class, HashSet.class, ArrayList.class));
    return entity;
  }

  /**
   * 用于客户信息导入
   *
   * @param dto
   * @return
   */
  @Override
  public CustomerEntity importCreate(CustomerDto dto) {
    this.createValidation(dto);
    this.validateContacts(dto);
    if (StringUtils.isEmpty(dto.getCustomerCode())) {
      String ruleCode = StringUtils.join("KH", DateFormatUtils.format(new Date(), "yyyyMMdd"));
      String code = this.generateCodeService.generateCode(ruleCode, 1, 5, 2, TimeUnit.DAYS).get(0);
      dto.setCustomerCode(code);
    }
    dto.setTenantCode(TenantUtils.getTenantCode());    //新增租户编号判断条件
    List<CustomerEntity> list =
            this.customerRepository.findByCodes(
                    Lists.newArrayList(dto.getCustomerCode()), dto.getTenantCode());
    Validate.isTrue(CollectionUtils.isEmpty(list), "客户编码已经存在");
    CustomerEntity entity =
            this.nebulaToolkitService.copyObjectByWhiteList(
                    dto, CustomerEntity.class, HashSet.class, ArrayList.class);
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    entity.setLockState(FreezeStatusEnum.NORMAL.getCode());
    entity.setProcessStatus(ProcessStatusEnum.PASS.getDictCode());
    //新增租户编号
    entity.setTenantCode(TenantUtils.getTenantCode());
    this.customerRepository.save(entity);
    // 绑定客户关联数据
    this.rebindRelationData(dto);
    return entity;
  }

  @Override
  @Transactional
  public CustomerEntity update(CustomerDto dto) {
    this.updateValidation(dto);
    this.validateContacts(dto);
    CustomerEntity entity = this.customerRepository.findDetailsByIdOrCode(dto.getId(), null, TenantUtils.getTenantCode());
    Validate.notNull(entity, "客户信息不存在");
    Validate.isTrue(entity.getCustomerCode().equals(dto.getCustomerCode()), "客户编码不能修改");
    CustomerEntity updateEntity =
            this.nebulaToolkitService.copyObjectByWhiteList(
                    dto, CustomerEntity.class, HashSet.class, ArrayList.class);
    //新增租户编号
    updateEntity.setTenantCode(TenantUtils.getTenantCode());
    //重构修改方法
    this.customerRepository.updateByIdAndTenantCode(updateEntity,TenantUtils.getTenantCode());
    CustomerVo oldVo = this.findByCustomerEntity(entity);
    // 绑定客户关联数据
    this.rebindRelationData(dto);
    // 业务日志更新
    CustomerEntity detailsByIdOrCode = this.customerRepository.findDetailsByIdOrCode(dto.getId(), null, TenantUtils.getTenantCode());
    CustomerEventDto customerEventDto = new CustomerEventDto();
    customerEventDto.setOriginal(oldVo);
    CustomerVo nowVo =
            this.nebulaToolkitService.copyObjectByWhiteList(
                detailsByIdOrCode, CustomerVo.class, HashSet.class, ArrayList.class,"dockingList","contactList","saleAreaList","billList","fileList");
    if (StringUtils.isNotBlank(dto.getOrgCode())) {
      final CustomerRelateOrgVo curOrgVo = new CustomerRelateOrgVo();
      curOrgVo.setOrgCode(dto.getOrgCode());
      nowVo.setOrgList(Lists.newArrayList(curOrgVo));
    }
    customerEventDto.setNewest(nowVo);
    SerializableBiConsumer<CustomerEventListener, CustomerEventDto> onUpdate =
            CustomerEventListener::onUpdate;
    //this.nebulaNetEventClient.publish(customerEventDto, CustomerEventListener.class, onUpdate);
    return updateEntity;
  }

  @Override
  @Transactional
  public void deleteBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "缺失id");
    List<CustomerEntity> entities = this.customerRepository.listByIdsAndTenantCode(ids,TenantUtils.getTenantCode());
    Validate.isTrue(
            CollectionUtils.isNotEmpty(entities) && entities.size() == ids.size(), "数据删除个数不匹配");
    this.customerRepository.updateDelFlagByIds(ids);
    // 删除业务
    CustomerEventDto customerEventDto = new CustomerEventDto();
    List<CustomerVo> customerVos =
            (List<CustomerVo>)
                    this.nebulaToolkitService.copyCollectionByWhiteList(
                            entities, CustomerEntity.class, CustomerVo.class, HashSet.class, ArrayList.class);
    if (CollectionUtils.isNotEmpty(customerVos)) {
      Map<String, Set<String>> map = Maps.newHashMap();
      Set<String> customerCodes =
          customerVos.stream()
              .filter(a -> StringUtils.isNotBlank(a.getCustomerCode()))
              .map(CustomerVo::getCustomerCode)
              .collect(Collectors.toSet());
      if(!CollectionUtils.isEmpty(customerCodes)){
        final List<CustomerROrgEntity> customerOrgList =
            this.customerROrgService.findByCustomerCodes(customerCodes);
        if(!CollectionUtils.isEmpty(customerOrgList)){
          map =
              customerOrgList.stream()
                  .filter(a -> StringUtils.isNoneBlank(a.getCustomerCode(), a.getOrgCode()))
                  .collect(
                      Collectors.groupingBy(
                          CustomerROrgEntity::getCustomerCode,
                          Collectors.mapping(CustomerROrgEntity::getOrgCode, Collectors.toSet())));
        }
      }

      for (CustomerVo customerVo : customerVos) {
        Set<String> orgCodes = map.get(customerVo.getCustomerCode());
        if (CollectionUtils.isNotEmpty(orgCodes)) {
          List<CustomerRelateOrgVo> orgList = Lists.newLinkedList();
          for (String orgCode : orgCodes) {
            final CustomerRelateOrgVo cur = new CustomerRelateOrgVo();
            cur.setOrgCode(orgCode);
            orgList.add(cur);
          }
          customerVo.setOrgList(orgList);
        }
        customerEventDto.setOriginal(customerVo);
        customerEventDto.setNewest(null);
        SerializableBiConsumer<CustomerEventListener, CustomerEventDto> onDelete =
                CustomerEventListener::onDelete;
       // this.nebulaNetEventClient.publish(customerEventDto, CustomerEventListener.class, onDelete);
      }
    }
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "缺失id");
    List<CustomerEntity> entities = this.customerRepository.listByIdsAndTenantCode(ids,TenantUtils.getTenantCode());
    Validate.isTrue(
            CollectionUtils.isNotEmpty(entities) && entities.size() == ids.size(), "数据启用个数不匹配");
    this.customerRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
    List<CustomerVo> voList =
            (List<CustomerVo>)
                    this.nebulaToolkitService.copyCollectionByBlankList(
                            entities, CustomerEntity.class, CustomerVo.class, HashSet.class, ArrayList.class);
    for (CustomerVo item : voList) {
      CustomerEventDto customerEventDto = new CustomerEventDto();
      customerEventDto.setOriginal(item);
      SerializableBiConsumer<CustomerEventListener, CustomerEventDto> onEnable =
              CustomerEventListener::onEnable;
      // this.nebulaNetEventClient.publish(customerEventDto, CustomerEventListener.class, onEnable);
    }
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "缺失id");
    List<CustomerEntity> entities = this.customerRepository.listByIdsAndTenantCode(ids,TenantUtils.getTenantCode());
    Validate.isTrue(
            CollectionUtils.isNotEmpty(entities) && entities.size() == ids.size(), "数据禁用个数不匹配");
    this.customerRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
    List<CustomerVo> voList =
            (List<CustomerVo>)
                    this.nebulaToolkitService.copyCollectionByBlankList(
                            entities, CustomerEntity.class, CustomerVo.class, HashSet.class, ArrayList.class);
    for (CustomerVo item : voList) {
      CustomerEventDto customerEventDto = new CustomerEventDto();
      customerEventDto.setOriginal(item);
      SerializableBiConsumer<CustomerEventListener, CustomerEventDto> onDisable =
              CustomerEventListener::onDisable;
      // this.nebulaNetEventClient.publish(customerEventDto, CustomerEventListener.class, onDisable);
    }
  }

  @Override
  @Transactional
  public void freezeBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "缺失id");
    List<CustomerEntity> entities = this.customerRepository.listByIdsAndTenantCode(ids,TenantUtils.getTenantCode());
    Validate.isTrue(
            CollectionUtils.isNotEmpty(entities) && entities.size() == ids.size(), "数据冻结个数不匹配");
    this.customerRepository.freezeBatch(ids, FreezeStatusEnum.FREEZE);
    List<CustomerVo> voList =
            (List<CustomerVo>)
                    this.nebulaToolkitService.copyCollectionByBlankList(
                            entities, CustomerEntity.class, CustomerVo.class, HashSet.class, ArrayList.class);
    for (CustomerVo item : voList) {
      CustomerEventDto customerEventDto = new CustomerEventDto();
      customerEventDto.setOriginal(item);
      SerializableBiConsumer<CustomerEventListener, CustomerEventDto> onFreeze =
              CustomerEventListener::onFreeze;
     // this.nebulaNetEventClient.publish(customerEventDto, CustomerEventListener.class, onFreeze);
    }
  }

  @Override
  @Transactional
  public void unfreezeBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "缺失id");
    List<CustomerEntity> entities = this.customerRepository.listByIdsAndTenantCode(ids,TenantUtils.getTenantCode());
    Validate.isTrue(
            CollectionUtils.isNotEmpty(entities) && entities.size() == ids.size(), "数据解冻个数不匹配");
    this.customerRepository.freezeBatch(ids, FreezeStatusEnum.NORMAL);
    //单个字段更新业务日志
    List<CustomerVo> voList =
            (List<CustomerVo>)
                    this.nebulaToolkitService.copyCollectionByBlankList(
                            entities, CustomerEntity.class, CustomerVo.class, HashSet.class, ArrayList.class);
    for (CustomerVo item : voList) {
      CustomerEventDto customerEventDto = new CustomerEventDto();
      customerEventDto.setOriginal(item);
      SerializableBiConsumer<CustomerEventListener, CustomerEventDto> onUnfreeze =
              CustomerEventListener::onUnfreeze;
    //  this.nebulaNetEventClient.publish(customerEventDto, CustomerEventListener.class, onUnfreeze);
    }
  }

  @Override
  public Page<CustomerEntity> findByCustomerSelectDto(Pageable pageable, CustomerSelectDto dto) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    dto = ObjectUtils.defaultIfNull(dto, new CustomerSelectDto());
    dto.setTenantCode(TenantUtils.getTenantCode());
    return this.customerRepository.findByCustomerSelectDto(pageable, dto);
  }

  @Override
  public CustomerEntity findDetailsByIdOrCode(String id, String customerCode) {
    if (StringUtils.isAllBlank(id, customerCode)) {
      return null;
    }
    return this.customerRepository.findDetailsByIdOrCode(
            id, customerCode, TenantUtils.getTenantCode());
  }

  @Override
  @Transactional
  public void rebindOrg(CustomerRebindOrgDto dto) {
    Validate.notBlank(dto.getOrgCode(), "缺失组织编码");
    Validate.notBlank(dto.getOriginOrgCode(), "缺失源组织编码");
    Validate.isTrue(CollectionUtils.isNotEmpty(dto.getCustomerCodeList()), "缺失客户编码");
    this.customerROrgService.rebindOrgCode(
            dto.getOrgCode(), dto.getOriginOrgCode(), dto.getCustomerCodeList());
  }

  @Override
  @Transactional
  public void rebindCustomerOrg(CustomerRebindOrgDto dto) {
    Validate.notBlank(dto.getOrgCode(), "缺失组织编码");
    Validate.notBlank(dto.getOriginOrgCode(), "缺失源组织编码");
    Validate.isTrue(CollectionUtils.isNotEmpty(dto.getCustomerCodeList()), "缺失客户编码");
    this.customerRepository.rebindCustomerOrgCode(
            dto.getOrgCode(),
            dto.getOriginOrgCode(),
            dto.getCustomerCodeList(),
            TenantUtils.getTenantCode());
  }

  @Override
  public List<CustomerEntity> findByMultipleConditionModel(MultipleConditionModel model) {
    if (Objects.isNull(model)) {
      return Lists.newArrayList();
    }
    model.setTenantCode(TenantUtils.getTenantCode());
    return this.customerRepository.findByMultipleConditionModel(model);
  }

  @Override
  public List<CustomerEntity> findByOrgCodes(List<String> orgCodeList) {
    if (CollectionUtils.isEmpty(orgCodeList)) {
      return Lists.newArrayList();
    }
    return this.customerRepository.findByOrgCodes(orgCodeList);
  }

  @Override
  public List<CustomerEntity> findByCustomerCodes(List<String> customerCodeList) {
    if (CollectionUtils.isEmpty(customerCodeList)) {
      return Lists.newArrayList();
    }
    String cacheKey = StringUtils.join(TenantUtils.getTenantCode(), customerCodeList);
    List<CustomerEntity> graph = cache.getIfPresent(cacheKey);
    if (graph == null) {
      graph = this.customerRepository.findByCodes(customerCodeList, TenantUtils.getTenantCode());
      cache.put(cacheKey, graph);
    }
    return graph;
  }

  /**
   * 绑定客户关联数据
   *
   * @param dto 客户信息dto
   */
  private void rebindRelationData(CustomerDto dto) {
    this.customerBillService.rebindCustomerCode(dto.getBillList(), dto.getCustomerCode());
    this.customerContactService.rebindCustomerCode(dto.getContactList(), dto.getCustomerCode());
    this.customerMediaService.rebindCustomerCode(dto.getFileList(), dto.getCustomerCode());
    this.customerDockingService.rebindCustomerCode(dto.getDockingList(), dto.getCustomerCode());
//    this.customerSaleAreaService.rebindCustomerCode(dto.getSaleAreaList(), dto.getCustomerCode());
//    customerROrgService.rebindCustomerCode(dto.getOrgCode(), dto.getCustomerCode());
    this.customerROrgService.rebindCustomerCode(dto);
  }

  /**
   * 在创建customer模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void createValidation(CustomerDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
//    if (StringUtils.isNotBlank(dto.getCustomerLevel())
//            && CustomerLevelEnum.T2.getDictCode().equals(dto.getCustomerLevel())) {
//      Validate.notBlank(dto.getParentCustomerCode(), "上级经销商不能为空");
//    }
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getCustomerName(), "缺失客户名称");
    Validate.isTrue(dto.getCustomerName().length() < 128, "客户名称，在进行添加时填入值超过了限定长度(128)，请检查!");
//    Validate.isTrue(
//            StringUtils.isBlank(dto.getCustomerCode()) || dto.getCustomerCode().length() < 64,
//            "客户编码，在进行添加时填入值超过了限定长度(64)，请检查!");


  }

  /**
   * 在修改customer模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void updateValidation(CustomerDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getId(), "修改信息时，id不能为空！");
    if (StringUtils.isNotBlank(dto.getCustomerLevel())
            && CustomerLevelEnum.T2.getDictCode().equals(dto.getCustomerLevel())) {
      Validate.notBlank(dto.getParentCustomerCode(), "上级经销商不能为空");
    }
    Validate.notBlank(dto.getCustomerCode(), "缺失客户编码");
    Validate.notBlank(dto.getCustomerName(), "缺失客户名称");
    Validate.isTrue(dto.getCustomerCode().length() < 128, "客户名称，在进行编辑时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(dto.getCustomerCode().length() < 64, "客户编码，在进行编辑时填入值超过了限定长度(64)，请检查!");
  }

  @Override
  public Page<CustomerEntity> findByParentCustomerIsNull(Pageable pageable) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    String status = EnableStatusEnum.ENABLE.getCode();
    return customerRepository.findByParentCustomerIsNull(pageable, status);
  }

  /**
   * 根据客户编码查询下级客户
   *
   * @param pageable
   * @param customerCode
   * @return
   */
  @Override
  public Page<CustomerEntity> findChildrenByCustomerCode(Pageable pageable, String customerCode) {
    String tenantCode = TenantUtils.getTenantCode();
    return customerRepository.findChildrenByCustomerCode(pageable, tenantCode, customerCode);
  }

  /**
   * 根据标签id查询
   *
   * @param pageable
   * @param tagId
   * @return
   */
  @Override
  public Page<CustomerEntity> findByTagId(Pageable pageable, String tagId) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (StringUtils.isBlank(tagId)) {
      return new Page<>();
    }
    return customerRepository.findByTagId(pageable, tagId);
  }

  /**
   * 根据客户编码或客户名称查询
   *
   * @param
   * @return
   */
  @Override
  public List<CustomerEntity> findByCustomerCodeLikeOrCustomerNameLike(String customerCodeLikeOrNameLike) {
    return this.customerRepository.findByCustomerCodeLikeOrCustomerNameLike(customerCodeLikeOrNameLike);
  }

  /**
   * 根据状态查询
   *
   * @param status
   * @return
   */
  @Override
  public List<CustomerEntity> findByStatus(String status) {
    if (StringUtils.isBlank(status)) {
      return null;
    }
    return customerRepository.findByStatus(status);
  }

  @Override
  public void onProcessSuccess(CustomerEntity customer) {
    //事件引擎通知
    CustomerNebulaEventDto customerNebulaEventDto = this.nebulaToolkitService.copyObjectByWhiteList(customer, CustomerNebulaEventDto.class, HashSet.class, ArrayList.class);
    SerializableBiConsumer<CustomerNebulaEventListener, CustomerNebulaEventDto> onCreateNebulaEvent = CustomerNebulaEventListener::onCreate;
   // this.nebulaNetEventClient.publish(customerNebulaEventDto, CustomerNebulaEventListener.class, onCreateNebulaEvent);
  }

  /**
   * 校验联系人信息
   *
   * @param dto
   */
  private void validateContacts(CustomerDto dto) {
    Validate.notNull(dto, "终端信息缺失");
    List<CustomerContactDto> contacts = dto.getContactList();
    if (CollectionUtils.isEmpty(contacts)) {
      return;
    }
    final List<CustomerContactDto> list =
        contacts.stream()
            .filter(a -> !StringUtils.isAllBlank(a.getContactName(), a.getContactPhone()))
            .collect(Collectors.toList());
    dto.setContactList(list);
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    // 校验字段
    for (CustomerContactDto contact : contacts) {
      if (StringUtils.isNotBlank(contact.getContactName())
              || StringUtils.isNotBlank(contact.getContactPhone())
              || !Objects.isNull(contact.getContactMain())) {
        Validate.notBlank(contact.getContactPhone(), "缺失联系电话");
        Validate.notNull(contact.getContactMain(), "请选择是否是主联系人");
        Validate.notBlank(contact.getContactName(), "缺失姓名");
      }
    }
    // 校验是否存在主联系人
    List<CustomerContactDto> contanctsByMain =
            contacts.stream().filter(o -> o.getContactMain() == true).collect(Collectors.toList());
    Validate.isTrue(contanctsByMain.size() == 1, "必须有且仅有一个主联系人");
  }

  @Autowired(required = false)
  private ProcessBusinessService processBusinessService;

  @Value("${crm.business.customer.process-key:}")
  private String defaultProcessKey;

  /**
   * 经销商新增提交工作流进行审批，提交成功返回流程实例ID，提交失败则抛出异常
   *
   * @param dto 授信请求DTO
   */
  private String commitProcess(CustomerDto dto) {
    ProcessBusinessDto processBusiness = dto.getProcessBusiness();
    String processKey = processBusiness.getProcessKey();
    if (StringUtils.isBlank(processKey)) {
      processBusiness.setProcessKey(defaultProcessKey);
    } else {
      processBusiness.setProcessKey(processKey);
    }
    processBusiness.setBusinessNo(dto.getCustomerCode());
    processBusiness.setBusinessFormJson(JsonUtils.obj2JsonString(dto));
    processBusiness.setBusinessCode(CustomerConstant.CUSTOMER_PROCESS_NAME);
    processBusiness.setProcessTitle(CUSTOMER_PROCESS_NAME);
    ProcessBusinessVo processBusinessVo = processBusinessService.processStart(processBusiness);
    return processBusinessVo.getProcessNo();
  }

  /**
   * 根据实体信息获取vo信息，带组织信息
   *
   * @param entity
   * @return
   */
  private CustomerVo findByCustomerEntity(CustomerEntity entity) {
    if (Objects.isNull(entity) || StringUtils.isBlank(entity.getCustomerCode())) {
      return null;
    }
    CustomerVo vo =
        this.nebulaToolkitService.copyObjectByWhiteList(
            entity,
            CustomerVo.class,
            HashSet.class,
            ArrayList.class,
            "dockingList",
            "contactList",
            "saleAreaList",
            "billList",
            "fileList");
    List<CustomerROrgEntity> orgList =
        this.customerROrgService.findByCustomerCodes(Sets.newHashSet(entity.getCustomerCode()));
    if (CollectionUtils.isEmpty(orgList)) {
      return vo;
    }
    List<CustomerRelateOrgVo> relateOrgList = Lists.newLinkedList();
    for (CustomerROrgEntity item : orgList) {
      final CustomerRelateOrgVo curOrgVo = new CustomerRelateOrgVo();
      curOrgVo.setOrgCode(item.getOrgCode());
      relateOrgList.add(curOrgVo);
    }
    vo.setOrgList(relateOrgList);
    return vo;
  }

  @Override
  public Page<CustomerPageVo> findByCustomerPageDto(Pageable pageable, CustomerPageDto dto) {
    Page<CustomerPageVo> page = new Page<>(pageable.getPageNumber(),pageable.getPageSize());
    dto.setTenantCode(TenantUtils.getTenantCode());
    dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    return this.customerRepository.findByCustomerPageDto(page,dto);
  }
}