package com.biz.crm.mdm.business.customer.local.service.internal;

import cn.hutool.core.collection.CollUtil;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.customer.local.entity.*;
import com.biz.crm.mdm.business.customer.local.repository.CustomerDockingRepository;
import com.biz.crm.mdm.business.customer.local.repository.CustomerRepository;
import com.biz.crm.mdm.business.customer.local.repository.DealerSalesRegionRepository;
import com.biz.crm.mdm.business.customer.local.service.CustomerContactService;
import com.biz.crm.mdm.business.customer.local.service.CustomerROrgService;
import com.biz.crm.mdm.business.customer.local.service.CustomerService;
import com.biz.crm.mdm.business.customer.local.service.CustomerTagService;
import com.biz.crm.mdm.business.customer.local.service.*;
import com.biz.crm.mdm.business.customer.org.sdk.service.CustomerOrgVoSdkService;
import com.biz.crm.mdm.business.customer.org.sdk.vo.CustomerOrgVo;
import com.biz.crm.mdm.business.customer.sdk.dto.*;
import com.biz.crm.mdm.business.customer.sdk.enums.CustomerExtendEunm;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.service.DealerOrgRelationshipService;
import com.biz.crm.mdm.business.customer.sdk.service.SupplyRelationshipSdkService;
import com.biz.crm.mdm.business.customer.sdk.vo.*;
import com.biz.crm.mdm.business.org.sdk.common.constant.OrgCodeConstant;
import com.biz.crm.mdm.business.org.sdk.dto.OrgQueryDto;
import com.biz.crm.mdm.business.org.sdk.dto.RelateOrgCodeQueryDto;
import com.biz.crm.mdm.business.org.sdk.service.OrgVoService;
import com.biz.crm.mdm.business.org.sdk.vo.OrgVo;
import com.biz.crm.mdm.business.position.sdk.service.PositionVoService;
import com.biz.crm.mdm.business.position.sdk.vo.AbstractRelationView;
import com.biz.crm.mdm.business.position.sdk.vo.PositionVo;
import com.biz.crm.mdm.business.region.sdk.service.RegionVoService;
import com.biz.crm.mdm.business.user.sdk.service.UserInfoVoService;
import com.biz.crm.mdm.business.user.sdk.vo.UserInfoVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.Map.Entry;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 客户信息VO服务接口实现类
 *
 * @author ning.zhang
 * @date 2021/10/27
 */
@Service
public class CustomerVoServiceImpl implements CustomerVoService {

  @Autowired(required = false)
  private CustomerService customerService;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private OrgVoService orgVoService;
  @Autowired(required = false)
  private CustomerOrgVoSdkService customerOrgVoSdkService;
  @Autowired(required = false)
  private UserInfoVoService userInfoVoService;
  @Autowired(required = false)
  private PositionVoService positionVoService;
  @Autowired(required = false)
  private CustomerRepository customerRepository;
  @Autowired(required = false)
  private CustomerROrgService customerROrgService;
  @Autowired(required = false)
  private CustomerDockingRepository customerDockingRepository;
  @Autowired(required = false)
  private CustomerContactService customerContactService;
  @Autowired
  private CustomerTagService customerTagService;
  @Autowired(required = false)
  private DealerOrgRelationshipService dealerOrgRelationshipService;
  @Autowired(required = false)
  private SupplyRelationshipSdkService supplyRelationshipSdkService;

  @Autowired(required = false)
  private DealerSalesRegionRepository dealerSalesRegionRepository;

  @Autowired(required = false)
  private CustomerSaleAreaService customerSaleAreaService;


  /**
   * 基于数据库执行的数据视图执行内容缓存（最多500毫秒）
   */
  private static volatile Cache<String, CustomerVo> cache = null;

  public CustomerVoServiceImpl(){
    if(cache == null) {
      synchronized (CustomerVoServiceImpl.class) {
        while(cache == null) {
          cache = CacheBuilder.newBuilder()
              .initialCapacity(10000)
              .expireAfterWrite(500, TimeUnit.MILLISECONDS)
              .maximumSize(100000)
              .build();
        }
      }
    }
  }

  @Override
  public CustomerVo findDetailsByIdOrCode(String id, String customerCode) {
    if (StringUtils.isAllBlank(id, customerCode)) {
      return null;
    }
    CustomerVo customerVo;
    String cacheKey = StringUtils.join(TenantUtils.getTenantCode(), id, customerCode);
    if ((customerVo = cache.getIfPresent(cacheKey)) == null) {
      synchronized (this){
        if ((customerVo = cache.getIfPresent(cacheKey)) == null) {
          CustomerEntity entity = customerService.findDetailsByIdOrCode(id, customerCode);
          if (Objects.isNull(entity)) {
            return null;
          }
          customerVo = this.covertEntityToVo(entity);
          this.perfectDockingInfo(customerVo.getDockingList());
          this.perfectSaleAreaInfo(customerVo.getSaleAreaList());
          cache.put(cacheKey, customerVo);
        }
      }
    }

    return customerVo;
  }

  /**
   * 完善销售区域信息
   * 1.获取销售区域编码集合
   * 2.获取销售区域集合对应区域信息映射MAP
   * 3.遍历销售区域信息,设置销售区域其他信息
   *
   * @param saleAreaList 销售区域信息
   */
  private void perfectSaleAreaInfo(List<CustomerSaleAreaVo> saleAreaList) {
    if (CollectionUtils.isEmpty(saleAreaList)) {
      return;
    }
    List<String> orgCodeList = Lists.newArrayList();
    saleAreaList.forEach(customerSaleAreaVo -> {
      if (StringUtils.isNotBlank(customerSaleAreaVo.getAreaOne())) {
        orgCodeList.add(customerSaleAreaVo.getAreaOne());
      }
      if (StringUtils.isNotBlank(customerSaleAreaVo.getAreaTwo())) {
        orgCodeList.add(customerSaleAreaVo.getAreaTwo());
      }
      if (StringUtils.isNotBlank(customerSaleAreaVo.getAreaThree())) {
        orgCodeList.add(customerSaleAreaVo.getAreaThree());
      }
    });
    if (CollectionUtils.isEmpty(orgCodeList)) {
      return;
    }
    List<OrgVo> orgList = this.orgVoService.findByOrgCodes(orgCodeList);
    if (CollectionUtils.isEmpty(orgList)) {
      return;
    }
    Map<String, String> orgNameMap = orgList.stream().collect(Collectors.toMap(OrgVo::getOrgCode, OrgVo::getOrgName, (key1, key2) -> key2));
    for (CustomerSaleAreaVo customerSaleAreaVo : saleAreaList) {
      customerSaleAreaVo.setAreaOneName(orgNameMap.get(customerSaleAreaVo.getAreaOne()));
      customerSaleAreaVo.setAreaTwoName(orgNameMap.get(customerSaleAreaVo.getAreaTwo()));
      customerSaleAreaVo.setAreaThreeName(orgNameMap.get(customerSaleAreaVo.getAreaThree()));
    }
  }

  /**
   * 完善对接人信息
   * 1.获取对接人职位编码集合
   * 2.获取职位集合对应用户信息映射MAP和对应职位详细信息映射MAP
   * 3.遍历对接人信息,设置用户和职位组织信息
   *
   * @param dockingList 对接人信息
   */
  private void perfectDockingInfo(List<CustomerDockingVo> dockingList) {
    if (CollectionUtils.isEmpty(dockingList)) {
      return;
    }
    Set<String> positionCodes = dockingList.stream().map(CustomerDockingVo::getPositionCode).collect(Collectors.toSet());
    Map<String, UserInfoVo> userMap = Maps.newHashMap();
    Map<String, PositionVo> positionMap = Maps.newHashMap();
    List<UserInfoVo> userList = this.userInfoVoService.findByPositionCodes(positionCodes);
    if (!CollectionUtils.isEmpty(userList)) {
      userMap = userList.stream().collect(Collectors.toMap(UserInfoVo::getPositionCode, t -> t, (key1, key2) -> key2));
    }
    List<PositionVo> positionList = positionVoService.findDetailsByIdsOrCodes(null, Lists.newArrayList(positionCodes));
    if (!CollectionUtils.isEmpty(positionList)) {
      positionMap = positionList.stream().collect(Collectors.toMap(PositionVo::getPositionCode, t -> t, (key1, key2) -> key2));
    }
    //完善对接人用户和组织信息
    this.perfectDockingPositionUserInfo(dockingList, userMap, positionMap);
    //完善对接人组织信息
    this.perfectDockingOrgInfo(dockingList, positionMap);
  }

  /**
   * 完善对接人职位用户信息
   *
   * @param dockingList 对接口信息列表
   * @param userMap     用户信息映射
   * @param positionMap 职位信息映射
   */
  private void perfectDockingPositionUserInfo(List<CustomerDockingVo> dockingList, Map<String, UserInfoVo> userMap, Map<String, PositionVo> positionMap) {
    for (CustomerDockingVo customerDockingVo : dockingList) {
      UserInfoVo userInfoVo = userMap.get(customerDockingVo.getPositionCode());
      PositionVo positionVo = positionMap.get(customerDockingVo.getPositionCode());
      if (Objects.nonNull(userInfoVo)) {
        customerDockingVo.setFullName(userInfoVo.getFullName());
        customerDockingVo.setContact(userInfoVo.getUserPhone());
        customerDockingVo.setUserName(userInfoVo.getUserName());
      }
      if (Objects.nonNull(positionVo)) {
        customerDockingVo.setPositionName(positionVo.getPositionName());
      }
    }
  }

  /**
   * 完善对接人组织信息
   *
   * @param dockingList 对接口信息列表
   */
  private void perfectDockingOrgInfo(List<CustomerDockingVo> dockingList, Map<String, PositionVo> positionMap) {
    Set<String> orgCodeSet = Sets.newHashSet();
    dockingList.forEach(customerDockingVo -> {
      PositionVo positionVo = positionMap.get(customerDockingVo.getPositionCode());
      if (Objects.isNull(positionVo) || CollectionUtils.isEmpty(positionVo.getRelationData())) {
        return;
      }
      positionVo.getRelationData().stream().filter(prv -> OrgCodeConstant.KEY.equals(prv.getRelationKey()) && !CollectionUtils.isEmpty(prv.getRelationData()))
          .collect(Collectors.toList()).forEach(positionRelationVo -> {
        List<String> orgCodes = positionRelationVo.getRelationData().stream().map(AbstractRelationView::getCode).collect(Collectors.toList());
        customerDockingVo.setOrgCode(String.join(",", orgCodes));
        orgCodeSet.addAll(orgCodes);
      });
    });
    if (CollectionUtils.isEmpty(orgCodeSet)) {
      return;
    }
    List<OrgVo> orgVoList = this.orgVoService.findByOrgCodes(Lists.newArrayList(orgCodeSet));
    if (CollectionUtils.isEmpty(orgVoList)) {
      return;
    }
    Map<String, OrgVo> orgMap = orgVoList.stream()
        .collect(Collectors.toMap(OrgVo::getOrgCode, t -> t, (key1, key2) -> key2));
    dockingList.forEach(customerDockingVo -> {
      String orgCode = customerDockingVo.getOrgCode();
      if (StringUtils.isBlank(orgCode)) {
        return;
      }
      List<String> orgNames = Arrays.stream(orgCode.split(",")).map(s -> {
        OrgVo orgVo = orgMap.get(s);
        return Objects.isNull(orgVo) ? null : orgVo.getOrgName();
      }).filter(Objects::nonNull).collect(Collectors.toList());
      if (!CollectionUtils.isEmpty(orgNames)) {
        customerDockingVo.setOrgName(String.join(",", orgNames));
      }
    });
  }

  @Override
  public Page<CustomerVo> findByCustomerSelectDto(Pageable pageable, CustomerSelectDto dto) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    dto = ObjectUtils.defaultIfNull(dto, new CustomerSelectDto());
    dto.setTenantCode(TenantUtils.getTenantCode());
    Page<CustomerEntity> entityPage = customerService.findByCustomerSelectDto(pageable, dto);
    Page<CustomerVo> pageResult = new Page<>(entityPage.getCurrent(), entityPage.getSize(),
        entityPage.getTotal());
    pageResult.setRecords(covertEntityToVo(entityPage.getRecords()));
    return pageResult;
  }

  /**
   * 分页查询
   *
   * @param pageable 分页信息
   * @param dto  分页参数dto
   * @return 查询结果
   */
  public Page<CustomerVo> findByConditions(Pageable pageable, CustomerDto dto){
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new CustomerDto());
    Page<CustomerEntity> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    dto.setTenantCode(TenantUtils.getTenantCode());
    Page<CustomerEntity> entityPage = this.customerRepository.findByConditions(page, dto);
    Page<CustomerVo> pageResult =
      new Page<>(entityPage.getCurrent(), entityPage.getSize(), entityPage.getTotal());
    if (CollUtil.isEmpty(entityPage.getRecords())) {
      return pageResult;
    }
    List<String> list = entityPage.getRecords().stream().map(CustomerEntity::getCustomerCode).collect(Collectors.toList());
    List<CustomerTagEntity> tagEntityList = customerTagService.findByCustomerCodes(list);
    List<CustomerVo> terminalVoList = this.buildTerminalVoList(entityPage.getRecords(), tagEntityList);
    pageResult.setRecords(terminalVoList);
    return pageResult;
  }

  /**
   * 封装数据
   *
   * @param customerEntities 经销商数据
   * @param tagEntityList    标签数据
   * @return
   */
  private List<CustomerVo> buildTerminalVoList(List<CustomerEntity> customerEntities, List<CustomerTagEntity> tagEntityList) {
    List<CustomerVo> customerVos = new ArrayList<>();
    if (CollUtil.isEmpty(customerEntities)) {
      return customerVos;
    }
    customerVos =
      (List<CustomerVo>)
        this.nebulaToolkitService.copyCollectionByWhiteList(
          customerEntities, CustomerEntity.class, CustomerVo.class, HashSet.class, ArrayList.class);
    Map<String, List<CustomerTagVo>> tagVos = Maps.newHashMap();
    // 标签
    if (CollUtil.isNotEmpty(tagEntityList)) {
      List<CustomerTagVo> list =
        (List<CustomerTagVo>)
          this.nebulaToolkitService.copyCollectionByWhiteList(
            tagEntityList,
            CustomerTagEntity.class,
            CustomerTagVo.class,
            HashSet.class,
            ArrayList.class);
      tagVos = list.stream().collect(Collectors.groupingBy(CustomerTagVo::getCustomerCode));
    }
    for (CustomerVo customerVo : customerVos) {
      customerVo.setTagVos(tagVos.get(customerVo.getCustomerCode()));
    }
    return customerVos;
  }

  /**
   * 批量设置是否分利
   *
   * @param customerCodeList 编码集合
   * @param bool             设置值
   */
  @Override
  @Transactional
  public void modifyShareBenefits(List<String> customerCodeList, Boolean bool){
    if (CollUtil.isNotEmpty(customerCodeList)){
      customerRepository.modifyShareBenefits(customerCodeList, bool);
    }
  }

  @Override
  public List<CustomerVo> findByOrgCodes(List<String> orgCodeList) {
    if (CollectionUtils.isEmpty(orgCodeList)) {
      return Lists.newLinkedList();
    }
    List<CustomerEntity> list = this.customerService.findByOrgCodes(orgCodeList);
    if (CollectionUtils.isEmpty(list)) {
      return Lists.newLinkedList();
    }
    return (List<CustomerVo>) this.nebulaToolkitService
        .copyCollectionByBlankList(list, CustomerEntity.class, CustomerVo.class, HashSet.class,
            ArrayList.class);
  }

  @Override
  public List<CustomerVo> findByCustomerCodes(List<String> customerCodeList) {
    if (CollectionUtils.isEmpty(customerCodeList)) {
      return Lists.newLinkedList();
    }
    List<CustomerEntity> list = this.customerService.findByCustomerCodes(customerCodeList);
    if (CollectionUtils.isEmpty(list)) {
      return Lists.newLinkedList();
    }
    List<CustomerVo> customerVos =
        (List<CustomerVo>) this.nebulaToolkitService.copyCollectionByBlankList(list, CustomerEntity.class, CustomerVo.class, HashSet.class, ArrayList.class);
    List<String> customerCodes = customerVos.stream()
        .filter(a -> StringUtils.isNotBlank(a.getCustomerCode()))
        .map(CustomerVo::getCustomerCode)
        .collect(Collectors.toList());
    //封装企业组织信息
    this.buildOrgInfo(customerCodes, customerVos);
    //封装客户组织信息
    this.buildCustomerOrgInfo(customerVos);
    //封装客户对接人信息
    this.buildDockingInfo(customerCodes, customerVos);
    //封装客户联系人信息
    this.buildContactInfo(customerCodes, customerVos);
    //封装标签数据

    return customerVos;
  }

  @Override
  public List<CustomerVo> findCustomerAndContactByCustomerCodes(List<String> customerCodeList) {
    if (CollectionUtils.isEmpty(customerCodeList)) {
      return Lists.newLinkedList();
    }
    List<CustomerEntity> customerEntities = this.customerService.findByCustomerCodes(customerCodeList);
    if (CollectionUtils.isEmpty(customerEntities)) {
      return Lists.newLinkedList();
    }
    List<CustomerVo> customerVos = (List<CustomerVo>) this.nebulaToolkitService.copyCollectionByBlankList(
            customerEntities, CustomerEntity.class, CustomerVo.class, HashSet.class, ArrayList.class);
    List<String> customerCodes = customerVos.stream()
            .map(CustomerVo::getCustomerCode)
            .filter(StringUtils::isNotBlank)
            .collect(Collectors.toList());
    this.buildContactInfo(customerCodes, customerVos);
    return customerVos;
  }

  /**
   * 封装客户组织信息
   *
   * @param customerVos
   */
  private void buildCustomerOrgInfo(List<CustomerVo> customerVos) {
    if (CollectionUtils.isEmpty(customerVos)) {
      return;
    }
    Set<String> customerOrgCodeSet =
        customerVos.stream()
            .filter(a -> StringUtils.isNotBlank(a.getCustomerOrgCode()))
            .map(CustomerVo::getCustomerOrgCode)
            .collect(Collectors.toSet());
    if (CollectionUtils.isEmpty(customerOrgCodeSet)) {
      return;
    }
    List<CustomerOrgVo> orgVoList =
        this.customerOrgVoSdkService.findListByCodes(Lists.newArrayList(customerOrgCodeSet));
    if (CollectionUtils.isEmpty(orgVoList)) {
      return;
    }
    Map<String, String> map =
        orgVoList.stream()
            .filter(a -> StringUtils.isNoneBlank(a.getCustomerOrgCode(), a.getCustomerOrgName()))
            .collect(
                Collectors.toMap(
                    CustomerOrgVo::getCustomerOrgCode,
                    CustomerOrgVo::getCustomerOrgName,
                    (a, b) -> a));
    for (CustomerVo item : customerVos) {
      if (StringUtils.isBlank(item.getCustomerOrgCode())) {
        continue;
      }
      item.setCustomerOrgName(map.get(item.getCustomerOrgCode()));
    }
  }

  /**
   * 封装客户联系人信息
   *
   * @param customerCodes 客户编码集合
   * @param customerVos   待封装客户信息
   */
  private void buildContactInfo(List<String> customerCodes, List<CustomerVo> customerVos) {
    List<CustomerContactVo> contactVos = this.customerContactService.findByCustomerCodes(customerCodes);
    if (CollectionUtils.isEmpty(contactVos)) {
      return;
    }
    // k-customerCode,v-contactList
    Map<String, List<CustomerContactVo>> customerContactVoMap = contactVos.stream().collect(Collectors.groupingBy(CustomerContactVo::getCustomerCode));
    customerVos.forEach(customerVo -> customerVo.setContactList(customerContactVoMap.get(customerVo.getCustomerCode())));
  }

  /**
   * 封装客户对接人信息
   *
   * @param customerCodes 客户编码集合
   * @param customerVos   待封装客户信息
   */
  private void buildDockingInfo(List<String> customerCodes, List<CustomerVo> customerVos) {
    List<CustomerDockingEntity> dockingEntities =
        this.customerDockingRepository.findByCustomerCodes(customerCodes, TenantUtils.getTenantCode());
    if (CollectionUtils.isEmpty(dockingEntities)) {
      return;
    }
    // k-customerCode,v-dockingList
    Map<String, List<CustomerDockingEntity>> dockingMap = Maps.newHashMap();
    dockingEntities.forEach(customerDockingEntity -> {
      List<CustomerDockingEntity> list = dockingMap.getOrDefault(customerDockingEntity.getCustomerCode(), Lists.newArrayList());
      list.add(customerDockingEntity);
      dockingMap.put(customerDockingEntity.getCustomerCode(), list);
    });
    List<CustomerDockingVo> allDockingList = Lists.newArrayList();
    customerVos.forEach(customerVo -> {
      List<CustomerDockingEntity> entities = dockingMap.get(customerVo.getCustomerCode());
      if (CollectionUtils.isEmpty(entities)) {
        return;
      }
      customerVo.setDockingList((List<CustomerDockingVo>) this.nebulaToolkitService
          .copyCollectionByBlankList(entities, CustomerDockingEntity.class, CustomerDockingVo.class, HashSet.class, ArrayList.class));
      allDockingList.addAll(customerVo.getDockingList());
    });
    //完善对接人信息
    this.perfectDockingInfo(allDockingList);
  }

  /**
   * 封装企业组织信息
   *
   * @param customerCodes 客户编码集合
   * @param customerVos   待封装客户信息
   */
  private void buildOrgInfo(List<String> customerCodes, List<CustomerVo> customerVos) {
    List<CustomerROrgEntity> customerROrgEntityList = Lists.newLinkedList();
    if (!CollectionUtils.isEmpty(customerCodes)) {
      customerROrgEntityList =
          this.customerROrgService.findByCustomerCodes(new HashSet<>(customerCodes));
      Validate.isTrue(!CollectionUtils.isEmpty(customerROrgEntityList), "客户编码%s未查询到组织信息！", customerCodes);
    }
    // k-customerCode,v-orgCodeList
    Map<String, List<String>> map =
        customerROrgEntityList.stream()
            .filter(a -> StringUtils.isNoneBlank(a.getCustomerCode(), a.getOrgCode()))
            .collect(
                Collectors.groupingBy(
                    CustomerROrgEntity::getCustomerCode,
                    Collectors.mapping(CustomerROrgEntity::getOrgCode, Collectors.toList())));

    Set<String> orgCodeSet =
        customerROrgEntityList.stream()
            .filter(a -> StringUtils.isNotBlank(a.getOrgCode()))
            .map(CustomerROrgEntity::getOrgCode)
            .collect(Collectors.toSet());
    List<OrgVo> orgVoList = this.orgVoService.findByOrgCodes(Lists.newArrayList(orgCodeSet));
    Map<String, String> orgMap =
        orgVoList.stream()
            .filter(a -> StringUtils.isNoneBlank(a.getOrgCode(), a.getOrgName()))
            .collect(Collectors.toMap(OrgVo::getOrgCode, OrgVo::getOrgName, (a, b) -> a));

    for (CustomerVo item : customerVos) {
      List<String> curOrgCodeList = map.get(item.getCustomerCode());
      if (CollectionUtils.isEmpty(curOrgCodeList)) {
        continue;
      }
      List<CustomerRelateOrgVo> curOrgList = Lists.newLinkedList();
      for (String orgCode : curOrgCodeList) {
        if (!orgMap.containsKey(orgCode)) {
          continue;
        }
        CustomerRelateOrgVo cur = new CustomerRelateOrgVo();
        cur.setCustomerCode(item.getCustomerCode());
        cur.setOrgCode(orgCode);
        cur.setOrgName(orgMap.get(orgCode));
        curOrgList.add(cur);
      }
      item.setOrgList(curOrgList);
    }
  }

  @Override
  public List<CustomerVo> findByChannels(List<String> channelList) {
    if (CollectionUtils.isEmpty(channelList)) {
      return Lists.newLinkedList();
    }
    List<CustomerEntity> list = this.customerRepository.findByChannels(channelList, TenantUtils.getTenantCode());
    if (CollectionUtils.isEmpty(list)) {
      return Lists.newLinkedList();
    }
    return (List<CustomerVo>) this.nebulaToolkitService
        .copyCollectionByBlankList(list, CustomerEntity.class, CustomerVo.class, HashSet.class,
            ArrayList.class);
  }

  @Override
  public List<CustomerVo> findByTypes(List<String> typeList) {
    if (CollectionUtils.isEmpty(typeList)) {
      return Lists.newLinkedList();
    }
    List<CustomerEntity> list = this.customerRepository.findByTypes(typeList, TenantUtils.getTenantCode());
    if (CollectionUtils.isEmpty(list)) {
      return Lists.newLinkedList();
    }
    return (List<CustomerVo>) this.nebulaToolkitService
        .copyCollectionByBlankList(list, CustomerEntity.class, CustomerVo.class, HashSet.class,
            ArrayList.class);
  }

  @Override
  public Set<String> findCustomerCodesByOrgCodesAndChannelsAndTags(List<String> orgCodes, List<String> channels, List<String> tags) {
    return this.customerRepository.findCustomerCodesByOrgCodesAndChannelsAndTags(orgCodes, channels, tags);
  }

  @Override
  public List<CustomerVo> findForPriceByCustomerCodes(Set<String> customerCodeSet) {
    if (CollectionUtils.isEmpty(customerCodeSet)) {
      return Lists.newLinkedList();
    }
    List<CustomerEntity> list = this.customerService.findByCustomerCodes(Lists.newArrayList(customerCodeSet));
    if (CollectionUtils.isEmpty(list)) {
      return Lists.newLinkedList();
    }
    List<CustomerVo> customerVos =
        (List<CustomerVo>) this.nebulaToolkitService.copyCollectionByBlankList(list, CustomerEntity.class, CustomerVo.class, HashSet.class, ArrayList.class);
    List<String> customerCodes = customerVos.stream()
        .filter(a -> StringUtils.isNotBlank(a.getCustomerCode()))
        .map(CustomerVo::getCustomerCode)
        .collect(Collectors.toList());
    //封装企业组织信息
    this.buildOrgInfo(customerCodes, customerVos);
    return customerVos;
  }

  @Override
  public List<CustomerVo> findByAmapIds(Set<String> amapIds) {
    if (org.apache.commons.collections.CollectionUtils.isEmpty(amapIds)) {
      return Lists.newArrayList();
    }
    List<CustomerEntity> list = this.customerRepository.findByAmapIds(amapIds);
    if (org.apache.commons.collections.CollectionUtils.isEmpty(list)) {
      return Lists.newArrayList();
    }
    return (List<CustomerVo>) this.nebulaToolkitService.copyCollectionByWhiteList(list, CustomerEntity.class, CustomerVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public Set<String> findByCustomerQueryDto(CustomerQueryDto dto) {
    if (Objects.isNull(dto)
        || StringUtils.isAllBlank(
        dto.getCustomerCode(),
        dto.getCustomerName(),
        dto.getChannel(),
        dto.getOrgName(),
        dto.getDelFlag(),
        dto.getProcessStatus())) {
      return Sets.newHashSet();
    }

    if (StringUtils.isNotBlank(dto.getOrgName())) {
      final OrgQueryDto queryDto = new OrgQueryDto();
      queryDto.setOrgName(dto.getOrgName());
      final Set<String> orgCodeSet = this.orgVoService.findByOrgQueryDto(queryDto);
      if (CollectionUtils.isEmpty(orgCodeSet)) {
        return Sets.newHashSet();
      }
      dto.setOrgCodeSet(orgCodeSet);
    }

    return this.customerRepository.findByCustomerQueryDto(dto);
  }

  @Override
  public Map<String, Set<String>> findAllowSaleCustomerByOrgCodes(Set<String> orgCodes) {
    if (CollectionUtils.isEmpty(orgCodes)) {
      return Maps.newHashMap();
    }
    final RelateOrgCodeQueryDto queryDto = new RelateOrgCodeQueryDto();
    queryDto.setOrgCodeSet(orgCodes);
    queryDto.setSearchType(-1);
    final Map<String, String> orgRuleMap = this.orgVoService.findByRelateOrgCodeQueryDto(queryDto);
    if (CollectionUtils.isEmpty(orgRuleMap)) {
      return Maps.newHashMap();
    }
    List<CustomerROrgEntity> list =
        this.customerROrgService.findAllowSaleCustomerByOrgCodes(orgRuleMap.keySet());
    if (CollectionUtils.isEmpty(list)) {
      return Maps.newHashMap();
    }
    Map<String, Set<String>> map =
        list.stream()
            .filter(
                a ->
                    StringUtils.isNoneBlank(a.getCustomerCode(), a.getOrgCode())
                        && orgRuleMap.keySet().contains(a.getOrgCode()))
            .collect(
                Collectors.groupingBy(
                    CustomerROrgEntity::getCustomerCode,
                    Collectors.mapping(CustomerROrgEntity::getOrgCode, Collectors.toSet())));
    Map<String, Set<String>> re = Maps.newHashMap();
    for (Entry<String, Set<String>> item : map.entrySet()) {
      Set<String> rule = Sets.newHashSet();
      for (String orgCode : item.getValue()) {
        final String s = orgRuleMap.get(orgCode);
        if (StringUtils.isBlank(s)) {
          continue;
        }
        rule.add(s);
      }
      if (CollectionUtils.isEmpty(rule)) {
        continue;
      }
      re.put(item.getKey(), rule);
    }
    return re;
  }

  @Override
  public List<CustomerVo> findCountByCreateAccountAndFromTypeAndCreateTimeScope(String createAccount, String fromType, String startDate, String endDate) {
    if (StringUtils.isAnyBlank(createAccount, fromType, startDate, endDate)) {
      return new ArrayList<>(0);
    }
    List<CustomerEntity> customerEntities = this.customerRepository.findCountByCreateAccountAndFromTypeAndCreateTimeScope(createAccount, fromType, startDate, endDate);
    return (List<CustomerVo>) this.nebulaToolkitService.copyCollectionByWhiteList(customerEntities, CustomerEntity.class, CustomerVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public Boolean existByCustomerCodeAndChannels(List<String> channelList, String customerCode) {
    CustomerEntity entity = customerRepository.findByCustomerCodeAndChannels(customerCode ,channelList);
    return ObjectUtils.isNotEmpty(entity);
  }

  @Override
  public Boolean existByCustomerCode(String customerCode) {
    CustomerEntity entity = customerRepository.findByCustomerCode(customerCode);
    return ObjectUtils.isNotEmpty(entity);
  }

  @Override
  public Boolean existByCustomer7OrgIn7OrgNotIn(String customerCode, List<String> orgCodeIn, List<String> orgCodeNotIn) {
    if (CollectionUtils.isEmpty(orgCodeIn)) {
      return false;
    }
    List<CustomerEntity> entities = customerRepository.findByCustomer7OrgIn7OrgNotIn(customerCode, orgCodeIn, orgCodeNotIn);
    return !CollectionUtils.isEmpty(entities);
  }

  @Override
  public CustomerVo findOrgByCode(String customerCode) {
    List<CustomerROrgEntity> list = customerROrgService.findByCustomerCodes(Sets.newHashSet(customerCode));
    if (!CollectionUtils.isEmpty(list)){
      return this.nebulaToolkitService.copyObjectByWhiteList(list.get(0), CustomerVo.class, HashSet.class, ArrayList.class);
    }
    return null;
  }

    @Override
    public List<CustomerRelateOrgVo> findAllOrgByCode(String customerCode) {
        List<CustomerRelateOrgVo> voList = new ArrayList<>();
        if (StringUtils.isBlank(customerCode)) {
            return voList;
        }
        List<CustomerROrgEntity> list = customerROrgService.findByCustomerCodes(Sets.newHashSet(customerCode));
        if (CollectionUtils.isEmpty(list)) {
            return voList;
        }
        voList = (List<CustomerRelateOrgVo>) this.nebulaToolkitService.copyCollectionByBlankList(list, CustomerROrgEntity.class, CustomerRelateOrgVo.class, HashSet.class, ArrayList.class);
        //填充名称
        List<String> orgCodes = voList.stream().map(CustomerRelateOrgVo::getOrgCode).collect(Collectors.toList());
        List<OrgVo> orgVos = this.orgVoService.findByOrgCodes(orgCodes);
        Map<String, OrgVo> orgMap = new HashMap<>();
        if (!CollectionUtils.isEmpty(orgVos)) {
            orgMap = orgVos.stream().collect(Collectors.toMap(OrgVo::getOrgCode, Function.identity(), (oldV, newV) -> newV));
        }
        for (CustomerRelateOrgVo relateOrgVo : voList) {
            if (orgMap.containsKey(relateOrgVo.getOrgCode())) {
              OrgVo o = orgMap.get(relateOrgVo.getOrgCode());
                relateOrgVo.setOrgName(o.getOrgName());
                relateOrgVo.setOrgId(o.getId());
            }
        }
        return voList;
    }

  @Override
  public List<CustomerVo> findDetailsByIds(CustomerSearchDto dto) {
   List<CustomerVo> list = Lists.newLinkedList();
   List<String> ids =dto.getIds();
   if (CollectionUtils.isEmpty(ids)){
     return list;
   }
   List<CustomerEntity> customerList = this.customerRepository.lambdaQuery()
           .in(CustomerEntity::getId,ids)
           .eq(CustomerEntity::getTenantCode,TenantUtils.getTenantCode())
           .eq(CustomerEntity::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
           .list();
   if (CollectionUtils.isEmpty(customerList)){
     return list;
   }
    return this.covertEntityToVo(customerList,dto);
  }

  private List<CustomerVo> covertEntityToVo(List<CustomerEntity> customerList, CustomerSearchDto dto) {
    if (CollectionUtils.isEmpty(customerList)){
      return Lists.newLinkedList();
    }
    List<String> dealerIds = customerList.stream().filter(o -> StringUtils.isNotBlank(o.getDealerId())).map(CustomerEntity::getDealerId).collect(Collectors.toList());
    Set<String> customerCodes =customerList.stream().filter(o->StringUtils.isNotBlank(o.getCustomerCode())).map(CustomerEntity::getCustomerCode).collect(Collectors.toSet());
    Set<String> extendSet = dto.getExtendSet();
    List<CustomerVo> result = (List<CustomerVo>) this.nebulaToolkitService.copyCollectionByBlankList(customerList,CustomerEntity.class,CustomerVo.class, HashSet.class, ArrayList.class);
    if (extendSet.contains(CustomerExtendEunm.ORG.getCode())){
      List<CustomerROrgEntity> customerROrgEntities = this.customerROrgService.findByCustomerCodes(customerCodes);
      Map<String, List<CustomerROrgEntity>> map = customerROrgEntities.stream().collect(Collectors.groupingBy(CustomerROrgEntity::getCustomerCode));
      result.forEach(o->{
        List<CustomerROrgEntity> customerROrgEntityList =map.get(o.getCustomerCode());
        if (!CollectionUtils.isEmpty(customerROrgEntityList)) {
          List<CustomerROrgVo> list = (List<CustomerROrgVo>) this.nebulaToolkitService.copyCollectionByWhiteList(customerROrgEntityList, CustomerROrgEntity.class, CustomerROrgVo.class, HashSet.class, ArrayList.class);
          o.setCustomerROrgVos(list);
        }
      });
    }
    //采供关系关联
    if (extendSet.contains(CustomerExtendEunm.SUPPLY.getCode())){
      Map<String, List<SupplyRelationshipVo>> map = this.supplyRelationshipSdkService.findByBuyerIds(dealerIds);
      if (!CollectionUtils.isEmpty(map)) {
        result.stream().forEach(o -> {
          o.setSupplyRelationshipVos(map.get(o.getDealerId()));
        });
      }
    }
    //销售区域关联
    if (extendSet.contains(CustomerExtendEunm.SALEAREA.getCode())) {
      List<DealerSalesRegionEntity> list = this.dealerSalesRegionRepository.findByDealerIds(dealerIds);
      if (!CollectionUtils.isEmpty(list)) {
        Map<String, List<DealerSalesRegionEntity>> map = list.stream().collect(Collectors.groupingBy(DealerSalesRegionEntity::getDealerId));
        result.forEach(o -> {
          List<DealerSalesRegionEntity> saleAreaEntities = map.get(o.getDealerId());
          if (!CollectionUtils.isEmpty(saleAreaEntities)) {
            List<DealerSalesRegionVo> customerSaleAreaVoList = (List<DealerSalesRegionVo>) this.nebulaToolkitService.copyCollectionByWhiteList(saleAreaEntities, DealerSalesRegionEntity.class, DealerSalesRegionVo.class, HashSet.class, ArrayList.class);
            o.setDealerSalesRegionVos(customerSaleAreaVoList);
          }
        });
      }
    }
    return result;
  }


  /**
   * 通过经销商id（EAS）查询客户信息
   *
   * @param dto 经销商id（EAS）
   * @return CustomerVo
   */
  @Override
  public CustomerVo findBySourcesId(CustomerSourceIdDto dto) {
    if (StringUtils.isBlank(dto.getSourcesId())) {
      return null;
    }
    CustomerEntity customer = this.customerRepository.lambdaQuery()
            .eq(CustomerEntity::getSourcesId, dto.getSourcesId())
            .eq(CustomerEntity::getTenantCode, TenantUtils.getTenantCode())
            .eq(CustomerEntity::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
            .one();
    if (Objects.isNull(customer)) {
      return null;
    }
    return this.nebulaToolkitService.copyObjectByBlankList(customer, CustomerVo.class, HashSet.class, ArrayList.class);
  }


  /**
   * 客户实体类转客户信息VO
   *
   * @param entity 客户实体类
   * @return 客户信息VO
   */
  private CustomerVo covertEntityToVo(CustomerEntity entity) {
    List<CustomerVo> list = this.covertEntityToVo(Lists.newArrayList(entity));
    return list.get(0);
  }

  @Autowired(required = false)
  private RegionVoService regionVoService;

  /**
   * 客户实体类转客户信息VO(集合)
   *
   * @param entities 客户实体类列表
   * @return 客户信息VO列表
   */
  private List<CustomerVo> covertEntityToVo(List<CustomerEntity> entities) {
    if (CollectionUtils.isEmpty(entities)) {
      return Lists.newArrayList();
    }
    List<CustomerVo> list = Lists.newArrayList(
        this.nebulaToolkitService.copyCollectionByBlankList(entities, CustomerEntity.class
            , CustomerVo.class, HashSet.class, ArrayList.class));
    List<String> codes = Lists.newLinkedList();
    for (CustomerVo customerVo : list) {
      codes.add(customerVo.getCityCode());
      codes.add(customerVo.getDistrictCode());
      codes.add(customerVo.getProvinceCode());
    }
    Map<String, String> regionCodeNameMap = regionVoService.findRegionCodeNameMap(codes);
    Map<String, List<OrgVo>> orgMap = this.buildOrgMap(entities);
    Map<String, CustomerOrgVo> customerOrgMap = this.buildCustomerOrgMap(entities);

    //经销商组织关系
    List<String> dealerIds = entities.stream().map(o -> o.getDealerId()).filter(StringUtils::isNotBlank).collect(Collectors.toList());
    //Map<String, List<DealerOrgRelationshipVo>> dealerOrgRelationshipMap = this.dealerOrgRelationshipService.findDealerOrgRelationshipMap(dealerIds);
    Map<String, List<CustomerROrgEntity>> dealerOrgRelationshipMap = this.customerROrgService.findDealerOrgRelationshipMap(dealerIds);
    if (Objects.isNull(dealerOrgRelationshipMap)){
      dealerOrgRelationshipMap = new HashMap<>();
    }

    //采供关系
    Map<String, List<SupplyRelationshipVo>> supplyRelationshipListMap = this.supplyRelationshipSdkService.findByBuyerIds(dealerIds);
    if (Objects.isNull(supplyRelationshipListMap)){
      supplyRelationshipListMap = new HashMap<>();
    }

    for (CustomerVo customerVo : list) {
      List<OrgVo> orgList = orgMap.get(customerVo.getCustomerCode());
      if (!CollectionUtils.isEmpty(orgList)) {
        List<String> orgNameList = Lists.newArrayList();
        List<String> orgCodeList = Lists.newArrayList();
        orgList.forEach(a -> {
          orgNameList.add(a.getOrgName());
          orgCodeList.add(a.getOrgCode());
        });
        customerVo.setOrgName(String.join(",", orgNameList));
        customerVo.setOrgCode(String.join(",", orgCodeList));
      }
      CustomerOrgVo customerOrgVo = customerOrgMap.get(customerVo.getCustomerOrgCode());
      if (Objects.nonNull(customerOrgVo)) {
        customerVo.setCustomerOrgName(customerOrgVo.getCustomerOrgName());
      }
      customerVo.setUnionName(
          String.format("%s/%s", ObjectUtils.defaultIfNull(customerVo.getCustomerCode(), "")
              , ObjectUtils.defaultIfNull(customerVo.getCustomerName(), "")));
      customerVo.setCityName(regionCodeNameMap.get(customerVo.getCityCode()));
      customerVo.setDistrictName(regionCodeNameMap.get(customerVo.getDistrictCode()));
      customerVo.setProvinceName(regionCodeNameMap.get(customerVo.getProvinceCode()));

      //销售区域
      customerVo.setDealerSalesRegionVos(this.getSalesRegion(customerVo.getDealerId()));

      //经销商组织关系
     // customerVo.setDealerOrgRelationshipVos(dealerOrgRelationshipMap.get(customerVo.getDealerId()));
      customerVo.setCustomerROrgVos((List<CustomerROrgVo>) this.nebulaToolkitService.copyCollectionByBlankList(dealerOrgRelationshipMap.get(customerVo.getDealerId()),CustomerROrgEntity.class,CustomerROrgVo.class,HashSet.class,ArrayList.class));

      //采供关系
      customerVo.setSupplyRelationshipVos(supplyRelationshipListMap.get(customerVo.getDealerId()));

      //客户信息里的orgList填充组织名称
      if (!CollectionUtils.isEmpty(orgList)){
        Map<String, String> relateOrgMap = orgList.stream().collect(Collectors.toMap(OrgVo::getOrgCode, OrgVo::getOrgName, (oldV, newV) -> newV));
        for (CustomerRelateOrgVo relateOrgVo : customerVo.getOrgList()) {
          relateOrgVo.setOrgName(relateOrgMap.getOrDefault(relateOrgVo.getOrgCode(), null));
        }
      }
    }
    return list;
  }

  private List<DealerSalesRegionVo> getSalesRegion(String dealerId){
    List<DealerSalesRegionEntity> entities = this.dealerSalesRegionRepository.findByDealerId(dealerId);
    Collection<DealerSalesRegionVo> regionVos = this.nebulaToolkitService.copyCollectionByBlankList(entities, DealerSalesRegionEntity.class, DealerSalesRegionVo.class, HashSet.class, ArrayList.class);
    return new ArrayList<>(regionVos);
  }

  /**
   * 封装客户信息关联企业组织映射(key:客户编码,value:客户关联企业组织集合)
   *
   * @param entities 客户信息列表
   * @return 客户信息关联企业组织映射
   */
  private Map<String, List<OrgVo>> buildOrgMap(List<CustomerEntity> entities) {
    Map<String, List<OrgVo>> resultMap = Maps.newHashMap();
    if (CollectionUtils.isEmpty(entities)) {
      return resultMap;
    }
    Set<String> orgCodeList = Sets.newHashSet();
    entities.forEach(customerEntity -> {
      if (!CollectionUtils.isEmpty(customerEntity.getOrgList())) {
        orgCodeList.addAll(customerEntity.getOrgList().stream().map(CustomerROrgEntity::getOrgCode)
            .collect(Collectors.toList()));
      }
    });
    if (CollectionUtils.isEmpty(orgCodeList)) {
      return resultMap;
    }
    List<OrgVo> orgVoList = this.orgVoService.findByOrgCodes(Lists.newArrayList(orgCodeList));
    if (CollectionUtils.isEmpty(orgVoList)) {
      return resultMap;
    }
    Map<String, OrgVo> orgMap = orgVoList.stream()
        .collect(Collectors.toMap(OrgVo::getOrgCode, t -> t, (key1, key2) -> key2));
    //封装客户信息关联企业组织映射(key:客户编码,value:客户关联企业组织)
    entities.forEach(customerEntity -> {
      if (CollectionUtils.isEmpty(customerEntity.getOrgList())) {
        return;
      }
      List<OrgVo> list = resultMap.get(customerEntity.getCustomerCode());
      if (CollectionUtils.isEmpty(list)) {
        list = Lists.newArrayList();
        resultMap.put(customerEntity.getCustomerCode(), list);
      }
      for (CustomerROrgEntity customerROrgEntity : customerEntity.getOrgList()) {
        OrgVo orgVo = orgMap.get(customerROrgEntity.getOrgCode());
        if (Objects.nonNull(orgVo)) {
          list.add(orgVo);
        }
      }
    });
    return resultMap;
  }

  /**
   * 封装客户信息关联客户组织映射(key:客户编码,value:客户关联客户组织)
   *
   * @param entities 客户信息列表
   * @return 客户信息关联客户组织映射
   */
  private Map<String, CustomerOrgVo> buildCustomerOrgMap(List<CustomerEntity> entities) {
    Map<String, CustomerOrgVo> resultMap = Maps.newHashMap();
    if (CollectionUtils.isEmpty(entities)) {
      return resultMap;
    }
    List<String> orgCodeList = entities.stream().map(CustomerEntity::getCustomerOrgCode)
        .collect(Collectors.toList());
    orgCodeList = orgCodeList.stream().filter(StringUtils::isNotBlank).collect(Collectors.toList());
    if (CollectionUtils.isEmpty(orgCodeList)) {
      return resultMap;
    }
    List<CustomerOrgVo> orgVoList = this.customerOrgVoSdkService.findListByCodes(orgCodeList);
    if (CollectionUtils.isEmpty(orgVoList)) {
      return resultMap;

    }
    Map<String, CustomerOrgVo> orgMap = orgVoList.stream()
        .collect(Collectors.toMap(CustomerOrgVo::getCustomerOrgCode, t -> t, (key1, key2) -> key2));
    //封装客户信息关联客户组织映射(key:客户编码,value:客户关联客户组织)
    entities.forEach(customerEntity -> {
      CustomerOrgVo customerOrgVo = orgMap.get(customerEntity.getCustomerOrgCode());
      if (Objects.nonNull(customerOrgVo)) {
        resultMap.put(customerEntity.getCustomerOrgCode(), customerOrgVo);
      }
    });
    return resultMap;
  }
}
