package com.biz.crm.mdm.business.customer.local.service.internal;

import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.customer.local.entity.SupplyRelationshipDetailEntity;
import com.biz.crm.mdm.business.customer.local.entity.SupplyRelationshipEntity;
import com.biz.crm.mdm.business.customer.local.repository.SupplyRelationshipDetailRepository;
import com.biz.crm.mdm.business.customer.local.repository.SupplyRelationshipRepository;
import com.biz.crm.mdm.business.customer.sdk.dto.SupplyRelationshipDetailDto;
import com.biz.crm.mdm.business.customer.sdk.dto.SupplyRelationshipDto;
import com.biz.crm.mdm.business.customer.sdk.service.SupplyRelationshipSdkService;
import com.biz.crm.mdm.business.customer.sdk.vo.SupplyRelationshipDetailVo;
import com.biz.crm.mdm.business.customer.sdk.vo.SupplyRelationshipVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import liquibase.pro.packaged.P;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @ClassName SupplyRelationshipSdkService
 * @Description 采供关系接口
 * @AUTHOR WangJJ
 * @DATE 2023/6/15 14:16
 **/
@Service
@Slf4j
public class SupplyRelationshipSdkServiceImpl implements SupplyRelationshipSdkService {

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private SupplyRelationshipRepository supplyRelationshipRepository;

    @Autowired(required = false)
    private SupplyRelationshipDetailRepository detailRepository;

    /**
     * 保存更新逻辑  1、清空当前采购人的供货关系  2、新增供货关系和供货关系明细
     * @param dtos
     * @param buyerId 采购方ID（兆信）
     */
    @Override
    public void savaOrUpdate(List<SupplyRelationshipDto> dtos, String buyerId) {
        //校验
        this.createValidate(dtos);

        if (StringUtils.isBlank(buyerId)){
            buyerId = dtos.get(0).getBuyerId();
        }

        //先删除
        List<SupplyRelationshipEntity> oldEntities = this.supplyRelationshipRepository.findByBuyerId(buyerId);
        if (CollectionUtils.isNotEmpty(oldEntities)){
            List<String> ids = oldEntities.stream().map(o -> o.getId()).filter(StringUtils::isNotBlank).collect(Collectors.toList());
            this.supplyRelationshipRepository.removeByBuyerId(buyerId);
            this.detailRepository.removeBySupplyRelationshipIds(ids);
        }

        List<SupplyRelationshipEntity> entities = (ArrayList<SupplyRelationshipEntity>)nebulaToolkitService.copyCollectionByWhiteList(dtos, SupplyRelationshipDto.class , SupplyRelationshipEntity.class, HashSet.class, ArrayList.class);
        for (SupplyRelationshipEntity entity : entities) {
            entity.setId(null);
            entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            entity.setTenantCode(TenantUtils.getTenantCode());
            entity.setBuyerId(buyerId);
        }

        //再保存
        this.supplyRelationshipRepository.saveBatch(entities);

        Map<String, SupplyRelationshipDto> dtoMap = dtos.stream().collect(Collectors.toMap(o -> o.getBuyerId() + o.getSupplierId(), Function.identity(), (v1, v2) -> v2));
        Map<String, SupplyRelationshipEntity> entityMap = entities.stream().collect(Collectors.toMap(o -> o.getBuyerId() + o.getSupplierId(), Function.identity(), (v1, v2) -> v2));

        List<SupplyRelationshipDetailEntity> detailEntities = new ArrayList<>();
        for (Map.Entry<String, SupplyRelationshipDto> entry : dtoMap.entrySet()) {
            SupplyRelationshipEntity entity = entityMap.get(entry.getKey());
            for (SupplyRelationshipDetailDto dto : entry.getValue().getDetailDtos()) {
                SupplyRelationshipDetailEntity detailEntity = this.nebulaToolkitService.copyObjectByWhiteList(dto, SupplyRelationshipDetailEntity.class, HashSet.class, ArrayList.class);
                detailEntity.setId(null);
                detailEntity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                detailEntity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                detailEntity.setTenantCode(TenantUtils.getTenantCode());
                detailEntity.setSupplyRelationshipId(entity.getId());
                detailEntities.add(detailEntity);
            }

        }
        this.detailRepository.saveBatch(detailEntities);
    }

    @Override
    public List<SupplyRelationshipVo> findByBuyerId(String buyerId) {
        if (StringUtils.isBlank(buyerId)) {
            return Lists.newArrayList();
        }
        List<SupplyRelationshipEntity> entities = this.supplyRelationshipRepository.findByBuyerId(buyerId);
        if (CollectionUtils.isEmpty(entities)){
            return Lists.newArrayList();
        }
        //获取明细数据
        List<String> ids = entities.stream().map(o -> o.getId()).collect(Collectors.toList());
        List<SupplyRelationshipDetailEntity> detailEntities = this.detailRepository.findBySupplyRelationshipIds(ids);

        List<SupplyRelationshipVo> supplyRelationshipVos = (ArrayList<SupplyRelationshipVo>)this.nebulaToolkitService.copyCollectionByWhiteList(entities, SupplyRelationshipEntity.class, SupplyRelationshipVo.class, HashSet.class, ArrayList.class);

        List<SupplyRelationshipDetailVo> detailVos = (ArrayList<SupplyRelationshipDetailVo>)this.nebulaToolkitService.copyCollectionByWhiteList(detailEntities, SupplyRelationshipDetailEntity.class, SupplyRelationshipDetailVo.class, HashSet.class, ArrayList.class);
        Map<String, List<SupplyRelationshipDetailVo>> detailVoListMap = detailVos.stream().collect(Collectors.groupingBy(SupplyRelationshipDetailVo::getSupplyRelationshipId));

        supplyRelationshipVos.forEach(o -> o.setDetailVos(detailVoListMap.get(o.getId())));

        return supplyRelationshipVos;
    }

    @Override
    public Map<String, List<SupplyRelationshipVo>> findByBuyerIds(List<String> buyerIds) {
        if (CollectionUtils.isEmpty(buyerIds)){
            return Maps.newHashMap();
        }
        List<SupplyRelationshipEntity> entities = this.supplyRelationshipRepository.findByBuyerIds(buyerIds);
        if (CollectionUtils.isEmpty(entities)){
            return Maps.newHashMap();
        }
        List<String> ids = entities.stream().map(o -> o.getId()).collect(Collectors.toList());
        List<SupplyRelationshipDetailEntity> detailEntities = this.detailRepository.findBySupplyRelationshipIds(ids);

        List<SupplyRelationshipVo> supplyRelationshipVos = (ArrayList<SupplyRelationshipVo>)this.nebulaToolkitService.copyCollectionByWhiteList(entities, SupplyRelationshipEntity.class, SupplyRelationshipVo.class, HashSet.class, ArrayList.class);

        List<SupplyRelationshipDetailVo> detailVos = (ArrayList<SupplyRelationshipDetailVo>)this.nebulaToolkitService.copyCollectionByWhiteList(detailEntities, SupplyRelationshipDetailEntity.class, SupplyRelationshipDetailVo.class, HashSet.class, ArrayList.class);
        Map<String, List<SupplyRelationshipDetailVo>> detailVoListMap = detailVos.stream().collect(Collectors.groupingBy(SupplyRelationshipDetailVo::getSupplyRelationshipId));

        supplyRelationshipVos.forEach(o -> o.setDetailVos(detailVoListMap.get(o.getId())));

        Map<String, List<SupplyRelationshipVo>> listMap = supplyRelationshipVos.stream().collect(Collectors.groupingBy(SupplyRelationshipVo::getBuyerId));

        return listMap;
    }

    private void createValidate(List<SupplyRelationshipDto> dtos){

        Validate.notEmpty(dtos,"请求参数不能为空");
        dtos.forEach(dto -> {
            Validate.notBlank(dto.getBuyerId(),"采购单位ID不能为空");
            Validate.notBlank(dto.getSupplierId(),"供货单位ID不能为空");
            Validate.notEmpty(dto.getDetailDtos(),"采供明细不能为空");
            dto.getDetailDtos().forEach(o -> {
                Validate.notBlank(o.getProductCode(),"产品编码不能为空");
//                Validate.notBlank(o.getProductClassifyCode(),"产品分类编码不能为空");
                Validate.notBlank(o.getProductSpe(),"产品规格不能为空");
            });
        });
    }


}
