package com.biz.crm.mdm.business.customer.material.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mdm.business.customer.material.local.entity.CustomerMaterial;
import com.biz.crm.mdm.business.customer.material.local.repository.CustomerMaterialRepository;
import com.biz.crm.mdm.business.customer.material.local.service.CustomerMaterialService;
import com.biz.crm.mdm.business.customer.material.sdk.dto.CustomerMaterialDto;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.Date;
import java.util.List;
import java.util.Optional;

/**
 * 客户物料service实现类
 *
 * @author jerry7
 */
@Service
public class CustomerMaterialServiceImpl implements CustomerMaterialService {

  @Autowired(required = false)
  private CustomerMaterialRepository customerMaterialRepository;

  @Autowired(required = false)
  private LoginUserService loginUserService;

  @Override
  public Page<CustomerMaterial> findByConditions(Pageable pageable, CustomerMaterialDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new CustomerMaterialDto());
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    Page<CustomerMaterial> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return customerMaterialRepository.findByConditions(page, dto);
  }

  @Override
  public CustomerMaterial findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.customerMaterialRepository.getById(id);
  }

  @Override
  @Transactional(rollbackFor = Exception.class)
  public CustomerMaterial create(CustomerMaterial customerMaterial) {
    //校验数据
    this.createValidate(customerMaterial);
    customerMaterial.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    customerMaterial.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    customerMaterial.setCreateTime(new Date());
    customerMaterial.setCreateAccount(loginUserService.getLoginAccountName());
    customerMaterial.setTenantCode(TenantUtils.getTenantCode());
    this.customerMaterialRepository.save(customerMaterial);
    return customerMaterial;
  }

  @Override
  @Transactional(rollbackFor = Exception.class)
  public CustomerMaterial update(CustomerMaterial customerMaterial) {
    Validate.notNull(customerMaterial, "更新客户物料对象不能为空");
    Validate.notBlank(customerMaterial.getId(), "更新时数据主键不能为空！");
    CustomerMaterial entity = this.customerMaterialRepository.getById(customerMaterial.getId());
    Validate.notNull(entity, "待更新的客户物料不存在！");
    this.createValidate(customerMaterial);
    this.customerMaterialRepository.updateById(customerMaterial);
    customerMaterial.setModifyTime(new Date());
    customerMaterial.setModifyAccount(loginUserService.getLoginAccountName());
    return customerMaterial;
  }

  /**
   * 批量删除
   * 此删除方法非物理删除，而是将数据的删除标记置为true
   *
   * @param ids 客户物料主键编码
   */
  @Override
  @Transactional(rollbackFor = Exception.class)
  public void deleteBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "待删除的数据主键不能为空");
    this.customerMaterialRepository.updateDelStatusByIdIn(DelFlagStatusEnum.DELETE, ids);
  }

  /**
   * 批量启用
   *
   * @param ids 客户物料主键编码
   */
  @Override
  @Transactional(rollbackFor = Exception.class)
  public void enableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "待修正的数据主键不能为空");
    this.customerMaterialRepository.updateEnableStatusByIdIn(EnableStatusEnum.ENABLE, ids);
  }

  /**
   * 批量禁用
   *
   * @param ids 客户物料主键编码
   */
  @Override
  @Transactional(rollbackFor = Exception.class)
  public void disableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "待修正的数据主键不能为空");
    this.customerMaterialRepository.updateEnableStatusByIdIn(EnableStatusEnum.DISABLE, ids);
  }

  /**
   * 通过客户组织编码批量查询
   *
   * @param customerOrgCodes
   * @return
   */
  @Override
  public List<CustomerMaterial> findByCustomerOrgCodes(List<String> customerOrgCodes) {
    Validate.isTrue(!CollectionUtils.isEmpty(customerOrgCodes), "组织编码集合不能为空");
    return customerMaterialRepository.findByCustomerOrgCodes(customerOrgCodes,DelFlagStatusEnum.NORMAL.getCode());
  }

  /**
   * 客户物料新增校验
   *
   * @param customerMaterial 客户物料信息
   */
  public void createValidate(CustomerMaterial customerMaterial) {
    Validate.notNull(customerMaterial, "新增客户物料对象不能为空");
    Validate.notBlank(customerMaterial.getCustomerOrgCode(), "客户组织编码不能为空");
    Validate.notBlank(customerMaterial.getMaterialCode(), "物料编码不能为空");
    Validate.notBlank(customerMaterial.getBarCode(), "条形码不能为空");
    Validate.isTrue(customerMaterial.getMaterialCode().length() <= 64, "新增客户物料时物料编码长度不能超过64");
    Validate.isTrue(customerMaterial.getCustomerOrgCode().length() <= 64, "新增客户物料时客户组织编码长度不能超过64");
    Validate.isTrue(customerMaterial.getMaterialCode().length() <= 64, "新增客户物料时条形码长度不能超过64");
  }
}
