package com.biz.crm.mdm.business.customer.org.local.repository;
/**
 * Created by Bao Hongbin on 2021-10-27 18:01.
 */


import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.vo.LazyTreeVo;
import com.biz.crm.mdm.business.customer.org.local.entity.CustomerOrg;
import com.biz.crm.mdm.business.customer.org.local.mapper.CustomerOrgMapper;
import com.biz.crm.mdm.business.customer.org.sdk.dto.CustomerOrgPaginationDto;
import com.biz.crm.mdm.business.customer.org.sdk.vo.CustomerOrgVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.util.ObjectUtils;

import java.util.List;
import java.util.Set;

/**
 * @program: crm
 * @description: 客户组织的数据库访问类
 * @author: Bao Hongbin
 * @create: 2021-10-27 18:01
 **/
@Component
public class CustomerOrgRepository extends ServiceImpl<CustomerOrgMapper, CustomerOrg> {

  /**
   * 根据客户组织名称查询list
   *
   * @param customerOrgName
   * @param isLike
   * @param tenantCode
   * @return
   */
  public List<CustomerOrg> findListByName(String customerOrgName, boolean isLike, String tenantCode) {
    Validate.notBlank(customerOrgName, "客户组织名称不能为空");
    return this.lambdaQuery()
        .eq(CustomerOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(CustomerOrg::getTenantCode, tenantCode)
        .eq(!isLike, CustomerOrg::getCustomerOrgName, customerOrgName)
        .like(isLike, CustomerOrg::getCustomerOrgName, customerOrgName)
        .list();
  }

  /**
   * 根据客户组织编码查询详情
   *
   * @param customerOrgCode
   * @param tenantCode
   * @return
   */
  public CustomerOrg findDetailsByCode(String customerOrgCode, String tenantCode) {
    Validate.notBlank(customerOrgCode, "客户组织编码不能为空");
    return this.lambdaQuery()
        .eq(CustomerOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(CustomerOrg::getTenantCode, tenantCode)
        .eq(CustomerOrg::getCustomerOrgCode, customerOrgCode)
        .one();
  }

  /**
   * 根据客户组织编码查询list
   *
   * @param customerOrgCode
   * @param tenantCode
   * @return
   */
  public List<CustomerOrg> findListByCode(String customerOrgCode, String tenantCode) {
    Validate.notBlank(customerOrgCode, "客户组织编码不能为空");
    return this.lambdaQuery()
        .eq(CustomerOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(CustomerOrg::getTenantCode, tenantCode)
        .eq(CustomerOrg::getCustomerOrgCode, customerOrgCode)
        .list();
  }

  public CustomerOrg findDetailsById(String id, String tenantCode) {
    Validate.notBlank(id, "客户组织id不能为空");
    return this.lambdaQuery()
        .eq(CustomerOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(CustomerOrg::getTenantCode, tenantCode)
        .eq(CustomerOrg::getId, id)
        .one();
  }

  public List<CustomerOrg> findChildrenListByParentCode(String parentCode, String tenantCode) {
    return this.lambdaQuery()
        .eq(CustomerOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(CustomerOrg::getTenantCode, tenantCode)
        .isNull(!StringUtils.isNotEmpty(parentCode), CustomerOrg::getParentCode)
        .eq(StringUtils.isNotEmpty(parentCode), CustomerOrg::getParentCode, parentCode).list();
  }

  /**
   * 通过id集合查询客户组织数据
   *
   * @param ids
   * @param tenantCode
   * @return
   */
  public List<CustomerOrg> findListByIds(List<String> ids, String tenantCode) {
    return this.lambdaQuery()
        .eq(CustomerOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(CustomerOrg::getTenantCode, tenantCode)
        .in(CustomerOrg::getId, ids).list();
  }

  /**
   * 通过RuleCode集合查询客户组织数据
   *
   * @param ruleCodes
   * @param tenantCode
   * @return
   */
  public List<CustomerOrg> findListByRuleCodes(Set<String> ruleCodes, String tenantCode) {
    return this.lambdaQuery()
        .eq(CustomerOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(CustomerOrg::getTenantCode, tenantCode)
        .in(CustomerOrg::getRuleCode, ruleCodes).list();
  }

  /**
   * 通过启用状态和RuleCodes模糊查询自身及子集
   *
   * @param ruleCodes
   * @param enableStatus
   * @param tenantCode
   * @return
   */
  public List<CustomerOrg> findCurAndChildrenByRuleCodeList(List<String> ruleCodes, String enableStatus, String tenantCode) {
    return this.baseMapper.findCurAndChildrenByRuleCodeList(ruleCodes, enableStatus, tenantCode, DelFlagStatusEnum.NORMAL);
  }

  /**
   * 根据父级Codes查询所有子集
   *
   * @param parentCodes
   * @param tenantCode
   * @return
   */
  public List<CustomerOrg> findChildrenListByParentCodes(List<String> parentCodes, String tenantCode) {
    return this.lambdaQuery()
        .eq(CustomerOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(CustomerOrg::getTenantCode, tenantCode)
        .in(CustomerOrg::getParentCode, parentCodes).list();
  }

  /**
   * 查找parentCode不为空但找不到对应上级的数据,设置parentCode为null
   *
   * @return
   */
  public void updateOrphanParentCodeNull(String tenantCode) {
    this.baseMapper.updateOrphanParentCodeNull(tenantCode, DelFlagStatusEnum.NORMAL);
  }

  /**
   * 查询所有没有父级编码的子集
   *
   * @param tenantCode
   * @return
   */
  public List<CustomerOrg> findListWithoutParentCode(String tenantCode) {
    return this.lambdaQuery()
        .eq(CustomerOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(CustomerOrg::getTenantCode, tenantCode)
        .isNull(CustomerOrg::getParentCode)
        .list();
  }

  /**
   * 查询所有客户组织
   *
   * @param tenantCode
   * @return
   */
  public List<CustomerOrg> findAll(String tenantCode) {
    return this.lambdaQuery()
        .eq(CustomerOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(CustomerOrg::getTenantCode, tenantCode)
        .list();
  }

  /**
   * 分页条件查询
   *
   * @param pageable
   * @param customerOrgPaginationDto
   * @return
   */
  public Page<CustomerOrgVo> findByConditions(Pageable pageable, CustomerOrgPaginationDto customerOrgPaginationDto) {
    Page<CustomerOrgVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return this.baseMapper.findByConditions(page, customerOrgPaginationDto, DelFlagStatusEnum.NORMAL);
  }

  /**
   * 查询客户组织懒加载数据
   *
   * @param enableStatus    启用状态
   * @param topOnly         传true只查第一层
   * @param parentCode      只查询该编码下一级
   * @param codeList        只查询这些编码
   * @param ruleCodeList    只查询这些降维编码
   * @param excludeRuleCode 排除这个降维编码的下级
   * @param tenantCode
   * @return
   */
  public List<LazyTreeVo> findLazyTreeList(String enableStatus,
                                           Boolean topOnly,
                                           String parentCode,
                                           List<String> codeList,
                                           List<String> ruleCodeList,
                                           String excludeRuleCode,
                                           String tenantCode) {
    return this.baseMapper.findLazyTreeList(enableStatus, topOnly, parentCode, codeList, ruleCodeList, excludeRuleCode, tenantCode, DelFlagStatusEnum.NORMAL);
  }

  /**
   * 通过名称模糊查询客户组织list
   *
   * @param enableStatus
   * @param name
   * @param tenantCode
   * @return
   */
  public List<CustomerOrg> findListLikeName(String enableStatus, String name, String tenantCode) {
    return this.lambdaQuery()
        .eq(CustomerOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(CustomerOrg::getTenantCode, tenantCode)
        .eq(StringUtils.isNotEmpty(enableStatus), CustomerOrg::getEnableStatus, enableStatus)
        .like(StringUtils.isNotEmpty(name), CustomerOrg::getCustomerOrgName, name)
        .list();
  }

  /**
   * 通过code集合查询客户组织数据
   *
   * @param codes
   * @param tenantCode
   * @return
   */
  public List<CustomerOrg> findListByCodes(List<String> codes, String tenantCode) {
    return this.lambdaQuery()
        .eq(CustomerOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(CustomerOrg::getTenantCode, tenantCode)
        .in(CustomerOrg::getCustomerOrgCode, codes).list();
  }

  /**
   * 通过组织机构集合查询当前组织及其以下组织的信息
   *
   * @param orgCodes
   * @param enableStatus
   * @param tenantCode
   * @return
   */
  public List<CustomerOrg> findCurAndChildrenByOrgCodeList(List<String> orgCodes, String enableStatus, String tenantCode) {
    return this.baseMapper.findCurAndChildrenByOrgCodeList(orgCodes, enableStatus, tenantCode);
  }

  /**
   * 手动设置父级编码为空
   *
   * @param id
   */
  public void setParentCodeNull(String id) {
    this.lambdaUpdate().set(CustomerOrg::getParentCode, null)
        .eq(CustomerOrg::getId, id)
        .eq(CustomerOrg::getTenantCode,TenantUtils.getTenantCode())    //新增租户编号判断条件
        .update();
  }

  /**
   * 手动设置父级编码，规则编码为空
   *
   * @param id
   */
  public void setParentCodeAndRuleCodeNull(String id) {
    this.lambdaUpdate()
        .set(CustomerOrg::getParentCode, null)
        .set(CustomerOrg::getRuleCode, "")
        .eq(CustomerOrg::getId, id)
        .eq(CustomerOrg::getTenantCode,TenantUtils.getTenantCode())    //新增租户编号判断条件
        .update();
  }

  public CustomerOrg findByCustomerOrgCode(String code) {
    return this.lambdaQuery()
        .eq(CustomerOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(CustomerOrg::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
        .eq(CustomerOrg::getCustomerOrgCode,code)
        .eq(CustomerOrg::getTenantCode,TenantUtils.getTenantCode())    //新增租户编号判断条件
        .one();
  }

  public List<CustomerOrg> findByCustomerOrgCodes(List<String> code) {
    return this.lambdaQuery()
        .eq(CustomerOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(CustomerOrg::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
        .eq(CustomerOrg::getTenantCode,TenantUtils.getTenantCode())    //新增租户编号判断条件
        .in(CustomerOrg::getCustomerOrgCode,code)
        .list();
  }

  /**
   * 重构修改方法 通过id和租户编号修改
   * @param customerOrgNew
   * @param tenantCode
   */
  public void updateByIdAndTenantCode(CustomerOrg customerOrgNew, String tenantCode) {
    LambdaUpdateWrapper<CustomerOrg> lambdaUpdateWrapper= Wrappers.lambdaUpdate();
    lambdaUpdateWrapper.eq(CustomerOrg::getTenantCode,tenantCode);
    lambdaUpdateWrapper.in(CustomerOrg::getId,customerOrgNew.getId());
    this.baseMapper.update(customerOrgNew,lambdaUpdateWrapper);
  }

  /**
   * 重构修改方法  通过集合的id和租户编号修改
   * @param customerOrgList
   * @param tenantCode
   */
  public void updateBatchByIdAndTenantCode(List<CustomerOrg> customerOrgList, String tenantCode) {
    customerOrgList.stream().forEach(customerOrg ->{
      LambdaUpdateWrapper<CustomerOrg> lambdaUpdateWrapper= Wrappers.lambdaUpdate();
      lambdaUpdateWrapper.eq(CustomerOrg::getTenantCode,tenantCode);
      lambdaUpdateWrapper.in(CustomerOrg::getId,customerOrg.getId());
      this.baseMapper.update(customerOrg,lambdaUpdateWrapper);
    });
  }
  /**
   * 根据客户组织编码查询ruleCode规则编码
   * @param customerOrgCode
   */
  public String findRuleCodeByCustomerOrgCode(String customerOrgCode) {
    CustomerOrg customerOrg = this.lambdaQuery()
        .eq(CustomerOrg::getTenantCode,TenantUtils.getTenantCode())
        .eq(CustomerOrg::getDelFlag,DelFlagStatusEnum.NORMAL.getCode())
        .eq(CustomerOrg::getCustomerOrgCode,customerOrgCode)
        .select(CustomerOrg::getRuleCode)
        .one();
    if(ObjectUtils.isEmpty(customerOrg)){
      return null;
    }else{
      return customerOrg.getRuleCode();
    }
  }
}
