package com.biz.crm.mdm.business.customer.org.local.service.helper;
/**
 * Created by Bao Hongbin on 2021-10-28 14:11.
 */

import com.biz.crm.mdm.business.customer.org.local.entity.CustomerOrg;
import com.biz.crm.mdm.business.customer.org.local.repository.CustomerOrgRepository;
import com.biz.crm.mdm.business.customer.org.sdk.dto.CustomerOrgCreateDto;
import com.biz.crm.mdm.business.customer.org.sdk.dto.CustomerOrgUpdateDto;
import com.biz.crm.mdm.business.customer.org.sdk.event.CustomerOrgEventListener;
import com.biz.crm.mdm.business.customer.org.sdk.vo.CustomerOrgVo;
import com.bizunited.nebula.common.util.JsonUtils;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * @program: crm
 * @description: 客户组织服务助手
 * @author: Bao Hongbin
 * @create: 2021-10-28 14:11
 **/
@Component
@Slf4j
public class CustomerOrgServiceHelper {
  @Autowired(required = false)
  private CustomerOrgRepository customerOrgRepository;
  @Autowired(required = false)
  @Lazy
  private List<CustomerOrgEventListener> customerOrgEventListeners;

  /**
   * 验证创建请求参数
   *
   * @param customerOrgCreateDto
   */
  public void createValidation(CustomerOrgCreateDto customerOrgCreateDto) {
    Validate.notNull(customerOrgCreateDto, "进行当前操作时，信息对象必须传入!");
    log.info("新增客户组织请求dto：{}", JsonUtils.obj2JsonString(customerOrgCreateDto));
    Validate.notBlank(customerOrgCreateDto.getCustomerOrgName(), "客户组织名称不能为空");
    List<CustomerOrg> customerOrgs =
        customerOrgRepository.findListByName(customerOrgCreateDto.getCustomerOrgName(), false, TenantUtils.getTenantCode());
    Validate.isTrue(CollectionUtils.isEmpty(customerOrgs), "客户组织名称已存在");
    if (StringUtils.isNotEmpty(customerOrgCreateDto.getCustomerOrgCode())) {
      Validate.isTrue(customerOrgCreateDto.getCustomerOrgCode().length() < 64,
          "客户组织编码信息，在进行新增时填入值超过了限定长度(64)，请检查!");
      CustomerOrg customerOrg =
          customerOrgRepository.findDetailsByCode(customerOrgCreateDto.getCustomerOrgCode(), TenantUtils.getTenantCode());
      Validate.isTrue(null == customerOrg,
          "编码[" + customerOrgCreateDto.getCustomerOrgCode() + "]已存在");
    }
    if (StringUtils.isNotEmpty(customerOrgCreateDto.getParentCode())) {
      CustomerOrg parent =
          customerOrgRepository.findDetailsByCode(customerOrgCreateDto.getParentCode(), TenantUtils.getTenantCode());
      Validate.notNull(parent, "上级编码无效");
    }
    Validate.notBlank(customerOrgCreateDto.getCustomerOrgType(), "新增信息时，客户组织类型不能为空！");
    Validate.notNull(customerOrgCreateDto.getEnableStatus(), "新增信息时，客户组织启用状态不能为空！");
    Validate.isTrue(customerOrgCreateDto.getCustomerOrgName().length() < 128,
        "客户组织名称信息，在进行新增时填入值超过了限定长度(128)，请检查!");
  }

  /**
   * 验证更新请求参数
   *
   * @param customerOrgUpdateDto
   */
  public void updateValidation(CustomerOrgUpdateDto customerOrgUpdateDto) {
    Validate.notNull(customerOrgUpdateDto, "进行当前操作时，信息对象必须传入!");
    log.info("编辑客户组织请求dto：{}", JsonUtils.obj2JsonString(customerOrgUpdateDto));
    Validate.notBlank(customerOrgUpdateDto.getId(), "修改信息时，客户组织ID不能为空！");
    CustomerOrg customerOrg =
        customerOrgRepository.findDetailsById(customerOrgUpdateDto.getId(), TenantUtils.getTenantCode());
    Validate.notNull(customerOrg, "无效的业务技术编号信息");
    CustomerOrg detailsByCode =
        customerOrgRepository.findDetailsByCode(customerOrgUpdateDto.getCustomerOrgCode(), TenantUtils.getTenantCode());
    Validate.isTrue(detailsByCode == null ||
        customerOrgUpdateDto.getId().equals(detailsByCode.getId()), "客户组织编码已存在");
    List<String> idsByName =
        Optional.ofNullable(
            customerOrgRepository.findListByName(
                customerOrgUpdateDto.getCustomerOrgName(), false, TenantUtils.getTenantCode()))
            .orElse(new ArrayList<>())
            .stream().map(CustomerOrg::getId).collect(Collectors.toList());

    Validate.isTrue(CollectionUtils.isEmpty(idsByName) ||
        (idsByName.size() == 1 && idsByName.contains(customerOrgUpdateDto.getId())), "客户组织名称已存在");
    Validate.notBlank(customerOrg.getRuleCode(), "当前客户组织ruleCode不能为空");
    Validate.notNull(customerOrgUpdateDto.getCustomerOrgCode(), "修改信息时，客户组织编码不能为空！");
    Validate.isTrue(
        customerOrg.getCustomerOrgCode().equals(customerOrgUpdateDto.getCustomerOrgCode()),
        "客户组织编码不能修改");
    if (StringUtils.isNotEmpty(customerOrgUpdateDto.getParentCode())) {
      CustomerOrg parent =
          customerOrgRepository.findDetailsByCode(customerOrgUpdateDto.getParentCode(), TenantUtils.getTenantCode());
      Validate.notNull(parent, "上级客户组织不存在");
      Validate.isTrue(!parent.getId().equals(customerOrg.getId()), "上级客户组织不能是自己");
      Validate.isTrue(!parent.getRuleCode().startsWith(customerOrg.getRuleCode()),
          "上级客户组织不能是当前层级的下级");
      if (!parent.getCustomerOrgCode().equals(customerOrg.getParentCode())) {
        Validate.notBlank(parent.getRuleCode(), "上级客户组织ruleCode不能为空");
      }
    }
    Validate.notBlank(customerOrgUpdateDto.getCustomerOrgName(), "修改信息时，客户组织名称不能为空！");
    Validate.notNull(customerOrgUpdateDto.getCustomerOrgType(), "修改信息时，客户组织类型不能为空！");
    Validate.isTrue(customerOrgUpdateDto.getCustomerOrgName().length() < 128,
        "客户组织名称信息，在进行修改时填入值超过了限定长度(128)，请检查!");
  }

  /**
   * 推送创建事件
   *
   * @param customerOrgVos
   */
  public void sendCreateEvent(List<CustomerOrgVo> customerOrgVos) {
    if (Objects.nonNull(this.customerOrgEventListeners)) {
      for (CustomerOrgEventListener listener : this.customerOrgEventListeners) {
        listener.onBatchCreate(customerOrgVos);
      }
    }
  }

  /**
   * 推送更新事件
   *
   * @param customerOrgVos
   */
  public void sendUpdateEvent(List<CustomerOrgVo> customerOrgVos) {
    if (Objects.nonNull(this.customerOrgEventListeners)) {
      for (CustomerOrgEventListener listener : this.customerOrgEventListeners) {
        listener.onBatchUpdate(customerOrgVos);
      }
    }
  }

  /**
   * 推送启用事件
   *
   * @param customerOrgVos
   */
  public void sendEnableEvent(List<CustomerOrgVo> customerOrgVos) {
    if (Objects.nonNull(this.customerOrgEventListeners)) {
      for (CustomerOrgEventListener listener : this.customerOrgEventListeners) {
        listener.onBatchEnable(customerOrgVos);
      }
    }
  }


  /**
   * 推送禁用事件
   *
   * @param customerOrgVos
   */
  public void sendDisableEvent(List<CustomerOrgVo> customerOrgVos) {
    if (Objects.nonNull(this.customerOrgEventListeners)) {
      for (CustomerOrgEventListener listener : this.customerOrgEventListeners) {
        listener.onBatchDisable(customerOrgVos);
      }
    }
  }


  /**
   * 推送删除事件
   *
   * @param customerOrgVos
   */
  public void sendDeleteEvent(List<CustomerOrgVo> customerOrgVos) {
    if (Objects.nonNull(this.customerOrgEventListeners)) {
      for (CustomerOrgEventListener listener : this.customerOrgEventListeners) {
        listener.onBatchDelete(customerOrgVos);
      }
    }
  }
}
