package com.biz.crm.mdm.business.customer.retailer.local.repository;

import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.customer.retailer.local.entity.CustomerRetailer;
import com.biz.crm.mdm.business.customer.retailer.local.mapper.CustomerRetailerMapper;
import com.biz.crm.mdm.business.customer.retailer.sdk.dto.CustomerRetailerDto;
import com.biz.crm.mdm.business.customer.retailer.sdk.dto.CustomerRetailerSelectDto;
import com.biz.crm.mdm.business.customer.retailer.sdk.vo.CustomerRetailerVo;
import com.biz.crm.mn.common.base.constant.CommonSelectConstant;
import com.biz.crm.mn.common.base.dto.CommonSelectDto;
import com.biz.crm.mn.common.base.vo.CommonSelectVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.Collections;
import java.util.List;
import java.util.Optional;

/**
 * 客户零售商信息(repository)
 *
 * @describe: 简述
 * @author: huxmld
 * @version: v1.0.0
 * @date: 2022.11.4 22:53
 */
@Component
public class CustomerRetailerRepository extends ServiceImpl<CustomerRetailerMapper, CustomerRetailer> {

    /**
     * 分页
     *
     * @param page
     * @param dto
     * @return
     */
    public Page<CustomerRetailerVo> findByConditions(Page<CustomerRetailerVo> page, CustomerRetailerDto dto) {
        dto.setTenantCode(TenantUtils.getTenantCode());
        return this.baseMapper.findByConditions(page, dto);
    }

    /**
     * 根据id获取详情
     *
     * @param id 员工ID
     * @return
     */
    public CustomerRetailer findById(String id) {
        if (!StringUtils.hasText(id)) {
            return null;
        }
        return this.lambdaQuery()
                .eq(CustomerRetailer::getId, id)
                .eq(CustomerRetailer::getTenantCode, TenantUtils.getTenantCode())
                .eq(CustomerRetailer::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .one();
    }

    /**
     * 根据id集合获取详情集合
     *
     * @param idList ID集合
     * @return
     */
    public List<CustomerRetailer> findByIds(List<String> idList) {
        if (CollectionUtils.isEmpty(idList)) {
            return Collections.emptyList();
        }
        return this.lambdaQuery()
                .in(CustomerRetailer::getId, idList)
                .eq(CustomerRetailer::getTenantCode, TenantUtils.getTenantCode())
                .eq(CustomerRetailer::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
    }

    /**
     * 根据编码获取详情
     *
     * @param code 编码
     * @return
     */
    public CustomerRetailer findByCode(String code) {
        if (!StringUtils.hasText(code)) {
            return null;
        }
        return this.lambdaQuery()
                .eq(CustomerRetailer::getCustomerRetailerCode, code)
                .eq(CustomerRetailer::getTenantCode, TenantUtils.getTenantCode())
                .eq(CustomerRetailer::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .one();
    }

    /**
     * 根据编码集合获取详情集合
     *
     * @param codeList 编码集合
     * @return
     */
    public List<CustomerRetailer> findByCodes(List<String> codeList) {
        if (CollectionUtils.isEmpty(codeList)) {
            return Collections.emptyList();
        }
        return this.lambdaQuery()
                .in(CustomerRetailer::getCustomerRetailerCode, codeList)
                .eq(CustomerRetailer::getTenantCode, TenantUtils.getTenantCode())
                .eq(CustomerRetailer::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
    }

    /**
     * 根据编码查询是否存在
     *
     * @param code 编码
     * @return
     */
    public Integer countByCode(String code) {
        if (!StringUtils.hasText(code)) {
            return 0;
        }
        return this.lambdaQuery()
                .eq(CustomerRetailer::getTenantCode, TenantUtils.getTenantCode())
                .eq(CustomerRetailer::getCustomerRetailerCode, code)
                .count();
    }

    /**
     * 根据主键集合，修改 enable_status
     *
     * @param enable 状态
     * @param ids    ID集合
     */
    public void updateEnableStatusByIdIn(EnableStatusEnum enable, List<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return;
        }
        UpdateWrapper<CustomerRetailer> updateWrapper = new UpdateWrapper<>();
        updateWrapper.set("enable_status", enable.getCode());
        updateWrapper.in("id", ids);
        this.update(updateWrapper);
    }

    public Page<CustomerRetailerVo> findByCustomerSelectDto(Page<CustomerRetailerVo> page, CustomerRetailerSelectDto dto) {
        dto.setTenantCode(TenantUtils.getTenantCode());
        return this.baseMapper.findByCustomerSelectDto(page, dto);

    }

    /**
     * 零售商下拉
     *
     * @param dto 零售商查询参数
     */
    public List<CommonSelectVo> findCustomerRetailerSelectList(CommonSelectDto dto) {
        if (org.apache.commons.lang3.StringUtils.isNotEmpty(dto.getSelectedCode())) {
            List<String> selectCodeList = Optional.ofNullable(dto.getSelectedCodeList()).orElse(Lists.newArrayList());
            selectCodeList.add(dto.getSelectedCode());
            dto.setSelectedCodeList(selectCodeList);
        }
        return baseMapper.findCustomerRetailerSelectList(dto, CommonSelectConstant.COMMON_SELECT_DEFAULT_SIZE);
    }

    /**
     * 获取所有零售商编码
     *
     * @param tenantCode 租户
     * @return List < CommonSelectVo>
     */
    public List<CommonSelectVo> getAllRetailerCode(String tenantCode) {
        return baseMapper.getAllRetailerCode(tenantCode);
    }

    /**
     * 根据编码列表获取零售商名称
     *
     * @param codeList   零售商编码列表
     * @param tenantCode 租户
     * @return List < CustomerRetailerVo>
     */
    public List<CustomerRetailerVo> getRetailerByCodes(List<List<String>> codeList, String tenantCode) {
        return baseMapper.getRetailerByCodes(codeList,tenantCode);
    }

    /**
     * 根据名称查询零售商编码
     *
     * @param retailerName
     * @return
     */
    public List<String> findByName(String retailerName) {
        return baseMapper.findByName(retailerName);
    }

    /**
     * 根据编码或者名称查询
     * @param codeOrName
     * @return {@link List}<{@link CustomerRetailerVo}>
     */
    public List<CustomerRetailerVo> findByCodeOrName(List<String> codeOrName) {
        Assert.notEmpty(codeOrName, "编码名称查询条件不能为空");
        CustomerRetailerDto dto = new CustomerRetailerDto();
        dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        dto.setTenantCode(TenantUtils.getTenantCode());
        dto.setCodeOrNameList(codeOrName);
        return this.baseMapper.findByCustomerRetailerDto(dto);
    }
}
