package com.biz.crm.mdm.business.customer.retailer.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.customer.retailer.local.entity.CustomerRetailer;
import com.biz.crm.mdm.business.customer.retailer.local.entity.CustomerRetailerBusinessUnitRelation;
import com.biz.crm.mdm.business.customer.retailer.local.helper.CustomerRetailerHelper;
import com.biz.crm.mdm.business.customer.retailer.local.repository.CustomerRetailerBusinessUnitRelationRepository;
import com.biz.crm.mdm.business.customer.retailer.local.repository.CustomerRetailerRepository;
import com.biz.crm.mdm.business.customer.retailer.sdk.constant.CustomerRetailerConstant;
import com.biz.crm.mdm.business.customer.retailer.sdk.dto.CustomerRetailerDto;
import com.biz.crm.mdm.business.customer.retailer.sdk.dto.CustomerRetailerEventDto;
import com.biz.crm.mdm.business.customer.retailer.sdk.dto.CustomerRetailerSelectDto;
import com.biz.crm.mdm.business.customer.retailer.sdk.event.CustomerRetailerEventListener;
import com.biz.crm.mdm.business.customer.retailer.sdk.service.CustomerRetailerVoService;
import com.biz.crm.mdm.business.customer.retailer.sdk.vo.CustomerRetailerVo;
import com.biz.crm.mn.common.base.dto.CommonSelectDto;
import com.biz.crm.mn.common.base.vo.CommonSelectVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 客户零售商信息(CustomerRetailer)表服务实现类
 *
 * @author: huxmld
 * @version: v1.0.0
 * @date: 2022.11.4 23:13
 */
@Slf4j
@Service
public class CustomerRetailerVoServiceImpl implements CustomerRetailerVoService {

    @Autowired(required = false)
    private CustomerRetailerRepository customerRetailerRepository;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private RedisMutexService redisMutexService;
    @Autowired(required = false)
    private NebulaNetEventClient nebulaNetEventClient;

    @Autowired(required = false)
    private CustomerRetailerBusinessUnitRelationRepository customerRetailerBusinessUnitRelationRepository;

    @Autowired(required = false)
    private CustomerRetailerHelper customerRetailerHelper;

    /**
     * 查询分页列表
     *
     * @param pageable 分页参数
     * @param dto      查询参数
     * @return com.baomidou.mybatisplus.extension.plugins.pagination.Page<com.biz.crm.mdm.business.customer.retailer.sdk.vo.CustomerRetailerVo>
     * @describe: 简述
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.4 23:52
     */
    @Override
    public Page<CustomerRetailerVo> findByConditions(Pageable pageable, CustomerRetailerDto dto) {
        pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
        dto = Optional.ofNullable(dto).orElse(new CustomerRetailerDto());
        dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        Page<CustomerRetailerVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.customerRetailerRepository.findByConditions(page, dto);
    }


    /**
     * 零售商下拉框查询接口
     *
     * @param pageable 分页参数
     * @param dto      查询参数
     * @return 零售商列表
     */
    @Override
    public Page<CustomerRetailerVo> findByCustomerSelectDto(Pageable pageable, CustomerRetailerSelectDto dto) {
        pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
        dto = Optional.ofNullable(dto).orElse(new CustomerRetailerSelectDto());
        dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        Page<CustomerRetailerVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        page = this.customerRetailerRepository.findByCustomerSelectDto(page, dto);
        return page;
    }

    /**
     * 按id查询详情
     *
     * @param id 员工ID
     * @return java.util.List<com.biz.crm.mdm.business.customer.retailer.local.entity.CustomerRetailer>
     * @describe: 简述
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.4 23:09
     */
    @Override
    public CustomerRetailerVo findById(String id) {
        if (StringUtils.isEmpty(id)) {
            return null;
        }
        CustomerRetailer promoters = this.customerRetailerRepository.findById(id);
        if (Objects.isNull(promoters)) {
            return null;
        }
        return this.nebulaToolkitService.copyObjectByBlankList(
                promoters, CustomerRetailerVo.class,
                HashSet.class, ArrayList.class);
    }

    /**
     * 按id集合查询详情
     *
     * @param idList 员工ID集合
     * @return java.util.List<com.biz.crm.mdm.business.customer.retailer.local.entity.CustomerRetailer>
     * @describe: 简述
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.4 23:09
     */
    @Override
    public List<CustomerRetailerVo> findByIds(List<String> idList) {
        if (CollectionUtils.isEmpty(idList)) {
            return Collections.emptyList();
        }
        List<CustomerRetailer> promotersList = this.customerRetailerRepository.findByIds(idList);
        if (CollectionUtils.isEmpty(promotersList)) {
            return Collections.emptyList();
        }

        return (List<CustomerRetailerVo>) this.nebulaToolkitService.copyCollectionByBlankList(
                promotersList, CustomerRetailer.class, CustomerRetailerVo.class,
                HashSet.class, ArrayList.class);
    }

    /**
     * 按编码查询详情
     *
     * @param code 编码
     * @return java.util.List<com.biz.crm.mdm.business.customer.retailer.local.entity.CustomerRetailer>
     * @describe: 简述
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.4 23:09
     */
    @Override
    public CustomerRetailerVo findByCode(String code) {
        if (StringUtils.isEmpty(code)) {
            return null;
        }
        CustomerRetailer entity = this.customerRetailerRepository.findByCode(code);
        if (Objects.isNull(entity)) {
            return null;
        }
        return this.nebulaToolkitService.copyObjectByBlankList(
                entity, CustomerRetailerVo.class,
                HashSet.class, ArrayList.class);
    }


    /**
     * 按编码集合查询详情
     *
     * @param codeList 编码集合
     * @return java.util.List<com.biz.crm.mdm.business.customer.retailer.local.entity.CustomerRetailer>
     * @describe: 简述
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.4 23:09
     */
    @Override
    public List<CustomerRetailerVo> findByCodes(List<String> codeList) {
        if (CollectionUtils.isEmpty(codeList)) {
            return Collections.emptyList();
        }
        List<CustomerRetailer> entityList = this.customerRetailerRepository.findByCodes(codeList);
        if (CollectionUtils.isEmpty(entityList)) {
            return Collections.emptyList();
        }
        return (List<CustomerRetailerVo>) this.nebulaToolkitService.copyCollectionByBlankList(
                entityList, CustomerRetailer.class, CustomerRetailerVo.class,
                HashSet.class, ArrayList.class);
    }

    /**
     * 批量新增或修改
     *
     * @param dtoList 数据集合
     * @return java.util.List<com.biz.crm.mdm.business.customer.retailer.sdk.vo.CustomerRetailerVo>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.5 16:42
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public List<CustomerRetailerVo> createOrUpdate(List<CustomerRetailerDto> dtoList) {
        if (CollectionUtils.isEmpty(dtoList)) {
            return Collections.emptyList();
        }
        //基础验证和构建
        Set<String> codeSet = this.baseValAndBuild(dtoList);
        String redisKey = CustomerRetailerConstant.MDM_LOCK_CUSTOMER_RETAILER_CODE;
        Set<String> hasLockSet = new HashSet<>(codeSet.size());
        try {
            codeSet.forEach(code -> {
                boolean lock = redisMutexService.tryLock(redisKey + code, TimeUnit.MINUTES, 30);
                Assert.isTrue(lock, "客户零售商[" + code + "]其他人员正在操作,请稍后再试!");
                hasLockSet.add(code);
            });
            List<CustomerRetailer> oldEntityList = this.customerRetailerRepository.findByCodes(new ArrayList<>(codeSet));
            Map<String, CustomerRetailer> oldMap = new HashMap<>(codeSet.size());
            if (CollectionUtils.isNotEmpty(oldEntityList)) {
                oldEntityList.forEach(entity -> {
                    oldMap.put(entity.getCustomerRetailerCode(), entity);
                });
            }
            List<CustomerRetailer> entityList = (List<CustomerRetailer>) this.nebulaToolkitService.copyCollectionByBlankList(
                    dtoList, CustomerRetailerDto.class, CustomerRetailer.class,
                    HashSet.class, ArrayList.class);
            //新增集合
            List<CustomerRetailer> addList = this.buildAddMap(entityList, oldMap);
            //更新集合
            List<CustomerRetailer> updateList = this.buildUpdateMap(entityList, oldMap);
            if (CollectionUtils.isNotEmpty(addList)) {
                this.customerRetailerRepository.saveBatch(addList);
            }
            if (CollectionUtils.isNotEmpty(updateList)) {
                this.customerRetailerRepository.updateBatchById(updateList);
            }
            //保存日志
            this.saveLog(addList, updateList, oldMap);
        } catch (Exception e) {
            log.error("", e);
            throw e;
        } finally {
            if (CollectionUtils.isNotEmpty(hasLockSet)) {
                hasLockSet.forEach(code -> {
                    redisMutexService.unlock(redisKey + code);
                });
            }
        }
        List<CustomerRetailer> resultList = this.customerRetailerRepository.findByCodes(new ArrayList<>(codeSet));
        if (CollectionUtils.isEmpty(resultList)) {
            return Collections.emptyList();
        }
        return (List<CustomerRetailerVo>) this.nebulaToolkitService.copyCollectionByBlankList(
                resultList, CustomerRetailer.class, CustomerRetailerVo.class,
                HashSet.class, ArrayList.class);
    }

    /**
     * 基础构建和验证
     *
     * @param dtoList
     * @return
     */
    private Set<String> baseValAndBuild(List<CustomerRetailerDto> dtoList) {
        Set<String> codeSet = new HashSet<>(dtoList.size());
        dtoList.forEach(dto -> {
            this.commonValidate(dto);
            String code = dto.getCustomerRetailerCode();
            Assert.isTrue(!codeSet.contains(code), "当前集合内编码[" + code + "]重复");
            if (StringUtils.isEmpty(dto.getDelFlag())) {
                dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            }
            if (StringUtils.isEmpty(dto.getEnableStatus())) {
                dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            }
            dto.setTenantCode(TenantUtils.getTenantCode());
            codeSet.add(code);
        });
        return codeSet;
    }

    /**
     * 保存日志
     *
     * @param addList    新增集合
     * @param updateList 更新集合
     * @param oldMap     更新前数据Map
     */
    private void saveLog(List<CustomerRetailer> addList,
                         List<CustomerRetailer> updateList,
                         Map<String, CustomerRetailer> oldMap) {
        if (CollectionUtils.isNotEmpty(addList)) {
            List<CustomerRetailerDto> list = (List<CustomerRetailerDto>) this.nebulaToolkitService.copyCollectionByBlankList(
                    addList, CustomerRetailer.class, CustomerRetailerDto.class,
                    HashSet.class, ArrayList.class);
            list.forEach(dto -> {
                CustomerRetailerEventDto eventDto = new CustomerRetailerEventDto();
                eventDto.setNewest(dto);
                SerializableBiConsumer<CustomerRetailerEventListener, CustomerRetailerEventDto> consumer = CustomerRetailerEventListener::onCreate;
                this.nebulaNetEventClient.publish(eventDto, CustomerRetailerEventListener.class, consumer);

            });
        }
        if (CollectionUtils.isNotEmpty(updateList)) {
            List<CustomerRetailerDto> list = (List<CustomerRetailerDto>) this.nebulaToolkitService.copyCollectionByBlankList(
                    updateList, CustomerRetailer.class, CustomerRetailerDto.class,
                    HashSet.class, ArrayList.class);
            list.forEach(dto -> {
                CustomerRetailerEventDto eventDto = new CustomerRetailerEventDto();
                eventDto.setOriginal(this.nebulaToolkitService.copyObjectByBlankList(
                        oldMap.getOrDefault(dto.getCustomerRetailerCode(), new CustomerRetailer()), CustomerRetailerDto.class,
                        HashSet.class, ArrayList.class));
                eventDto.setNewest(dto);
                SerializableBiConsumer<CustomerRetailerEventListener, CustomerRetailerEventDto> consumer = CustomerRetailerEventListener::onUpdate;
                this.nebulaNetEventClient.publish(eventDto, CustomerRetailerEventListener.class, consumer);
            });
        }
    }

    /**
     * 构建新增集合
     *
     * @param entityList 入参
     * @param oldMap     数据库数据
     * @return
     */
    private List<CustomerRetailer> buildAddMap(List<CustomerRetailer> entityList, Map<String, CustomerRetailer> oldMap) {
        if (CollectionUtils.isEmpty(entityList)) {
            return Collections.emptyList();
        }
        List<CustomerRetailer> result = new ArrayList<>(entityList.size());
        entityList.stream()
                .filter(k -> !oldMap.containsKey(k.getCustomerRetailerCode()))
                .filter(k -> Objects.isNull(oldMap.get(k.getCustomerRetailerCode())))
                .forEach(result::add);
        return result;
    }


    /**
     * 构建更新集合
     *
     * @param entityList 入参
     * @param oldMap     数据库数据
     * @return
     */
    private List<CustomerRetailer> buildUpdateMap(List<CustomerRetailer> entityList, Map<String, CustomerRetailer> oldMap) {
        if (CollectionUtils.isEmpty(entityList)) {
            return Collections.emptyList();
        }
        List<CustomerRetailer> result = new ArrayList<>(entityList.size());
        entityList.stream()
                .filter(k -> oldMap.containsKey(k.getCustomerRetailerCode()))
                .filter(k -> Objects.nonNull(oldMap.get(k.getCustomerRetailerCode())))
                .forEach(entity -> {
                    CustomerRetailer oldEntity = oldMap.get(entity.getCustomerRetailerCode());
                    entity.setId(oldEntity.getId());
                    entity.setCreateTime(oldEntity.getCreateTime());
                    entity.setCreateName(oldEntity.getCreateName());
                    entity.setCreateAccount(oldEntity.getCreateAccount());
                    result.add(entity);
                });
        return result;
    }


    /**
     * 新增
     *
     * @param dto
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public CustomerRetailerVo create(CustomerRetailerDto dto) {
        this.createValidate(dto);
        String code = dto.getCustomerRetailerCode();
        String redisKey = CustomerRetailerConstant.MDM_LOCK_CUSTOMER_RETAILER_CODE + code;
        boolean lock = redisMutexService.tryLock(redisKey, TimeUnit.MINUTES, 3);
        Assert.isTrue(lock, "客户零售商[" + code + "]其他人员正在操作,请稍后再试!");
        try {
            int count = this.customerRetailerRepository.countByCode(code);
            Assert.isTrue(count < 1, "客户零售商[" + code + "]已存在,不可新增!");
            CustomerRetailer entity = this.nebulaToolkitService.copyObjectByBlankList(
                    dto, CustomerRetailer.class,
                    HashSet.class, ArrayList.class);
            dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            if(!org.springframework.util.CollectionUtils.isEmpty(dto.getBusinessUnitCodes())) {
                entity.setBusinessUnitCode(String.join(",", dto.getBusinessUnitCodes()));
            }
            this.customerRetailerRepository.saveOrUpdate(entity);
            //构建零售商业务单元关系
            List<CustomerRetailerBusinessUnitRelation> relations = this.customerRetailerHelper.buildRelations(dto);
            //保存关系数据
            this.customerRetailerBusinessUnitRelationRepository.saveRelations(dto.getCustomerRetailerCode(), relations);
            CustomerRetailerEventDto eventDto = new CustomerRetailerEventDto();
            eventDto.setOriginal(null);
            eventDto.setNewest(this.nebulaToolkitService.copyObjectByBlankList(
                    entity, CustomerRetailerDto.class,
                    HashSet.class, ArrayList.class));
            SerializableBiConsumer<CustomerRetailerEventListener, CustomerRetailerEventDto> consumer = CustomerRetailerEventListener::onCreate;
            this.nebulaNetEventClient.publish(eventDto, CustomerRetailerEventListener.class, consumer);

            //更新数据字典信息
            return this.nebulaToolkitService.copyObjectByBlankList(
                    entity, CustomerRetailerVo.class,
                    HashSet.class, ArrayList.class);
        } catch (Exception e) {
            log.error("", e);
            throw e;
        } finally {
            redisMutexService.unlock(redisKey);
        }

    }

    /**
     * 修改
     *
     * @param dto
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public CustomerRetailerVo update(CustomerRetailerDto dto) {
        this.updateValidate(dto);
        String code = dto.getCustomerRetailerCode();
        String redisKey = CustomerRetailerConstant.MDM_LOCK_CUSTOMER_RETAILER_CODE + code;
        boolean lock = redisMutexService.tryLock(redisKey, TimeUnit.MINUTES, 3);
        Assert.isTrue(lock, "客户零售商[" + code + "]其他人员正在操作,请稍后再试!");
        try {
            CustomerRetailerEventDto eventDto = new CustomerRetailerEventDto();
            eventDto.setOriginal(dto);
            CustomerRetailer oldEntity = this.customerRetailerRepository.findByCode(code);
            Assert.notNull(oldEntity, "客户零售商[" + code + "]不存在,编辑失败!");
            dto.setId(oldEntity.getId());
            CustomerRetailer entity = this.nebulaToolkitService.copyObjectByBlankList(
                    dto, CustomerRetailer.class,
                    HashSet.class, ArrayList.class);
            if(!org.springframework.util.CollectionUtils.isEmpty(dto.getBusinessUnitCodes())) {
                entity.setBusinessUnitCode(String.join(",", dto.getBusinessUnitCodes()));
            }
            this.customerRetailerRepository.saveOrUpdate(entity);
            //构建零售商业务单元关系
            List<CustomerRetailerBusinessUnitRelation> relations = this.customerRetailerHelper.buildRelations(dto);
            //保存关系数据
            this.customerRetailerBusinessUnitRelationRepository.saveRelations(dto.getCustomerRetailerCode(), relations);
            eventDto.setNewest(this.nebulaToolkitService.copyObjectByBlankList(
                    entity, CustomerRetailerDto.class,
                    HashSet.class, ArrayList.class));
            SerializableBiConsumer<CustomerRetailerEventListener, CustomerRetailerEventDto> consumer = CustomerRetailerEventListener::onUpdate;
            this.nebulaNetEventClient.publish(eventDto, CustomerRetailerEventListener.class, consumer);


            //更新数据字典信息
            return this.nebulaToolkitService.copyObjectByBlankList(
                    entity, CustomerRetailerVo.class,
                    HashSet.class, ArrayList.class);
        } catch (Exception e) {
            log.error("", e);
            throw e;
        } finally {
            redisMutexService.unlock(redisKey);
        }
    }

    /**
     * 删除
     *
     * @param idList
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void delete(List<String> idList) {
        Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
        List<CustomerRetailer> entityList = this.customerRetailerRepository.findByIds(idList);
        entityList.forEach(entity -> {
            String code = entity.getCustomerRetailerCode();
            entity.setCustomerRetailerCode(code + entity.getId());
            entity.setDelFlag(DelFlagStatusEnum.DELETE.getCode());
        });
        this.customerRetailerRepository.updateBatchById(entityList);
        List<CustomerRetailerDto> list = (List<CustomerRetailerDto>) this.nebulaToolkitService.copyCollectionByBlankList(
                entityList, CustomerRetailer.class, CustomerRetailerDto.class,
                HashSet.class, ArrayList.class);
        list.forEach(dto -> {
            CustomerRetailerEventDto eventDto = new CustomerRetailerEventDto();
            eventDto.setOriginal(dto);
            eventDto.setNewest(null);
            SerializableBiConsumer<CustomerRetailerEventListener, CustomerRetailerEventDto> consumer = CustomerRetailerEventListener::onDelete;
            this.nebulaNetEventClient.publish(eventDto, CustomerRetailerEventListener.class, consumer);

        });
    }

    /**
     * 新建校验
     *
     * @param dto
     */
    private void createValidate(CustomerRetailerDto dto) {
        this.commonValidate(dto);
        dto.setId(null);

    }

    /**
     * 修改校验
     *
     * @param dto
     */
    private void updateValidate(CustomerRetailerDto dto) {
        this.commonValidate(dto);
        Validate.notBlank(dto.getId(), "更新时数据主键不能为空！");
        CustomerRetailer businessUnit = this.customerRetailerRepository.findById(dto.getId());
        Validate.notNull(businessUnit, "更新数据不存在！");
        Validate.isTrue(businessUnit.getCustomerRetailerCode().equals(dto.getCustomerRetailerCode()),
                "编码不可修改！");
    }

    /**
     * 基础共同校验
     *
     * @param dto
     */
    private void commonValidate(CustomerRetailerDto dto) {
        Validate.notBlank(dto.getCustomerRetailerCode(), "编码不能为空！");
        Validate.notBlank(dto.getCustomerRetailerName(), "名称不能为空！");
    }

    /**
     * 批量启用
     *
     * @param ids ID集合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(List<String> ids) {
        Validate.notEmpty(ids, "待启用的数据主键不能为空");
        List<CustomerRetailer> entityList = this.customerRetailerRepository.findByIds(ids);
        Validate.notEmpty(entityList, "待启用的数据不存在或已删除!");
        this.customerRetailerRepository.updateEnableStatusByIdIn(EnableStatusEnum.ENABLE, ids);
        List<CustomerRetailerDto> list = (List<CustomerRetailerDto>) this.nebulaToolkitService.copyCollectionByBlankList(
                entityList, CustomerRetailer.class, CustomerRetailerDto.class,
                HashSet.class, ArrayList.class);
        list.forEach(dto -> {
            CustomerRetailerEventDto eventDto = new CustomerRetailerEventDto();
            eventDto.setOriginal(dto);
            eventDto.setNewest(null);
            SerializableBiConsumer<CustomerRetailerEventListener, CustomerRetailerEventDto> consumer = CustomerRetailerEventListener::onEnable;
            this.nebulaNetEventClient.publish(eventDto, CustomerRetailerEventListener.class, consumer);
        });
    }

    /**
     * 批量禁用
     *
     * @param ids ID集合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(List<String> ids) {
        Validate.notEmpty(ids, "待禁用的数据主键不能为空");
        List<CustomerRetailer> entityList = this.customerRetailerRepository.findByIds(ids);
        Validate.notEmpty(entityList, "待禁用的数据不存在或已删除!");
        this.customerRetailerRepository.updateEnableStatusByIdIn(EnableStatusEnum.DISABLE, ids);
        List<CustomerRetailerDto> list = (List<CustomerRetailerDto>) this.nebulaToolkitService.copyCollectionByBlankList(
                entityList, CustomerRetailer.class, CustomerRetailerDto.class,
                HashSet.class, ArrayList.class);
        list.forEach(entity -> {
            CustomerRetailerEventDto eventDto = new CustomerRetailerEventDto();
            eventDto.setOriginal(entity);
            eventDto.setNewest(null);
            SerializableBiConsumer<CustomerRetailerEventListener, CustomerRetailerEventDto> consumer = CustomerRetailerEventListener::onDisable;
            this.nebulaNetEventClient.publish(eventDto, CustomerRetailerEventListener.class, consumer);
        });
    }

    /**
     * 零售商下拉
     *
     * @param dto 零售商查询参数
     */
    @Override
    public List<CommonSelectVo> findCustomerRetailerSelectList(CommonSelectDto dto) {
        return customerRetailerRepository.findCustomerRetailerSelectList(dto);
    }

    /**
     * 获取所有零售商编码
     *
     * @return List < String>
     */
    @Override
    public Map<String, String> getAllRetailerCode() {

        List<CommonSelectVo> list = customerRetailerRepository.getAllRetailerCode(TenantUtils.getTenantCode());
        Map<String, String> map = new HashMap<>();
        if (!CollectionUtils.isEmpty(list)) {
            map = list.stream().collect(Collectors.toMap(CommonSelectVo::getCode, CommonSelectVo::getValue));
        }
        return map;
    }

    @Override
    public Map<String, String> getAllRetailerNameCode() {
        List<CommonSelectVo> list = customerRetailerRepository.getAllRetailerCode(TenantUtils.getTenantCode());
        Map<String, String> map = new HashMap<>();
        if (!CollectionUtils.isEmpty(list)) {
            map = list.stream().collect(Collectors.toMap(CommonSelectVo::getValue, CommonSelectVo::getCode));
        }
        return map;
    }

    /**
     * 根据编码列表获取零售商名称
     *
     * @param codes 零售商编码列表
     * @return List < String>
     */
    @Override
    public Map<String, String> getRetailerByCodes(List<String> codes) {
        if (CollectionUtils.isEmpty(codes)) {
            return Maps.newHashMap();
        }
        List<List<String>> list = Lists.partition(codes, 500);
        List<CustomerRetailerVo> vos = this.customerRetailerRepository.getRetailerByCodes(list,TenantUtils.getTenantCode());
        if (CollectionUtils.isEmpty(vos)) {
            return Maps.newHashMap();
        }
        return vos.stream().collect(Collectors.toMap(CustomerRetailerVo::getCustomerRetailerCode, CustomerRetailerVo::getCustomerRetailerName,
                (oldValue, newValue) -> newValue));
    }

    /**
     * 根据名称查询零售商编码
     *
     * @param retailerName
     * @return
     */
    @Override
    public List<String> findByName(String retailerName) {
        if (StringUtils.isEmpty(retailerName)) {
            return null;
        }
        return this.customerRetailerRepository.findByName(retailerName);
    }

    /**
     * 根据编码或者名称查询
     * @param newArrayList
     * @return {@link List}<{@link CustomerRetailerVo}>
     */
    @Override
    public List<CustomerRetailerVo> findByCodeOrName(List<String> codeOrName) {
        return this.customerRetailerRepository.findByCodeOrName(codeOrName);
    }
}
