package com.biz.crm.mdm.business.customer.sdk.service;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.mdm.business.customer.sdk.dto.*;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerRelateOrgVo;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.springframework.data.domain.Pageable;

import java.util.*;

/**
 * 客户信息VO服务接口类
 *
 * @author ning.zhang
 * @date 2021/10/27
 */
public interface CustomerVoService {

  /**
   * 获取客户信息详情(包含关联数据)
   *
   * @param id           客户ID
   * @param customerCode 客户编码
   * @return 客户信息详情
   */
  CustomerVo findDetailsByIdOrCode(String id, String customerCode);

  /**
   * 客户信息下拉框分页列表
   *
   * @param dto      请求参数dto
   * @param pageable 分页信息
   * @return Page<CustomerVo> 客户信息下拉框分页列表
   */
  Page<CustomerVo> findByCustomerSelectDto(Pageable pageable, CustomerSelectDto dto);

  /**
   * 分页查询
   *
   * @param pageable 分页信息
   * @param dto  分页参数dto
   * @return 查询结果
   */
  default Page<CustomerVo> findByConditions(Pageable pageable, CustomerDto dto){
    return new Page<>();
  }

  /**
   * 批量设置是否分利
   *
   * @param customerCodeList 编码集合
   * @param bool             设置值
   */
  default void modifyShareBenefits(List<String> customerCodeList, Boolean bool){}

  /**
   * 根据企业组织编码获取匹配的客户信息
   *
   * @param orgCodeList 组织编码集合
   * @return 客户信息集合
   */
  List<CustomerVo> findByOrgCodes(List<String> orgCodeList);

  /**
   * 根据客户编码集合获取对应的客户信息
   *
   * @param customerCodeList 客户编码集合
   * @return 客户信息集合
   */
  List<CustomerVo> findByCustomerCodes(List<String> customerCodeList);

  /**
   * 根据客户编码集合获取对应的客户信息及联系人
   *
   * @param customerCodeList 客户编码集合
   * @return 客户信息集合
   */
  List<CustomerVo> findCustomerAndContactByCustomerCodes(List<String> customerCodeList);

  /**
   * 根据渠道编码集合获取对应的客户信息
   *
   * @param channelList 渠道集合
   * @return 客户信息集合
   */
  List<CustomerVo> findByChannels(List<String> channelList);

  /**
   * 根据客户类型集合获取对应的客户信息
   *
   * @param typeList 客户类型集合
   * @return 客户信息集合
   */
  List<CustomerVo> findByTypes(List<String> typeList);

  /**
   * 根据组织集合、渠道集合、标签集合查询客户编码集合
   * 参数非必填项所以不需要进行参数校验
   *
   * @param orgCodes 组织集合
   * @param channels 渠道集合
   * @param tags     标签集合
   * @return 客户编码集合
   */
  default Set<String> findCustomerCodesByOrgCodesAndChannelsAndTags(List<String> orgCodes, List<String> channels, List<String> tags) {
    return new HashSet<>();
  }

  /**
   * 根据客户编码集合获取对应的客户信息-包含主信息+组织信息
   *
   * @param customerCodeSet 客户编码集合
   * @return 客户信息集合
   */
  default List<CustomerVo> findForPriceByCustomerCodes(Set<String> customerCodeSet) {
    return Lists.newLinkedList();
  }

  /**
   * 根据高德id集合查询数据集合
   *
   * @param amapIds
   * @return
   */
  default List<CustomerVo> findByAmapIds(Set<String> amapIds) {
    return Lists.newArrayList();
  }

  /**
   * 根据CustomerQueryDto获取匹配的客户编码集合
   *
   * @param dto
   * @return
   */
  default Set<String> findByCustomerQueryDto(CustomerQueryDto dto) {
    return Sets.newHashSet();
  }

  /**
   * 根据组织获取组织及下级所有的审核通过且未删除的经销商信息
   *
   * @param orgCodes
   * @return k-经销商编码，v-关联组织对应的降维码集合
   */
  default Map<String, Set<String>> findAllowSaleCustomerByOrgCodes(Set<String> orgCodes) {
    return Maps.newHashMap();
  }

  /**
   * 根据创建人账号，年月日期，业务系统类型查询数据
   *
   * @param createAccount
   * @param fromType
   * @return
   */
  default List<CustomerVo> findCountByCreateAccountAndFromTypeAndCreateTimeScope(String createAccount, String fromType, String startDate, String endDate) {
    return new ArrayList<>(0);
  }


  /**
   * 找到客户代码和渠道
   *
   * @param channelList  频道列表
   * @param customerCode 客户代码
   * @return {@link String}
   */
  Boolean existByCustomerCodeAndChannels(List<String> channelList, String customerCode);

  /**
   * 存在客户代码
   *
   * @param customerCode 客户代码
   * @return {@link String}
   */
  Boolean existByCustomerCode(String customerCode);

  /**
   * 判断客户户是否在组织内
   *
   * @param customerCode 客户代码
   * @param orgCodeIn    组织代码
   * @param orgCodeNotIn org代码不
   * @return {@link Boolean}
   */
  Boolean existByCustomer7OrgIn7OrgNotIn(String customerCode, List<String> orgCodeIn, List<String> orgCodeNotIn);

  /**
   * 通过客户编码查询客户组织
   *
   * @param customerCode 客户编码
   * @return 客户信息详情
   */
  CustomerVo findOrgByCode(String customerCode);

  /**
   * 通过客户编码查询客户关联的所有组织
   *
   * @param customerCode 客户编码
   * @return 客户关联组织列表
   */
  List<CustomerRelateOrgVo> findAllOrgByCode(String customerCode);

  List<CustomerVo> findDetailsByIds(CustomerSearchDto dto);

  /**
   * 通过经销商id（EAS）查询客户信息
   *
   * @param dto 经销商id（EAS）
   * @return CustomerVo
   */
  CustomerVo findBySourcesId(CustomerSourceIdDto dto);


}
