package com.biz.crm.mdm.business.customer.user.local.service.internal;

import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerRelateOrgVo;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.customer.user.local.entity.CustomerUser;
import com.biz.crm.mdm.business.customer.user.local.entity.CustomerUserRelaCustomer;
import com.biz.crm.mdm.business.customer.user.local.entity.CustomerUserRelaRole;
import com.biz.crm.mdm.business.customer.user.local.repository.CustomerUserRelaCustomerRepository;
import com.biz.crm.mdm.business.customer.user.local.service.CustomerUserRelaCustomerService;
import com.biz.crm.mdm.business.customer.user.local.service.CustomerUserRelaRoleService;
import com.biz.crm.mdm.business.customer.user.local.service.CustomerUserService;
import com.biz.crm.mdm.business.customer.user.sdk.dto.CustomerUserDto;
import com.biz.crm.mdm.business.customer.user.sdk.dto.CustomerUserRelaCustomerDto;
import com.biz.crm.mdm.business.customer.user.sdk.dto.CustomerUserRelaRoleDto;
import com.biz.crm.mdm.business.customer.user.sdk.dto.CustomerUserResetPasswordDto;
import com.biz.crm.mdm.business.customer.user.sdk.event.CustomerUserEventListener;
import com.biz.crm.mdm.business.customer.user.sdk.service.CustomerUserVoService;
import com.biz.crm.mdm.business.customer.user.sdk.vo.CustomerUserRelaCustomerVo;
import com.biz.crm.mdm.business.customer.user.sdk.vo.CustomerUserRelaRoleVo;
import com.biz.crm.mdm.business.customer.user.sdk.vo.CustomerUserVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

/**
 * 客户用户(CustomerUser)表服务实现类
 *
 * @author sunx
 * @since 2021-10-20 16:35:05
 */
@Slf4j
@Service
public class CustomerUserVoServiceImpl implements CustomerUserVoService {

  @Autowired(required = false) private CustomerUserService customerUserService;

  @Autowired(required = false) private CustomerUserRelaCustomerService customerUserRelaCustomerService;

  @Autowired(required = false) private CustomerUserRelaRoleService customerUserRelaRoleService;

  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private List<CustomerUserEventListener> eventListeners;

  @Autowired(required = false)
  private CustomerUserRelaCustomerRepository customerUserRelaCustomerRepository;

  @Autowired(required = false)
  private CustomerVoService customerVoService;

  @Override
  public List<CustomerUserVo> findDetailsByIdsOrUserCodes(
      List<String> ids, List<String> userCodes) {
    List<CustomerUserVo> re = Lists.newLinkedList();
    if (CollectionUtils.isEmpty(ids) && CollectionUtils.isEmpty(userCodes)) {
      return re;
    }
    List<CustomerUser> customerUserList =
        customerUserService.findDetailsByIdsOrUserCodes(ids, userCodes);
    if (CollectionUtils.isEmpty(customerUserList)) {
      return re;
    }

    List<String> userCodeList =
        customerUserList.stream()
            .filter(a -> StringUtils.isNotBlank(a.getUserCode()))
            .map(CustomerUser::getUserCode)
            .collect(Collectors.toList());
    if (CollectionUtils.isEmpty(userCodeList)) {
      return re;
    }

    List<CustomerUserRelaCustomer> customerList =
        customerUserRelaCustomerService.findByUserCodes(userCodeList);

    List<CustomerUserRelaRole> roleList = customerUserRelaRoleService.findByUserCodes(userCodeList);

    re = this.buildCustomerUserVoList(customerUserList, customerList, roleList);
    return re;
  }

  @Override
  @Transactional
  public CustomerUserVo create(CustomerUserDto dto) {
    Validate.notNull(dto, "客户用户信息缺失");
    CustomerUser customerUser =
        this.nebulaToolkitService.copyObjectByWhiteList(
            dto, CustomerUser.class, HashSet.class, ArrayList.class);
    dto.setLockState(
        Optional.ofNullable(dto.getLockState()).orElse(EnableStatusEnum.ENABLE.getCode()));
    this.customerUserService.create(customerUser);
    dto.setUserCode(customerUser.getUserCode());
    this.bindExtInfo(dto);
    CustomerUserVo re = this.buildByDtoAndCustomerUser(dto, customerUser);
    // 发送通知
    if (CollectionUtils.isEmpty(eventListeners)) {
      return re;
    }
    for (CustomerUserEventListener eventListener : eventListeners) {
      eventListener.onCreate(re);
    }
    return re;
  }

  @Override
  @Transactional
  public CustomerUserVo update(CustomerUserDto dto) {
    Validate.notNull(dto, "客户用户信息缺失");
    dto.setLockState(
        Optional.ofNullable(dto.getLockState()).orElse(EnableStatusEnum.ENABLE.getCode()));
    Boolean flag = CollectionUtils.isNotEmpty(eventListeners);
    CustomerUserVo oldVo = null;
    if (Boolean.TRUE.equals(flag)) {
      List<CustomerUserVo> list =
          this.findDetailsByIdsOrUserCodes(Lists.newArrayList(dto.getId()), null);
      if (CollectionUtils.isNotEmpty(list)) {
        oldVo = list.get(0);
      }
    }
    CustomerUser customerUser =
        this.nebulaToolkitService.copyObjectByWhiteList(
            dto, CustomerUser.class, HashSet.class, ArrayList.class);
    this.customerUserService.update(customerUser);
    dto.setUserCode(customerUser.getUserCode());
    this.bindExtInfo(dto);
    CustomerUserVo re = this.buildByDtoAndCustomerUser(dto, customerUser);
    // 发送通知
    if (Boolean.FALSE.equals(flag)) {
      return re;
    }
    for (CustomerUserEventListener eventListener : eventListeners) {
      eventListener.onUpdate(oldVo, re);
    }
    return re;
  }

  @Override
  public CustomerUserVo findDetailsByUserPhone(String userPhone) {
    if (StringUtils.isBlank(userPhone)) {
      return null;
    }
    List<CustomerUser> customerUsers =
        this.customerUserService.findByUserPhone(userPhone, TenantUtils.getTenantCode());
    if (CollectionUtils.isEmpty(customerUsers)) {
      return null;
    }
    CustomerUserVo customerUserVo =
        this.nebulaToolkitService.copyObjectByWhiteList(
            customerUsers.get(0), CustomerUserVo.class, HashSet.class, ArrayList.class);
    ArrayList<String> userCodeList = Lists.newArrayList(customerUserVo.getUserCode());
    List<CustomerUserRelaCustomer> customerList =
        customerUserRelaCustomerService.findByUserCodes(userCodeList);
    if (CollectionUtils.isEmpty(customerList)) {
      return customerUserVo;
    }
    List<CustomerUserRelaCustomerVo> customerUserRelaCustomerVos =
        (List<CustomerUserRelaCustomerVo>)
            this.nebulaToolkitService.copyCollectionByWhiteList(
                customerList,
                CustomerUserRelaCustomer.class,
                CustomerUserRelaCustomerVo.class,
                HashSet.class,
                ArrayList.class);
    customerUserVo.setCustomerInfoList(customerUserRelaCustomerVos);
    return customerUserVo;
  }

  @Override
  public void updatePasswordByIds(CustomerUserResetPasswordDto dto) {
    this.customerUserService.resetPassword(dto);
  }

  /**
   * 保存客户用户关联信息
   *
   * @param dto
   */
  @Transactional
  public void bindExtInfo(CustomerUserDto dto) {
    /*
     * 保存客户用户关联信息：
     * 1、保存关联客户信息
     * 2、保存关联劫色信息
     * */
    Validate.notNull(dto, "客户用户信息缺失");
    List<CustomerUserRelaCustomer> customerList = Lists.newArrayList();
    if (CollectionUtils.isNotEmpty(dto.getCustomerInfoList())) {
      customerList =
          (List<CustomerUserRelaCustomer>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  dto.getCustomerInfoList(),
                  CustomerUserRelaCustomerDto.class,
                  CustomerUserRelaCustomer.class,
                  HashSet.class,
                  ArrayList.class);
      customerList.forEach(a -> a.setUserCode(dto.getUserCode()));
    }
    customerUserRelaCustomerService.saveBatch(customerList, dto.getUserCode());

    List<CustomerUserRelaRole> roleList = Lists.newArrayList();
    if (CollectionUtils.isNotEmpty(dto.getRoleInfoList())) {
      roleList =
          (List<CustomerUserRelaRole>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  dto.getRoleInfoList(),
                  CustomerUserRelaRoleDto.class,
                  CustomerUserRelaRole.class,
                  HashSet.class,
                  ArrayList.class);
      roleList.forEach(a -> a.setUserCode(dto.getUserCode()));
    }
    customerUserRelaRoleService.saveBatch(roleList, dto.getUserCode());
  }

  /**
   * 构建返回vo
   *
   * @param customerUserList
   * @param customerList
   * @return
   */
  private List<CustomerUserVo> buildCustomerUserVoList(
      List<CustomerUser> customerUserList,
      List<CustomerUserRelaCustomer> customerList,
      List<CustomerUserRelaRole> roleList) {
    List<CustomerUserVo> re = Lists.newLinkedList();
    Map<String, List<CustomerUserRelaCustomerVo>> mapCustomer = Maps.newHashMap();
    Map<String, List<CustomerUserRelaRoleVo>> mapRole = Maps.newHashMap();
    if (CollectionUtils.isEmpty(customerUserList)) {
      return re;
    }
    re =
        (List<CustomerUserVo>)
            this.nebulaToolkitService.copyCollectionByWhiteList(
                customerUserList,
                CustomerUser.class,
                CustomerUserVo.class,
                HashSet.class,
                ArrayList.class);

    if (CollectionUtils.isNotEmpty(customerList)) {
      List<CustomerUserRelaCustomerVo> voList =
          (List<CustomerUserRelaCustomerVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  customerList,
                  CustomerUserRelaCustomer.class,
                  CustomerUserRelaCustomerVo.class,
                  HashSet.class,
                  ArrayList.class);
      mapCustomer =
          voList.stream().collect(Collectors.groupingBy(CustomerUserRelaCustomerVo::getUserCode));
    }

    if (CollectionUtils.isNotEmpty(roleList)) {
      List<CustomerUserRelaRoleVo> voList =
          (List<CustomerUserRelaRoleVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  roleList,
                  CustomerUserRelaRole.class,
                  CustomerUserRelaRoleVo.class,
                  HashSet.class,
                  ArrayList.class);
      mapRole = voList.stream().collect(Collectors.groupingBy(CustomerUserRelaRoleVo::getUserCode));
    }

    for (CustomerUserVo item : re) {
      item.setCustomerInfoList(mapCustomer.get(item.getUserCode()));
      item.setRoleInfoList(mapRole.get(item.getUserCode()));
    }
    return re;
  }

  /**
   * 构建响应vo信息
   *
   * @param dto
   * @param customerUser
   * @return
   */
  private CustomerUserVo buildByDtoAndCustomerUser(CustomerUserDto dto, CustomerUser customerUser) {
    if (Objects.isNull(customerUser) || Objects.isNull(dto)) {
      return null;
    }
    CustomerUserVo vo =
        this.nebulaToolkitService.copyObjectByWhiteList(
            customerUser, CustomerUserVo.class, HashSet.class, ArrayList.class);
    if (CollectionUtils.isNotEmpty(dto.getCustomerInfoList())) {
      vo.setCustomerInfoList(
          (List<CustomerUserRelaCustomerVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  dto.getCustomerInfoList(),
                  CustomerUserRelaCustomerDto.class,
                  CustomerUserRelaCustomerVo.class,
                  HashSet.class,
                  ArrayList.class));
    }
    return vo;
  }

  /**
   * 根据用户名查询
   *
   * @param userName
   * @return
   */
  @Override
  public CustomerUserVo findByUserName(String userName) {
    CustomerUser customerUser = this.customerUserService.findByUserName(userName);
    if (ObjectUtils.isEmpty(customerUser)) {
      return null;
    }
    List<String> userCodeList = Lists.newArrayList();
    userCodeList.add(customerUser.getUserCode());
    // 查询关联客户
    List<CustomerUserRelaCustomerVo> customerList =
        customerUserRelaCustomerRepository.findCustomerUserRelaVoByUserCodes(userCodeList);
    //查询关联客户信息
    List<String> customerCodes = customerList.stream().map(CustomerUserRelaCustomerVo::getCustomerCode).collect(Collectors.toList());
    List<CustomerVo> userVoList = customerVoService.findByCustomerCodes(customerCodes);
    // 查询关联角色
    List<CustomerUserRelaRole> roleList = customerUserRelaRoleService.findByUserCodes(userCodeList);

    CustomerUserVo customerUserVo = this.buildCustomerUserVo(customerUser, customerList,userVoList, roleList);
    return customerUserVo;
  }

  @Override
  public CustomerUserVo findByPhone(String phone) {
    CustomerUser customerUser = this.customerUserService.findByPhone(phone);
    if (customerUser == null) {
      return null;
    }
    List<String> userCodeList = Lists.newArrayList();
    userCodeList.add(customerUser.getUserCode());
    // 查询关联客户
    List<CustomerUserRelaCustomerVo> customerList =
        customerUserRelaCustomerRepository.findCustomerUserRelaVoByUserCodes(userCodeList);
    //查询关联客户信息
    List<String> customerCodes = customerList.stream().map(CustomerUserRelaCustomerVo::getCustomerCode).collect(Collectors.toList());
    List<CustomerVo> userVoList = customerVoService.findByCustomerCodes(customerCodes);
    // 查询关联角色
    List<CustomerUserRelaRole> roleList = customerUserRelaRoleService.findByUserCodes(userCodeList);

    CustomerUserVo customerUserVo = this.buildCustomerUserVo(customerUser, customerList,userVoList, roleList);
    return customerUserVo;
  }

  @Override
  public List<CustomerUserVo> findByUserNames(Set<String> userNameSet) {
    List<CustomerUser> list = this.customerUserService.findByUserNames(userNameSet);
    if (ObjectUtils.isEmpty(list)) {
      return Lists.newLinkedList();
    }
    return (List<CustomerUserVo>)
        this.nebulaToolkitService.copyCollectionByBlankList(
            list, CustomerUser.class, CustomerUserVo.class, HashSet.class, ArrayList.class);
  }

  /**
   * 构建客户用户管理客户和角色信息
   * @param customerUser
   * @param customerUserRelaCustomerVoList
   * @param customerVoList
   * @param roleList
   * @return {@link CustomerUserVo}
   */
  private CustomerUserVo buildCustomerUserVo(CustomerUser customerUser
      , List<CustomerUserRelaCustomerVo> customerUserRelaCustomerVoList, List<CustomerVo> customerVoList, List<CustomerUserRelaRole> roleList) {
    CustomerUserVo customerUserVo = this.nebulaToolkitService
        .copyObjectByWhiteList(customerUser, CustomerUserVo.class, HashSet.class, ArrayList.class);
    //组装客户企业组织和客户组织
    if (CollectionUtils.isNotEmpty(customerUserRelaCustomerVoList)) {
      Map<String, CustomerVo> customerVoMap = customerVoList.stream()
          .collect(Collectors.toMap(CustomerVo::getCustomerCode, Function.identity()));
      customerUserRelaCustomerVoList.forEach(o -> {
        CustomerVo customerVo = customerVoMap.get(o.getCustomerCode());
        if(!ObjectUtils.isEmpty(customerVo)){
          if(!org.springframework.util.CollectionUtils.isEmpty(customerVo.getOrgList())){
            CustomerRelateOrgVo customerRelateOrgVo = customerVo.getOrgList().get(0);
            o.setOrgCode(customerRelateOrgVo.getOrgCode());
            o.setOrgName(customerRelateOrgVo.getOrgName());
          }
          o.setCustomerOrgCode(customerVo.getCustomerOrgCode());
          o.setCustomerOrgName(customerVo.getCustomerOrgName());
        }
      });
      customerUserVo.setCustomerInfoList(customerUserRelaCustomerVoList);
    }
    //组装客户用户关联角色
    if (CollectionUtils.isNotEmpty(roleList)) {
      List<CustomerUserRelaRoleVo> voList =
          (List<CustomerUserRelaRoleVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  roleList,
                  CustomerUserRelaRole.class,
                  CustomerUserRelaRoleVo.class,
                  HashSet.class,
                  ArrayList.class);
      customerUserVo.setRoleInfoList(voList);
    }
    return customerUserVo;
  }
}
