package com.biz.crm.mdm.business.customer.user.local.service.internal;

import com.biz.crm.mdm.business.customer.user.local.entity.CustomerUserRelaRole;
import com.biz.crm.mdm.business.customer.user.local.repository.CustomerUserRelaRoleRepository;
import com.biz.crm.mdm.business.customer.user.local.service.CustomerUserRelaRoleService;
import com.biz.crm.mdm.business.customer.user.sdk.dto.CustomerUserRelaRoleBindDto;
import com.biz.crm.mdm.business.customer.user.sdk.dto.CustomerUserRelaRoleUnbindDto;
import com.biz.crm.mdm.business.customer.user.sdk.event.CustomerUserRelateRoleEventListener;
import com.biz.crm.mdm.business.customer.user.sdk.vo.CustomerUserRelaRoleVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 客户用户关联角色(CustomerUserRRole)表服务实现类
 *
 * @author sunx
 * @date 2021-11-01 09:30:14
 */
@Service("customerUserRelaRoleService")
public class CustomerUserRelaRoleServiceImpl implements CustomerUserRelaRoleService {

  @Autowired(required = false) private CustomerUserRelaRoleRepository customerUserRelaRoleRepository;

  @Autowired(required = false)
  @Lazy
  private List<CustomerUserRelateRoleEventListener> customerUserRelateRoleEventListeners;

  @Autowired(required = false) private NebulaToolkitService nebulaToolkitService;

  @Override
  public List<CustomerUserRelaRole> findByUserCodes(List<String> userCodeList) {
    if (CollectionUtils.isEmpty(userCodeList)) {
      return Lists.newLinkedList();
    }
    return this.customerUserRelaRoleRepository.findByUserCodes(userCodeList);
  }

  @Override
  @Transactional
  public void saveBatch(List<CustomerUserRelaRole> roleList, String userCode) {
    Validate.notBlank(userCode, "客户用户编码信息不能为空");
    this.customerUserRelaRoleRepository.deleteByUserCodes(Lists.newArrayList(userCode));
    if (CollectionUtils.isEmpty(roleList)) {
      return;
    }
    Optional<CustomerUserRelaRole> first =
        roleList.stream()
            .filter(
                a ->
                    StringUtils.isBlank(a.getUserCode())
                        || StringUtils.isBlank(a.getRoleCode())
                        || !userCode.equals(a.getUserCode()))
            .findFirst();
    Validate.isTrue(!first.isPresent(), "客户用户编码或角色编码不能为空,且必须属于同一个客户用户");
    Map<String, List<CustomerUserRelaRole>> map =
        roleList.stream().collect(Collectors.groupingBy(CustomerUserRelaRole::getRoleCode));
    map.entrySet().forEach(a -> Validate.isTrue(a.getValue().size() <= 1, "存在重复的角色编码"));
    for (CustomerUserRelaRole item : roleList) {
      item.setId(null);
      item.setTenantCode(TenantUtils.getTenantCode());
    }
    this.customerUserRelaRoleRepository.saveBatch(roleList);
  }

  @Override
  public List<CustomerUserRelaRole> findByRoleCodesAndTenantCode(
      List<String> roleCodeList, String tenantCode) {
    if (CollectionUtils.isEmpty(roleCodeList) || StringUtils.isBlank(tenantCode)) {
      return Lists.newLinkedList();
    }
    return this.customerUserRelaRoleRepository.findByRoleCodesAndTenantCode(
        roleCodeList, tenantCode);
  }

  @Override
  @Transactional
  public void bind(CustomerUserRelaRoleBindDto dto) {
    Validate.notNull(dto, "参数不能为空");
    Validate.notBlank(dto.getNewRoleCode(), "角色编码不能为空");
    Validate.isTrue(CollectionUtils.isNotEmpty(dto.getCustomerUserCodeList()), "客户用户编码不能为空");
    Set<String> set = Sets.newHashSet();
    dto.getCustomerUserCodeList().forEach(a -> Validate.isTrue(set.add(a), "存在重复的客户用户编码信息"));
    List<String> roleCodeList = Lists.newArrayList(dto.getNewRoleCode());
    if (StringUtils.isNotBlank(dto.getOldRoleCode())) {
      roleCodeList.add(dto.getOldRoleCode());
    }
    List<CustomerUserRelaRole> list =
        this.customerUserRelaRoleRepository.findByUserCodesAndRoleCodes(
            dto.getCustomerUserCodeList(), roleCodeList);

    if (CollectionUtils.isNotEmpty(list)) {
      this.unbind(
          list.stream().map(CustomerUserRelaRole::getId).collect(Collectors.toList()), false);
    }

    List<CustomerUserRelaRole> addList = Lists.newLinkedList();
    for (String code : dto.getCustomerUserCodeList()) {
      CustomerUserRelaRole cur = new CustomerUserRelaRole();
      cur.setUserCode(code);
      cur.setRoleCode(dto.getNewRoleCode());
      cur.setTenantCode(TenantUtils.getTenantCode());
      addList.add(cur);
    }
    this.customerUserRelaRoleRepository.saveBatch(addList);
    // 发送通知
    if (CollectionUtils.isEmpty(customerUserRelateRoleEventListeners)) {
      return;
    }
    for (CustomerUserRelateRoleEventListener customerUserRelateRoleEventListener :
        customerUserRelateRoleEventListeners) {
      customerUserRelateRoleEventListener.onBind(
          dto.getOldRoleCode(), dto.getNewRoleCode(), dto.getCustomerUserCodeList());
    }
  }

  @Override
  @Transactional
  public void unbind(CustomerUserRelaRoleUnbindDto dto) {
    Validate.notNull(dto, "解绑参数缺失");
    Validate.notBlank(dto.getRoleCode(), "角色编码参数缺失");
    Validate.isTrue(CollectionUtils.isNotEmpty(dto.getCustomerUserCodeList()), "客户用户编码参数缺失");
    List<CustomerUserRelaRole> list =
        customerUserRelaRoleRepository.findByUserCodesAndRoleCodes(
            dto.getCustomerUserCodeList(), Lists.newArrayList(dto.getRoleCode()));
    Validate.isTrue(CollectionUtils.isNotEmpty(list), "无可解绑的数据");
    List<String> idList =
        list.stream().map(CustomerUserRelaRole::getId).collect(Collectors.toList());
    this.unbind(idList, true);
  }

  /**
   * 解绑
   *
   * @param idList 关系id集合
   * @param needNotifier true需要发通知，false不需要发通知
   */
  @Transactional
  public void unbind(List<String> idList, Boolean needNotifier) {
    Validate.isTrue(CollectionUtils.isNotEmpty(idList), "id集合不能为空");
    List<CustomerUserRelaRole> list = this.customerUserRelaRoleRepository.findByIds(idList);
    this.customerUserRelaRoleRepository.removeByIds(idList);

    if (Boolean.FALSE.equals(needNotifier)) {
      return;
    }
    // 发送通知
    if (CollectionUtils.isEmpty(customerUserRelateRoleEventListeners) || CollectionUtils.isEmpty(list)) {
      return;
    }
    List<CustomerUserRelaRoleVo> voList =
        (List<CustomerUserRelaRoleVo>)
            this.nebulaToolkitService.copyCollectionByBlankList(
                list,
                CustomerUserRelaRole.class,
                CustomerUserRelaRoleVo.class,
                HashSet.class,
                ArrayList.class);
    for (CustomerUserRelateRoleEventListener customerUserRelateRoleEventListener :
        customerUserRelateRoleEventListeners) {
      customerUserRelateRoleEventListener.onUnbind(voList);
    }
  }
}
