package com.biz.crm.mdm.business.customer.user.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.enums.LockStateEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mdm.business.customer.user.local.entity.CustomerUser;
import com.biz.crm.mdm.business.customer.user.local.entity.CustomerUserRelaCustomer;
import com.biz.crm.mdm.business.customer.user.local.repository.CustomerUserRepository;
import com.biz.crm.mdm.business.customer.user.local.service.CustomerUserRelaCustomerService;
import com.biz.crm.mdm.business.customer.user.local.service.CustomerUserService;
import com.biz.crm.mdm.business.customer.user.sdk.constant.CustomerUserConstant;
import com.biz.crm.mdm.business.customer.user.sdk.dto.CustomerUserPaginationDto;
import com.biz.crm.mdm.business.customer.user.sdk.dto.CustomerUserResetPasswordDto;
import com.biz.crm.mdm.business.customer.user.sdk.event.CustomerUserEventListener;
import com.biz.crm.mdm.business.customer.user.sdk.vo.CustomerUserVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.Aes128Utils;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.DigestUtils;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 客户用户(CustomerUser)表服务实现类
 *
 * @author sunx
 * @since 2021-10-20 16:35:05
 */
@Service("customerUserService")
public class CustomerUserServiceImpl implements CustomerUserService {

  @Autowired(required = false)
  private CustomerUserRepository customerUserRepository;

  @Autowired(required = false)
  @Lazy
  private List<CustomerUserEventListener> eventListeners;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private CustomerUserRelaCustomerService customerUserRelaCustomerService;

  @Override
  public Page<CustomerUser> findByConditions(Pageable pageable, CustomerUserPaginationDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new CustomerUserPaginationDto());
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    Page<CustomerUser> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    Page<CustomerUser> byConditions = customerUserRepository.findByConditions(page, dto);
    // 查询关联客户信息
    if (!(byConditions != null && CollectionUtils.isNotEmpty(byConditions.getRecords()))) {
      return byConditions;
    }
    List<CustomerUser> records = byConditions.getRecords();
    List<String> userCodes =
        records.stream()
            .filter(a -> StringUtils.isNotBlank(a.getUserCode()))
            .map(CustomerUser::getUserCode)
            .collect(Collectors.toList());
    List<CustomerUserRelaCustomer> cusUserRelaList =
        customerUserRelaCustomerService.findByUserCodes(userCodes);
    if (CollectionUtils.isEmpty(cusUserRelaList)) {
      return byConditions;
    }
    Map<String, List<CustomerUserRelaCustomer>> cusUserRelaMapByCode =
        cusUserRelaList.stream()
            .collect(Collectors.groupingBy(CustomerUserRelaCustomer::getUserCode));
    // 组装关联客户信息
    for (CustomerUser customerUser : records) {
      String userCode = customerUser.getUserCode();
      if (StringUtils.isNotBlank(userCode) && cusUserRelaMapByCode.containsKey(userCode)) {
        customerUser.setCustomerList(cusUserRelaMapByCode.get(userCode));
      }
    }
    return byConditions;
  }

  @Override
  public CustomerUser findDetailById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return customerUserRepository.findById(id);
  }

  @Override
  @Transactional
  public CustomerUser create(CustomerUser terminalUser) {
    // 校验
    this.createValidation(terminalUser);
    // 如果userCode为空需要期初一个编码，否则执行验重逻辑
    if (StringUtils.isEmpty(terminalUser.getUserCode())) {
      terminalUser.setUserCode(
          generateCodeService.generateCode(CustomerUserConstant.CUSTOMER_USER_CODE, 1).get(0));
    } else {
      Integer count = customerUserRepository.countByUserCode(terminalUser.getUserCode());
      Validate.isTrue(null == count || 1 > count, terminalUser.getUserCode() + "编码已存在");
    }
    terminalUser.setTenantCode(TenantUtils.getTenantCode());
    terminalUser.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    terminalUser.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    terminalUser.setLockState(LockStateEnum.UNLOCK.getCode());
    customerUserRepository.saveOrUpdate(terminalUser);
    return terminalUser;
  }

  @Override
  @Transactional
  public CustomerUser update(CustomerUser terminalUser) {
    this.updateValidation(terminalUser);
    customerUserRepository.saveOrUpdate(terminalUser);
    return terminalUser;
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    customerUserRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
    onEnableOrDisable(ids, EnableStatusEnum.ENABLE);
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    customerUserRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
    onEnableOrDisable(ids, EnableStatusEnum.DISABLE);
  }

  @Override
  @Transactional
  public void updateDelFlagByIds(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    customerUserRepository.updateDelFlagByIds(ids);
    if (CollectionUtils.isEmpty(eventListeners)) {
      return;
    }
    List<CustomerUserVo> voList = findVoListByIds(ids);
    if (CollectionUtils.isEmpty(voList)) {
      return;
    }
    for (CustomerUserEventListener eventListener : eventListeners) {
      eventListener.onDelete(voList);
    }
  }

  @Override
  public List<CustomerUser> findDetailsByIdsOrUserCodes(List<String> ids, List<String> userCodes) {
    if (CollectionUtils.isEmpty(ids) && CollectionUtils.isEmpty(userCodes)) {
      return Lists.newLinkedList();
    }
    return customerUserRepository.findDetailsByIdsOrUserCodes(ids, userCodes);
  }

  @Override
  @Transactional
  public void resetPassword(CustomerUserResetPasswordDto dto) {
    Validate.notNull(dto, "参数不能为空");
    Validate.isTrue(CollectionUtils.isNotEmpty(dto.getIds()), "客户用户id集合不能为空");
    String password =
        Aes128Utils.decrypt(
            dto.getPassword(),
            CustomerUserConstant.ENCRYPT_KEY,
            Aes128Utils.EncodeType.CBC,
            Aes128Utils.Padding.PKCS_7_PADDING);
    String md5Password = DigestUtils.md5DigestAsHex(password.getBytes(StandardCharsets.UTF_8));
    dto.setPassword(md5Password);
    customerUserRepository.resetPassword(dto);
  }

  @Override
  public List<CustomerUser> findByRoleCodesAndTenantCode(
      List<String> roleCodeList, String tenantCode) {
    if (CollectionUtils.isEmpty(roleCodeList) || StringUtils.isBlank(tenantCode)) {
      return Lists.newLinkedList();
    }
    return customerUserRepository.findByRoleCodesAndTenantCode(roleCodeList, tenantCode);
  }

  @Override
  public List<CustomerUser> findByUserPhone(String userPhone, String tenantCode) {
    if (StringUtils.isAnyBlank(userPhone, tenantCode)) {
      return new ArrayList<>();
    }
    return this.customerUserRepository.findByUserPhone(userPhone, tenantCode);
  }

  @Override
  public List<CustomerUser> findByUserNames(Set<String> userNameSet) {
    if (CollectionUtils.isEmpty(userNameSet)) {
      return Lists.newLinkedList();
    }
    return this.customerUserRepository.findByUserNames(userNameSet);
  }

  /**
   * 发送启用禁用变更通知
   *
   * @param ids
   * @param enableStatusEnum
   */
  private void onEnableOrDisable(List<String> ids, EnableStatusEnum enableStatusEnum) {
    if (CollectionUtils.isEmpty(eventListeners)) {
      return;
    }
    List<CustomerUserVo> voList = findVoListByIds(ids);
    if (CollectionUtils.isEmpty(voList)) {
      return;
    }
    for (CustomerUserEventListener event : eventListeners) {
      if (enableStatusEnum.equals(EnableStatusEnum.ENABLE)) {
        event.onEnable(voList);
      } else if (enableStatusEnum.equals(EnableStatusEnum.DISABLE)) {
        event.onDisable(voList);
      }
    }
  }

  /**
   * 只有主表信息，不包含扩展信息
   *
   * @param ids
   * @return
   */
  private List<CustomerUserVo> findVoListByIds(List<String> ids) {
    if (CollectionUtils.isEmpty(ids)) {
      return Lists.newLinkedList();
    }
    List<CustomerUser> list = this.findDetailsByIdsOrUserCodes(ids, null);
    if (CollectionUtils.isEmpty(list)) {
      return Lists.newLinkedList();
    }
    return (List<CustomerUserVo>)
        this.nebulaToolkitService.copyCollectionByWhiteList(
            list, CustomerUser.class, CustomerUserVo.class, HashSet.class, ArrayList.class);
  }

  /**
   * 创建时校验
   *
   * @param terminalUser
   */
  private void createValidation(CustomerUser terminalUser) {
    Validate.notNull(terminalUser, "客户用户信息缺失");
    Validate.notBlank(terminalUser.getUserName(), "客户用户账号信息不能为空");
    // 校验用户账号
    Integer count1 = customerUserRepository.countByUserName(terminalUser.getUserName());
    Validate.isTrue(null == count1 || 1 > count1, terminalUser.getUserName() + "账号已存在");
    // 校验电话
    Validate.isTrue(StringUtils.isNotBlank(terminalUser.getUserPhone()), "电话号码不能为空");
    CustomerUser customerUser =
        this.customerUserRepository.findUserByPhone(terminalUser.getUserPhone());
    Validate.isTrue(Objects.isNull(customerUser), "电话号码已经被占用");
    // 对密码进行 md5 加密
    String password =
        Aes128Utils.decrypt(
            terminalUser.getUserPassword(),
            CustomerUserConstant.ENCRYPT_KEY,
            Aes128Utils.EncodeType.CBC,
            Aes128Utils.Padding.PKCS_7_PADDING);
    String md5Password = DigestUtils.md5DigestAsHex(password.getBytes(StandardCharsets.UTF_8));
    terminalUser.setUserPassword(md5Password);
  }

  /**
   * 更新时校验
   *
   * @param terminalUser
   */
  private void updateValidation(CustomerUser terminalUser) {
    Validate.notNull(terminalUser, "客户用户信息缺失");
    Validate.isTrue(StringUtils.isNotBlank(terminalUser.getId()), "id不能为空");
    // 校验电话
    Validate.isTrue(StringUtils.isNotBlank(terminalUser.getUserPhone()), "电话号码不能为空");
    CustomerUser cusUserByPhone =
        this.customerUserRepository.findUserByPhone(terminalUser.getUserPhone());
    if (!Objects.isNull(cusUserByPhone) && !(cusUserByPhone.getId().equals(terminalUser.getId()))) {
      throw new IllegalArgumentException("电话号码已经被占用");
    }
    String currentId = terminalUser.getId();
    CustomerUser current = customerUserRepository.findById(currentId);
    Validate.notNull(current, "修改信息不存在");
    Validate.isTrue(terminalUser.getUserCode().equals(current.getUserCode()), "编码不能修改");
    Validate.isTrue(terminalUser.getUserName().equals(current.getUserName()), "账号不能修改");
    if (StringUtils.isNotBlank(terminalUser.getUserPassword())) {
      // 对密码进行 md5 加密
      String password =
          Aes128Utils.decrypt(
              terminalUser.getUserPassword(),
              CustomerUserConstant.ENCRYPT_KEY,
              Aes128Utils.EncodeType.CBC,
              Aes128Utils.Padding.PKCS_7_PADDING);
      if (!password.equals(current.getUserPassword())) {
        // 如果不是原来的密码需要md5加密
        String md5Password = DigestUtils.md5DigestAsHex(password.getBytes(StandardCharsets.UTF_8));
        terminalUser.setUserPassword(md5Password);
      } else {
        // 否则还是原来的密码
        terminalUser.setUserPassword(current.getUserPassword());
      }
    } else {
      // 否则还是原来的密码
      terminalUser.setUserPassword(current.getUserPassword());
    }
  }

  /**
   * 根据用户名查询
   *
   * @param userName
   * @return
   */
  @Override
  public CustomerUser findByUserName(String userName) {
    Validate.notBlank(userName, "用户名不能为空");
    return this.customerUserRepository.findByUserName(userName, TenantUtils.getTenantCode());
  }

  @Override
  public CustomerUser findByPhone(String phone) {
    if (StringUtils.isBlank(phone)) {
      return null;
    }
    CustomerUser customerUser = this.customerUserRepository.findUserByPhone(phone);
    return customerUser;
  }


}
