package com.biz.crm.mdm.business.customer.user.local.repository;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.customer.user.local.entity.CustomerUser;
import com.biz.crm.mdm.business.customer.user.local.mapper.CustomerUserMapper;
import com.biz.crm.mdm.business.customer.user.sdk.dto.CustomerUserPaginationDto;
import com.biz.crm.mdm.business.customer.user.sdk.dto.CustomerUserResetPasswordDto;
import com.biz.crm.mdm.business.customer.user.sdk.vo.CustomerUserRelateCustomerVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import java.util.Set;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Component;

import java.util.Date;
import java.util.List;

/**
 * 客户用户(repository)
 *
 * @author sunx
 * @since 2021-10-20 16:35:04
 */
@Component
public class CustomerUserRepository extends ServiceImpl<CustomerUserMapper, CustomerUser> {

  /**
   * 分页
   *
   * @param page
   * @param dto
   * @return
   */
  public Page<CustomerUser> findByConditions(
      Page<CustomerUser> page, CustomerUserPaginationDto dto) {
    dto.setTenantCode(TenantUtils.getTenantCode());
    return baseMapper.findByConditions(page, dto);
  }

  public CustomerUser findById(String id) {
    return this.baseMapper.selectById(id);
  }

  /**
   * 根据id集合获取详情集合
   *
   * @param ids
   * @return
   */
  public List<CustomerUser> findByIds(List<String> ids) {
    return this.lambdaQuery().in(CustomerUser::getId, ids).list();
  }

  /**
   * 根据id集合 更新组织启用/禁用状态
   *
   * @param ids
   * @param enable
   */
  public void updateEnableStatusByIds(List<String> ids, EnableStatusEnum enable) {
    this.lambdaUpdate()
        .in(CustomerUser::getId, ids)
        .set(CustomerUser::getEnableStatus, enable.getCode())
        .update();
  }

  /**
   * 逻辑删除
   *
   * @param ids
   */
  public void updateDelFlagByIds(List<String> ids) {
    this.lambdaUpdate()
        .in(CustomerUser::getId, ids)
        .eq(CustomerUser::getTenantCode,TenantUtils.getTenantCode())    //新增租户编号判断条件
        .set(CustomerUser::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
        .update();
  }

  /**
   * 根据编码获取集合
   *
   * @param userCodes
   * @return
   */
  public List<CustomerUser> findByUserCodes(List<String> userCodes) {
    return this.lambdaQuery()
        .eq(CustomerUser::getTenantCode, TenantUtils.getTenantCode())
        .in(CustomerUser::getUserCode, userCodes)
        .list();
  }

  /**
   * 根据id或编码获取集合
   *
   * @param ids
   * @param userCodes
   * @return
   */
  public List<CustomerUser> findDetailsByIdsOrUserCodes(List<String> ids, List<String> userCodes) {
    return this.lambdaQuery()
        .eq(CustomerUser::getTenantCode, TenantUtils.getTenantCode())
        .and(
            i ->
                i.in(CollectionUtils.isNotEmpty(ids), CustomerUser::getId, ids)
                    .or()
                    .in(
                        CollectionUtils.isNotEmpty(userCodes),
                        CustomerUser::getUserCode,
                        userCodes))
        .list();
  }

  public Integer countByUserCode(String userCode) {
    return lambdaQuery()
        .eq(CustomerUser::getTenantCode, TenantUtils.getTenantCode())
        .eq(CustomerUser::getUserCode, userCode)
        .count();
  }

  /**
   * 重置密码
   *
   * @param dto
   */
  public void resetPassword(CustomerUserResetPasswordDto dto) {
    lambdaUpdate()
        .eq(CustomerUser::getTenantCode, TenantUtils.getTenantCode())
        .in(CustomerUser::getId, dto.getIds())
        .set(CustomerUser::getUserPassword, dto.getPassword())
        .set(CustomerUser::getUpdatePasswordDate, new Date())
        .update();
  }

  public List<CustomerUser> findByRoleCodesAndTenantCode(
      List<String> roleCodeList, String tenantCode) {
    return this.baseMapper.findByRoleCodesAndTenantCode(
        roleCodeList, tenantCode, DelFlagStatusEnum.NORMAL.getCode());
  }

  /**
   * 获取用户于客户关联关系主要新
   *
   * @param customerCodeList
   * @return
   */
  public List<CustomerUserRelateCustomerVo> findRelateCustomerByCustomerCodes(
      List<String> customerCodeList) {
    return this.baseMapper.findRelateCustomerByCustomerCodes(
        customerCodeList, TenantUtils.getTenantCode(), DelFlagStatusEnum.NORMAL.getCode());
  }

  /**
   * 根据账号获取对应记录的数量
   *
   * @param userName
   * @return
   */
  public Integer countByUserName(String userName) {
    return lambdaQuery()
        .eq(CustomerUser::getTenantCode, TenantUtils.getTenantCode())
        .eq(CustomerUser::getUserName, userName)
        .count();
  }

  /**
   * 根据电话号码查询用户
   *
   * @param userPhone
   * @return
   */
  public CustomerUser findUserByPhone(String userPhone) {
    return lambdaQuery()
        .eq(CustomerUser::getTenantCode, TenantUtils.getTenantCode())
        .eq(CustomerUser::getUserPhone, userPhone)
        .one();
  }

  /**
   * 根据客户编码查询客户信息
   *
   * @param codeList
   * @param tenantCode
   * @return
   */
  public List<CustomerUser> findByCustomerCodes(List<String> codeList, String tenantCode) {
    return this.lambdaQuery()
        .eq(CustomerUser::getTenantCode, tenantCode)
        .in(CustomerUser::getUserCode, codeList)
        .eq(CustomerUser::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .list();
  }

  /**
   * 根据用户名查询
   * @param userName
   * @param tenantCode
   * @return
   */
  public CustomerUser findByUserName(String userName, String tenantCode) {
    return lambdaQuery()
        .eq(CustomerUser::getTenantCode, TenantUtils.getTenantCode())
        .eq(CustomerUser::getUserName, userName)
        .eq(CustomerUser::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .one();
  }

  /**
   * 根据用户手机号查询客户用户
   *
   * @param userPhone
   * @param tenantCode
   * @return
   */
  public List<CustomerUser> findByUserPhone(String userPhone, String tenantCode) {
    return this.lambdaQuery()
        .eq(CustomerUser::getTenantCode, tenantCode)
        .eq(CustomerUser::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(CustomerUser::getUserPhone, userPhone)
        .list();
  }

  /**
   * 根据用户账号集合获取用户信息
   *
   * @param userNameSet
   * @return
   */
  public List<CustomerUser> findByUserNames(Set<String> userNameSet) {
    return this.lambdaQuery()
        .eq(CustomerUser::getTenantCode, TenantUtils.getTenantCode())
        .in(CustomerUser::getUserName, userNameSet)
        .list();
  }

  /**
   * 重构查询方法 通过ID和租户编号查询
   * @param id
   * @param tenantCode
   * @return
   */
  public CustomerUser findByIdAndTenantCode(String id, String tenantCode) {
    return this.lambdaQuery()
        .eq(CustomerUser::getTenantCode,tenantCode)
        .in(CustomerUser::getId,id)
        .one();
  }
}
