package com.biz.crm.mdm.business.customerorg.local.service.internal;
/**
 * Created by Bao Hongbin on 2021-10-27 18:00.
 */

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.mdm.business.common.sdk.dto.TreeDto;
import com.biz.crm.mdm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.mdm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mdm.business.common.sdk.service.TreeRuleCodeStrategy;
import com.biz.crm.mdm.business.common.sdk.service.TreeRuleCodeStrategyHolder;
import com.biz.crm.mdm.business.customerorg.local.entity.CustomerOrg;
import com.biz.crm.mdm.business.customerorg.local.repository.CustomerOrgRepository;
import com.biz.crm.mdm.business.customerorg.local.service.CustomerOrgVoService;
import com.biz.crm.mdm.business.customerorg.local.service.helper.CustomerOrgServiceHelper;
import com.biz.crm.mdm.business.customerorg.sdk.common.constant.CustomerOrgConstant;
import com.biz.crm.mdm.business.customerorg.sdk.dto.CustomerOrgCreateDto;
import com.biz.crm.mdm.business.customerorg.sdk.dto.CustomerOrgPaginationDto;
import com.biz.crm.mdm.business.customerorg.sdk.dto.CustomerOrgUpdateDto;
import com.biz.crm.mdm.business.customerorg.sdk.vo.CustomerOrgVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @program: crm
 * @description: 客户组织服务实现
 * @author: Bao Hongbin
 * @create: 2021-10-27 18:00
 **/
@Service
public class CustomerOrgVoServiceImpl implements CustomerOrgVoService {
  @Autowired
  private CustomerOrgRepository customerOrgRepository;
  @Autowired
  private CustomerOrgServiceHelper helper;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private GenerateCodeService generateCode;
  @Autowired(required = false)
  private TreeRuleCodeStrategyHolder treeRuleCodeStrategyHolder;

  @Override
  public Page<CustomerOrgVo> findByConditions(Pageable pageable, CustomerOrgPaginationDto customerOrgPaginationDto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    customerOrgPaginationDto = Optional.ofNullable(customerOrgPaginationDto).orElse(new CustomerOrgPaginationDto());
    customerOrgPaginationDto.setTenantCode(TenantUtils.getTenantCode());
    //白名单ruleCode
    if (StringUtils.isNotEmpty(customerOrgPaginationDto.getUnderCustomerOrgCode())) {
      //如果条件成立，说明本次查询只查询该白名单里的数据及其子集
      CustomerOrg one = customerOrgRepository.findDetailsByCode(
          customerOrgPaginationDto.getUnderCustomerOrgCode(),
          customerOrgPaginationDto.getTenantCode());
      if (one != null) {
        customerOrgPaginationDto.setUnderThisRuleCode(one.getRuleCode());
      }
    }
    //黑名单ruleCode
    if (StringUtils.isNotEmpty(customerOrgPaginationDto.getNotUnderCustomerOrgCode())) {
      //如果条件成立，说明本次查询不查询该黑名单里的数据及其子集
      CustomerOrg one = customerOrgRepository.findDetailsByCode(
          customerOrgPaginationDto.getNotUnderCustomerOrgCode(),
          customerOrgPaginationDto.getTenantCode());
      if (one != null) {
        customerOrgPaginationDto.setNotUnderThisRuleCode(one.getRuleCode());
      }
    }
    //设置已选择项（如果查询条件符合，那么已选择项需要优先显示）
    List<String> selectedCodeList =
        Optional.ofNullable(customerOrgPaginationDto.getSelectedCodeList()).orElse(new ArrayList<>());
    if (StringUtils.isNotEmpty(customerOrgPaginationDto.getSelectedCode())) {
      selectedCodeList.add(customerOrgPaginationDto.getSelectedCode());
    }
    if (!CollectionUtils.isEmpty(selectedCodeList)) {
      customerOrgPaginationDto.setSelectedCodeList(selectedCodeList);
    }
    return customerOrgRepository.findByConditions(pageable, customerOrgPaginationDto);
  }

  @Override
  public CustomerOrgVo findDetailsById(String id) {
    if (!StringUtils.isNotEmpty(id)) {
      return null;
    }
    CustomerOrg one = customerOrgRepository.findDetailsById(id, TenantUtils.getTenantCode());
    if (one == null) {
      return null;
    }
    CustomerOrgVo respVo = nebulaToolkitService.copyObjectByWhiteList(
        one, CustomerOrgVo.class, HashSet.class, ArrayList.class);
    if (StringUtils.isNotEmpty(one.getParentCode())) {
      CustomerOrg parent = customerOrgRepository.findDetailsByCode(one.getParentCode(), TenantUtils.getTenantCode());
      if (parent != null) {
        respVo.setParentName(parent.getCustomerOrgName());
      }
    }
    return respVo;
  }

  @Override
  @Transactional
  public CustomerOrgVo create(CustomerOrgCreateDto customerOrgCreateDto) {
    return this.createForm(customerOrgCreateDto);
  }

  private CustomerOrgVo createForm(CustomerOrgCreateDto customerOrgCreateDto) {
    //1.数据验证
    helper.createValidation(customerOrgCreateDto);
    CustomerOrg customerOrg = this.nebulaToolkitService.copyObjectByWhiteList(
        customerOrgCreateDto, CustomerOrg.class, HashSet.class, ArrayList.class);
    customerOrg.setTenantCode(TenantUtils.getTenantCode());
    customerOrg.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    customerOrg.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    //2.判断产品层级编码是否为空，如果不为空那么查询是否存在，如果为空则设置
    if (StringUtils.isBlank(customerOrgCreateDto.getCustomerOrgCode())) {
      //TODO 各业务编码前缀以前可以在系统设置中进行设置，该功能将在后续改造中实现，先设置为固定默认值
      customerOrg.setCustomerOrgCode(generateCode.generateCode(CustomerOrgConstant.CODE, 1).get(0));
    }
    //3.根据父级设置层级
    int levelNum = 1;
    if (StringUtils.isNotEmpty(customerOrgCreateDto.getParentCode())) {
      CustomerOrg parent = customerOrgRepository.findDetailsByCode(customerOrgCreateDto.getParentCode(), TenantUtils.getTenantCode());
      levelNum = parent.getLevelNum() + 1;
    }
    //4.设置规则（降维）编码
    String ruleCode = getRuleCodeByParentCode(customerOrgCreateDto.getParentCode());
    customerOrg.setRuleCode(ruleCode);
    customerOrg.setLevelNum(levelNum);
    //5.保存数据
    customerOrgRepository.save(customerOrg);
    //6.推送onCreate事件
    CustomerOrgVo customerOrgVo = nebulaToolkitService.copyObjectByWhiteList(
        customerOrg, CustomerOrgVo.class, HashSet.class, ArrayList.class);
    helper.sendCreateEvent(Collections.singletonList(customerOrgVo));
    return customerOrgVo;
  }

  private String getRuleCodeByParentCode(String parentCode) {
    String parentRuleCode = null;
    if (StringUtils.isNotEmpty(parentCode)) {
      CustomerOrg parent = customerOrgRepository.findDetailsByCode(parentCode, TenantUtils.getTenantCode());
      Validate.notNull(parent, "上级客户组织不存在");
      parentRuleCode = parent.getRuleCode();
    }
    List<CustomerOrg> childrenListByParentCode =
        customerOrgRepository.findChildrenListByParentCode(parentCode, TenantUtils.getTenantCode());
    List<TreeDto> childrenDto =
        Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(childrenListByParentCode, CustomerOrg.class
            , TreeDto.class, HashSet.class, ArrayList.class));
    //TODO 编码降维编码长度以前可以在系统设置中进行设置，该功能将在后续改造中实现，先设置为固定默认值
    TreeRuleCodeStrategy treeRuleCodeStrategy = treeRuleCodeStrategyHolder.getStrategy(null);
    return treeRuleCodeStrategy.generate(
        CustomerOrgConstant.RULE_CODE_LENGTH, parentRuleCode, childrenDto);
  }

  @Override
  @Transactional
  public CustomerOrgVo update(CustomerOrgUpdateDto customerOrgUpdateDto) {
    return this.updateForm(customerOrgUpdateDto);
  }

  private CustomerOrgVo updateForm(CustomerOrgUpdateDto customerOrgUpdateDto) {
    //1.数据验证
    helper.updateValidation(customerOrgUpdateDto);
    CustomerOrg customerOrg =
        customerOrgRepository.findDetailsById(customerOrgUpdateDto.getId(), TenantUtils.getTenantCode());
    //2.根据父级设置层级
    int levelNum = 1;
    if (StringUtils.isNotEmpty(customerOrgUpdateDto.getParentCode())) {
      CustomerOrg parent = customerOrgRepository.findDetailsByCode(customerOrgUpdateDto.getParentCode(), TenantUtils.getTenantCode());
      levelNum = parent.getLevelNum() + 1;
    }
    //3.更新数据
    CustomerOrg customerOrgNew = this.nebulaToolkitService.copyObjectByWhiteList(
        customerOrgUpdateDto, CustomerOrg.class, HashSet.class, ArrayList.class);
    customerOrgRepository.updateById(customerOrgNew);
    //4.判断是否需要重置降维编码和更新启用状态
    boolean updateRuleCode = false;
    String enableStatusChangeTo = "";
    if (!(customerOrg.getParentCode() == null ? "" : customerOrg.getParentCode())
        .equals((customerOrgUpdateDto.getParentCode() == null ? "" : customerOrgUpdateDto.getParentCode()))) {
      //上级编码发生变化，重置降维编码
      updateRuleCode = true;
    }
    if (!(customerOrg.getEnableStatus()).equals(customerOrgUpdateDto.getEnableStatus())) {
      //启用状态发生变化
      enableStatusChangeTo = customerOrgUpdateDto.getEnableStatus();
    }
    if (updateRuleCode) {
      //4.1更新降维编码
      String ruleCode = getRuleCodeByParentCode(customerOrgUpdateDto.getParentCode());
      updateCurAndChildrenRuleCode(customerOrgNew.getCustomerOrgCode(), ruleCode, levelNum);
    }
    if (StringUtils.isNotEmpty(enableStatusChangeTo)) {
      //4.2更新启用禁用状态
      if (EnableStatusEnum.ENABLE.getCode().equals(enableStatusChangeTo)) {
        this.enableBatch(Collections.singletonList(customerOrgNew.getId()));
      } else if (EnableStatusEnum.DISABLE.getCode().equals(enableStatusChangeTo)) {
        this.disableBatch(Collections.singletonList(customerOrgNew.getId()));
      } else {
        throw new IllegalArgumentException("启用状态错误");
      }
    }
    //6.推送onUpdate事件
    //再次查询更新后的数据
    customerOrg = customerOrgRepository.findDetailsById(customerOrgUpdateDto.getId(), TenantUtils.getTenantCode());
    CustomerOrgVo customerOrgVo = nebulaToolkitService.copyObjectByWhiteList(
        customerOrg, CustomerOrgVo.class, HashSet.class, ArrayList.class);
    helper.sendUpdateEvent(Collections.singletonList(customerOrgVo));
    return customerOrgVo;
  }

  /**
   * 更新自己及子集的降维编码
   *
   * @param customerOrgCode
   * @param ruleCode
   * @param levelNum
   */
  private void updateCurAndChildrenRuleCode(String customerOrgCode, String ruleCode, int levelNum) {
    //更新当前
    CustomerOrg customerOrg = customerOrgRepository.findDetailsByCode(customerOrgCode, TenantUtils.getTenantCode());
    customerOrg.setRuleCode(ruleCode);
    customerOrg.setLevelNum(levelNum);
    customerOrgRepository.updateById(customerOrg);
    //查询下一层
    List<CustomerOrg> childrenList = customerOrgRepository.findChildrenListByParentCode(customerOrgCode, TenantUtils.getTenantCode());
    if (CollectionUtils.isEmpty(childrenList)) {
      return;
    }
    //遍历下级
    TreeRuleCodeStrategy treeRuleCodeStrategy = this.treeRuleCodeStrategyHolder.getStrategy(null);
    for (int i = 0; i < childrenList.size(); i++) {
      //递归调用
      updateCurAndChildrenRuleCode(childrenList.get(i).getCustomerOrgCode(),
          ruleCode + treeRuleCodeStrategy.generateByNum(CustomerOrgConstant.RULE_CODE_LENGTH, i + 1),
          levelNum + 1);
    }
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "当进行启用操作时，业务技术编号信息必须传入");
    //1.获取需要启用的数据
    List<CustomerOrg> customerOrgList =
        customerOrgRepository.findListByIds(ids, TenantUtils.getTenantCode())
            .stream().filter(x -> !EnableStatusEnum.ENABLE.getCode().equals(x.getEnableStatus()))
            .collect(Collectors.toList());
    if (CollectionUtils.isEmpty(customerOrgList)) {
      return;
    }
    //2.根据ruleCode获取其所有的上级
    List<String> collect = customerOrgList.stream().map(CustomerOrg::getRuleCode).collect(Collectors.toList());
    TreeRuleCodeStrategy treeRuleCodeStrategy = this.treeRuleCodeStrategyHolder.getStrategy(null);
    Set<String> parentRuleCodesExcludeSelf =
        treeRuleCodeStrategy.findParentRuleCodeByRuleCodesExcludeAnySelf(
            CustomerOrgConstant.RULE_CODE_LENGTH, collect);
    if (!parentRuleCodesExcludeSelf.isEmpty()) {
      //3.如果存在未启用的上级层级，不能启用当前层级
      List<CustomerOrg> parentList = customerOrgRepository.findListByRuleCodes(parentRuleCodesExcludeSelf, TenantUtils.getTenantCode())
          .stream().filter(x -> !EnableStatusEnum.ENABLE.getCode().equals(x.getEnableStatus()))
          .collect(Collectors.toList());
      Validate.isTrue(CollectionUtils.isEmpty(parentList), "存在未启用的上级层级，不能启用当前层级");
    }
    //4.更新启用状态
    customerOrgList.forEach(customerOrg -> customerOrg.setEnableStatus(EnableStatusEnum.ENABLE.getCode()));
    customerOrgRepository.updateBatchById(customerOrgList);
    //5.推送onEnable事件
    ArrayList<CustomerOrgVo> customerOrgVos =
        Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(customerOrgList, CustomerOrg.class
            , CustomerOrgVo.class, HashSet.class, ArrayList.class));
    helper.sendEnableEvent(customerOrgVos);
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "当进行禁用操作时，业务技术编号信息必须传入");
    //1.获取需要禁用的数据
    List<CustomerOrg> customerOrgList = customerOrgRepository.findListByIds(ids, TenantUtils.getTenantCode());
    if (CollectionUtils.isEmpty(customerOrgList)) {
      return;
    }
    //2.使用ruleCode模糊查询所有需要禁用的数据
    List<String> ruleCodes = customerOrgList.stream().map(CustomerOrg::getRuleCode).collect(Collectors.toList());
    List<CustomerOrg> childrenByRuleCodeList =
        customerOrgRepository.findCurAndChildrenByRuleCodeList(ruleCodes, EnableStatusEnum.ENABLE.getCode(), TenantUtils.getTenantCode());
    if (!CollectionUtils.isEmpty(childrenByRuleCodeList)) {
      //3.更新启用状态
      childrenByRuleCodeList.forEach(customerOrg -> customerOrg.setEnableStatus(EnableStatusEnum.DISABLE.getCode()));
      customerOrgRepository.updateBatchById(childrenByRuleCodeList);
      //4.推送onDisable事件
      ArrayList<CustomerOrgVo> customerOrgVos =
          Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(childrenByRuleCodeList, CustomerOrg.class
              , CustomerOrgVo.class, HashSet.class, ArrayList.class));
      helper.sendDisableEvent(customerOrgVos);
    }
  }

  @Override
  @Transactional
  public void deleteBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "当进行删除操作时，业务技术编号信息必须传入");
    //1.获取需要删除的数据
    List<CustomerOrg> customerOrgs = customerOrgRepository.findListByIds(ids, TenantUtils.getTenantCode());
    Validate.isTrue(!CollectionUtils.isEmpty(customerOrgs), "无效的业务技术编号信息");
    List<String> productLevelCodeList =
        customerOrgs.stream().map(CustomerOrg::getCustomerOrgCode).collect(Collectors.toList());
    //2.判断是否存在子层级
    List<CustomerOrg> childrenList =
        customerOrgRepository.findChildrenListByParentCodes(productLevelCodeList, TenantUtils.getTenantCode())
            .stream().filter(o -> !ids.contains(o.getId())).collect(Collectors.toList());
    Validate.isTrue(CollectionUtils.isEmpty(childrenList), "当前产品层级包含子层级，无法删除，若需要删除请先删除子层级");
    //3.删除产品层级
    customerOrgRepository.removeByIds(ids);
    //4.推送onDelete事件
    ArrayList<CustomerOrgVo> productLevelVos =
        Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(customerOrgs, CustomerOrg.class
            , CustomerOrgVo.class, HashSet.class, ArrayList.class));
    helper.sendDeleteEvent(productLevelVos);
  }

  @Override
  @Transactional
  public void resetAllRuleCode() {
    //1.查找parentCode不为空但找不到对应上级的数据，将其parentCode设为空
    customerOrgRepository.updateOrphanParentCodeNull(TenantUtils.getTenantCode());
    //2.查找所有parentCode为空的数据（相当于第一层数据）
    List<CustomerOrg> topList =
        customerOrgRepository.findListWithoutParentCode(TenantUtils.getTenantCode());
    //3.递归设置其ruleCode和其所有子级的ruleCode
    TreeRuleCodeStrategy treeRuleCodeStrategy = this.treeRuleCodeStrategyHolder.getStrategy(null);
    for (int i = 0; i < topList.size(); i++) {
      //递归调用
      updateCurAndChildrenRuleCode(topList.get(i).getCustomerOrgCode(),
          treeRuleCodeStrategy.generateByNum(CustomerOrgConstant.RULE_CODE_LENGTH, i + 1),
          1);
    }
  }

  @Override
  public List<CustomerOrgVo> findAll() {
    List<CustomerOrg> all = customerOrgRepository.findAll(TenantUtils.getTenantCode());
    return Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(all, CustomerOrg.class
        , CustomerOrgVo.class, HashSet.class, ArrayList.class));
  }
}
