package com.biz.crm.mdm.business.customeruser.local.service.internal;

import com.biz.crm.mdm.business.customeruser.local.entity.CustomerUserRRole;
import com.biz.crm.mdm.business.customeruser.local.repository.CustomerUserRRoleRepository;
import com.biz.crm.mdm.business.customeruser.local.service.CustomerUserRRoleService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 客户用户关联角色(CustomerUserRRole)表服务实现类
 *
 * @author sunx
 * @date 2021-11-01 09:30:14
 */
@Slf4j
@Service("customerUserRRoleService")
public class CustomerUserRRoleServiceImpl implements CustomerUserRRoleService {

  @Autowired
  private CustomerUserRRoleRepository customerUserRRoleRepository;

  @Override
  public List<CustomerUserRRole> findByUserCodes(List<String> userCodeList) {
    if (CollectionUtils.isEmpty(userCodeList)) {
      return Lists.newArrayList();
    }
    return customerUserRRoleRepository.findByUserCodes(userCodeList);
  }

  @Override
  @Transactional
  public void saveBatch(List<CustomerUserRRole> roleList, String userCode) {
    Validate.notBlank(userCode, "客户用户编码信息不能为空");
    customerUserRRoleRepository.deleteByUserCodes(Lists.newArrayList(userCode));
    if (CollectionUtils.isEmpty(roleList)) {
      return;
    }
    Optional<CustomerUserRRole> first = roleList.stream().filter(
        a -> StringUtils.isBlank(a.getUserCode())
            || StringUtils.isBlank(a.getRoleCode())
            || !userCode.equals(a.getUserCode()))
        .findFirst();
    Validate.isTrue(!first.isPresent(), "客户用户编码或角色编码不能为空,且必须属于同一个客户用户");
    Map<String, List<CustomerUserRRole>> map = roleList.stream().collect(
        Collectors.groupingBy(CustomerUserRRole::getRoleCode));
    map.entrySet().forEach(a -> {
      Validate.isTrue(a.getValue().size() <= 1, "存在重复的角色编码");
    });
    for (CustomerUserRRole item : roleList) {
      item.setId(null);
      item.setTenantCode(TenantUtils.getTenantCode());
    }
    customerUserRRoleRepository.saveBatch(roleList);
  }

  @Override
  public List<CustomerUserRRole> findByRoleCodesAndTenantCode(List<String> roleCodeList,
      String tenantCode) {
    if (CollectionUtils.isEmpty(roleCodeList) || StringUtils.isBlank(tenantCode)) {
      return Lists.newArrayList();
    }
    return customerUserRRoleRepository.findByRoleCodesAndTenantCode(roleCodeList, tenantCode);
  }
}
