package com.biz.crm.mdm.business.customeruser.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.mdm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.mdm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mdm.business.customeruser.local.entity.CustomerUser;
import com.biz.crm.mdm.business.customeruser.local.repository.CustomerUserRepository;
import com.biz.crm.mdm.business.customeruser.local.service.CustomerUserService;
import com.biz.crm.mdm.business.customeruser.sdk.constant.CustomerUserConstant;
import com.biz.crm.mdm.business.customeruser.sdk.dto.CustomerUserPaginationDto;
import com.biz.crm.mdm.business.customeruser.sdk.dto.CustomerUserResetPasswordDto;
import com.biz.crm.mdm.business.customeruser.sdk.event.CustomerUserEventListener;
import com.biz.crm.mdm.business.customeruser.sdk.vo.CustomerUserVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 客户用户(CustomerUser)表服务实现类
 *
 * @author sunx
 * @since 2021-10-20 16:35:05
 */
@Slf4j
@Service("terminalUserService")
public class CustomerUserServiceImpl implements CustomerUserService {

  @Autowired
  private CustomerUserRepository customerUserRepository;

  @Autowired(required = false)
  private List<CustomerUserEventListener> eventListeners;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public Page<CustomerUser> findByConditions(Pageable pageable, CustomerUserPaginationDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new CustomerUserPaginationDto());
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    Page<CustomerUser> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return customerUserRepository.findByConditions(page, dto);
  }

  @Override
  public CustomerUser findDetailById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return customerUserRepository.findById(id);
  }

  @Override
  @Transactional
  public CustomerUser create(CustomerUser terminalUser) {
    Validate.notNull(terminalUser, "客户用户信息缺失");
    // 如果userCode为空需要期初一个编码，否则执行验重逻辑
    if (StringUtils.isEmpty(terminalUser.getUserCode())) {
      terminalUser
          .setUserCode(
              generateCodeService.generateCode(CustomerUserConstant.CUSTOMER_USER_CODE, 1).get(0));
    } else {
      Integer count = customerUserRepository.countByUserCode(terminalUser.getUserCode());
      Validate.isTrue(null == count || 1 > count, terminalUser.getUserCode() + "编码已存在");
    }
    terminalUser.setTenantCode(TenantUtils.getTenantCode());
    terminalUser.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    terminalUser.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    customerUserRepository.saveOrUpdate(terminalUser);
    // 发送通知
    if (CollectionUtils.isEmpty(eventListeners)) {
      return terminalUser;
    }
    CustomerUserVo vo = this.nebulaToolkitService
        .copyObjectByWhiteList(terminalUser, CustomerUserVo.class, HashSet.class, ArrayList.class);
    for (CustomerUserEventListener eventListener : eventListeners) {
      eventListener.onCreate(vo);
    }
    return terminalUser;
  }

  @Override
  @Transactional
  public CustomerUser update(CustomerUser terminalUser) {
    Validate.notNull(terminalUser, "客户用户信息缺失");
    Validate.isTrue(StringUtils.isNotBlank(terminalUser.getId()), "id不能为空");
    String currentId = terminalUser.getId();
    CustomerUserVo oldVo = null;
    CustomerUser current = customerUserRepository.findById(currentId);
    current = Validate.notNull(current, "修改信息不存在");
    Validate.isTrue(terminalUser.getUserCode().equals(current.getUserCode()), "编码不能修改");
    if (CollectionUtils.isNotEmpty(eventListeners)) {
      oldVo = this.nebulaToolkitService
          .copyObjectByBlankList(current, CustomerUserVo.class, HashSet.class, ArrayList.class);
    }
    customerUserRepository.saveOrUpdate(terminalUser);
    // 发送修改通知
    if (CollectionUtils.isEmpty(eventListeners)) {
      return terminalUser;
    }
    CustomerUserVo newVo = this.nebulaToolkitService
        .copyObjectByWhiteList(terminalUser, CustomerUserVo.class, HashSet.class, ArrayList.class);
    for (CustomerUserEventListener eventListener : eventListeners) {
      eventListener.onUpdate(oldVo, newVo);
    }
    return terminalUser;
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    customerUserRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
    onEnableOrDisable(ids, EnableStatusEnum.ENABLE);
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    customerUserRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
    onEnableOrDisable(ids, EnableStatusEnum.ENABLE);
  }

  @Override
  @Transactional
  public void updateDelFlagByIds(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    customerUserRepository.updateDelFlagByIds(ids);
    if (CollectionUtils.isEmpty(eventListeners)) {
      return;
    }
    List<CustomerUserVo> voList = findVoListByIds(ids);
    if (CollectionUtils.isEmpty(voList)) {
      return;
    }
    for (CustomerUserEventListener eventListener : eventListeners) {
      eventListener.onDelete(voList);
    }
  }

  @Override
  public List<CustomerUser> findDetailsByIdsOrUserCodes(List<String> ids, List<String> userCodes) {
    if (CollectionUtils.isEmpty(ids) && CollectionUtils.isEmpty(userCodes)) {
      return Lists.newArrayList();
    }
    if (CollectionUtils.isNotEmpty(ids)) {
      return customerUserRepository.findByIds(ids);
    }
    return customerUserRepository.findByUserCodes(userCodes);
  }

  @Override
  @Transactional
  public void resetPassword(CustomerUserResetPasswordDto dto) {
    Validate.notNull(dto, "参数不能为空");
    Validate.isTrue(CollectionUtils.isNotEmpty(dto.getIds()), "客户用户id集合不能为空");
    customerUserRepository.resetPassword(dto);
  }

  @Override
  public List<CustomerUser> findByRoleCodesAndTenantCode(List<String> roleCodeList,
      String tenantCode) {
    if (CollectionUtils.isEmpty(roleCodeList) || StringUtils.isBlank(tenantCode)) {
      return Lists.newArrayList();
    }
    return customerUserRepository.findByRoleCodesAndTenantCode(roleCodeList, tenantCode);
  }

  /**
   * 发送启用禁用变更通知
   *
   * @param ids
   * @param enableStatusEnum
   */
  private void onEnableOrDisable(List<String> ids, EnableStatusEnum enableStatusEnum) {
    if (CollectionUtils.isEmpty(eventListeners)) {
      return;
    }
    List<CustomerUserVo> voList = findVoListByIds(ids);
    if (CollectionUtils.isEmpty(voList)) {
      return;
    }
    for (CustomerUserEventListener event : eventListeners) {
      if (enableStatusEnum.equals(EnableStatusEnum.ENABLE)) {
        event.onEnable(voList);
      } else if (enableStatusEnum.equals(EnableStatusEnum.DISABLE)) {
        event.onDisable(voList);
      }
    }
  }

  /**
   * 只有主表信息，不包含扩展信息
   *
   * @param ids
   * @return
   */
  private List<CustomerUserVo> findVoListByIds(List<String> ids) {
    if (CollectionUtils.isEmpty(ids)) {
      return Lists.newArrayList();
    }
    List<CustomerUser> list = this.findDetailsByIdsOrUserCodes(ids, null);
    if (CollectionUtils.isEmpty(list)) {
      return Lists.newArrayList();
    }
    return (List<CustomerUserVo>) this.nebulaToolkitService
        .copyCollectionByWhiteList(list, CustomerUser.class, CustomerUserVo.class, HashSet.class,
            ArrayList.class);
  }
}
