package com.biz.crm.mdm.business.customeruser.local.service.internal;

import com.biz.crm.mdm.business.customeruser.local.entity.CustomerUser;
import com.biz.crm.mdm.business.customeruser.local.entity.CustomerUserRCustomer;
import com.biz.crm.mdm.business.customeruser.local.entity.CustomerUserRRole;
import com.biz.crm.mdm.business.customeruser.local.service.CustomerUserRCustomerService;
import com.biz.crm.mdm.business.customeruser.local.service.CustomerUserRRoleService;
import com.biz.crm.mdm.business.customeruser.local.service.CustomerUserService;
import com.biz.crm.mdm.business.customeruser.sdk.dto.CustomerUserDto;
import com.biz.crm.mdm.business.customeruser.sdk.dto.CustomerUserRCustomerDto;
import com.biz.crm.mdm.business.customeruser.sdk.dto.CustomerUserRRoleDto;
import com.biz.crm.mdm.business.customeruser.sdk.service.CustomerUserVoService;
import com.biz.crm.mdm.business.customeruser.sdk.vo.CustomerUserRCustomerVo;
import com.biz.crm.mdm.business.customeruser.sdk.vo.CustomerUserRRoleVo;
import com.biz.crm.mdm.business.customeruser.sdk.vo.CustomerUserVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 客户用户(CustomerUser)表服务实现类
 *
 * @author sunx
 * @since 2021-10-20 16:35:05
 */
@Slf4j
@Service
public class CustomerUserVoServiceImpl implements CustomerUserVoService {

  @Autowired
  private CustomerUserService customerUserService;

  @Autowired
  private CustomerUserRCustomerService customerUserRCustomerService;

  @Autowired
  private CustomerUserRRoleService customerUserRRoleService;

  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public List<CustomerUserVo> findDetailsByIdsOrUserCodes(List<String> ids,
      List<String> userCodes) {
    List<CustomerUserVo> re = Lists.newArrayList();
    if (CollectionUtils.isEmpty(ids) && CollectionUtils.isEmpty(userCodes)) {
      return re;
    }
    List<CustomerUser> customerUserList = customerUserService
        .findDetailsByIdsOrUserCodes(ids, userCodes);
    if (CollectionUtils.isEmpty(customerUserList)) {
      return re;
    }

    List<String> userCodeList = customerUserList.stream()
        .filter(a -> StringUtils.isNotBlank(a.getUserCode())).map(CustomerUser::getUserCode)
        .collect(
            Collectors.toList());
    if (CollectionUtils.isEmpty(userCodeList)) {
      return re;
    }

    List<CustomerUserRCustomer> customerList = customerUserRCustomerService
        .findByUserCodes(userCodeList);

    List<CustomerUserRRole> roleList = customerUserRRoleService
        .findByUserCodes(userCodeList);

    re = this.buildCustomerUserVoList(customerUserList, customerList, roleList);
    return re;
  }


  @Override
  @Transactional
  public CustomerUserVo create(CustomerUserDto dto) {
    Validate.notNull(dto, "客户用户信息缺失");
    CustomerUser customerUser = this.nebulaToolkitService
        .copyObjectByWhiteList(dto, CustomerUser.class, HashSet.class, ArrayList.class);
    this.customerUserService.create(customerUser);
    dto.setUserCode(customerUser.getUserCode());
    this.bindExtInfo(dto);
    return this.buildByDtoAndCustomerUser(dto, customerUser);
  }


  @Override
  @Transactional
  public CustomerUserVo update(CustomerUserDto dto) {
    Validate.notNull(dto, "客户用户信息缺失");
    CustomerUser customerUser = this.nebulaToolkitService
        .copyObjectByWhiteList(dto, CustomerUser.class, HashSet.class, ArrayList.class);
    this.customerUserService.update(customerUser);
    dto.setUserCode(customerUser.getUserCode());
    this.bindExtInfo(dto);
    return this.buildByDtoAndCustomerUser(dto, customerUser);
  }

  /**
   * 保存客户用户关联信息
   *
   * @param dto
   */
  @Transactional
  public void bindExtInfo(CustomerUserDto dto) {
    /*
     * 保存客户用户关联信息：
     * 1、保存关联客户信息
     * 2、保存关联劫色信息
     * */
    Validate.notNull(dto, "客户用户信息缺失");
    List<CustomerUserRCustomer> customerList = Lists.newArrayList();
    if (CollectionUtils.isNotEmpty(dto.getCustomerList())) {
      customerList = (List<CustomerUserRCustomer>) this.nebulaToolkitService
          .copyCollectionByWhiteList(dto.getCustomerList(), CustomerUserRCustomerDto.class,
              CustomerUserRCustomer.class,
              HashSet.class, ArrayList.class);
      customerList.forEach(a -> a.setUserCode(dto.getUserCode()));
    }
    customerUserRCustomerService.saveBatch(customerList, dto.getUserCode());

    List<CustomerUserRRole> roleList = Lists.newArrayList();
    if (CollectionUtils.isNotEmpty(dto.getRoleList())) {
      roleList = (List<CustomerUserRRole>) this.nebulaToolkitService
          .copyCollectionByWhiteList(dto.getRoleList(), CustomerUserRRoleDto.class,
              CustomerUserRRole.class,
              HashSet.class, ArrayList.class);
      roleList.forEach(a -> a.setUserCode(dto.getUserCode()));
    }
    customerUserRRoleService.saveBatch(roleList, dto.getUserCode());
  }

  /**
   * 构建返回vo
   *
   * @param customerUserList
   * @param customerList
   * @return
   */
  private List<CustomerUserVo> buildCustomerUserVoList(List<CustomerUser> customerUserList,
      List<CustomerUserRCustomer> customerList, List<CustomerUserRRole> roleList) {
    List<CustomerUserVo> re = Lists.newArrayList();
    Map<String, List<CustomerUserRCustomerVo>> mapCustomer = Maps.newHashMap();
    Map<String, List<CustomerUserRRoleVo>> mapRole = Maps.newHashMap();
    if (CollectionUtils.isEmpty(customerUserList)) {
      return re;
    }
    re = (List<CustomerUserVo>) this.nebulaToolkitService
        .copyCollectionByWhiteList(customerUserList, CustomerUser.class, CustomerUserVo.class,
            HashSet.class, ArrayList.class);

    if (CollectionUtils.isNotEmpty(customerList)) {
      List<CustomerUserRCustomerVo> voList = (List<CustomerUserRCustomerVo>) this.nebulaToolkitService
          .copyCollectionByWhiteList(customerList, CustomerUserRCustomer.class,
              CustomerUserRCustomerVo.class,
              HashSet.class, ArrayList.class);
      mapCustomer = voList.stream()
          .collect(Collectors.groupingBy(CustomerUserRCustomerVo::getUserCode));
    }

    if (CollectionUtils.isNotEmpty(roleList)) {
      List<CustomerUserRRoleVo> voList = (List<CustomerUserRRoleVo>) this.nebulaToolkitService
          .copyCollectionByWhiteList(roleList, CustomerUserRRole.class,
              CustomerUserRRoleVo.class,
              HashSet.class, ArrayList.class);
      mapRole = voList.stream()
          .collect(Collectors.groupingBy(CustomerUserRRoleVo::getUserCode));
    }

    for (CustomerUserVo item : re) {
      item.setCustomerList(mapCustomer.get(item.getUserCode()));
      item.setRoleList(mapRole.get(item.getUserCode()));
    }
    return re;
  }

  /**
   * 构建响应vo信息
   *
   * @param dto
   * @param customerUser
   * @return
   */
  private CustomerUserVo buildByDtoAndCustomerUser(CustomerUserDto dto, CustomerUser customerUser) {
    if (Objects.isNull(customerUser) || Objects.isNull(dto)) {
      return null;
    }
    CustomerUserVo vo = this.nebulaToolkitService
        .copyObjectByWhiteList(customerUser, CustomerUserVo.class, HashSet.class, ArrayList.class);
    if (CollectionUtils.isNotEmpty(dto.getCustomerList())) {
      vo.setCustomerList(
          (List<CustomerUserRCustomerVo>) this.nebulaToolkitService
              .copyCollectionByWhiteList(dto.getCustomerList(),
                  CustomerUserRCustomerDto.class, CustomerUserRCustomerVo.class, HashSet.class,
                  ArrayList.class));
    }
    return vo;
  }
}
