package com.biz.crm.mdm.business.dictionary.local.repository;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.dictionary.local.entity.DictTypeEntity;
import com.biz.crm.mdm.business.dictionary.local.mapper.DictTypeMapper;
import com.biz.crm.mdm.business.dictionary.sdk.dto.DictTypeDto;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictTypeVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.ibatis.annotations.Param;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.Collection;
import java.util.List;
import java.util.Optional;

/**
 * 物料数据库操作层实现
 *
 * @author pengxi
 */
@Component
public class DictTypeRepository extends ServiceImpl<DictTypeMapper, DictTypeEntity> {

  /**
   * 分页条件查询
   *
   * @param pageable 分页参数
   * @param dto      业务参数
   * @return Page
   */
  public Page<DictTypeVo> findByConditions(Pageable pageable, @Param("dto") DictTypeDto dto) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new DictTypeDto());
    dto.setTenantCode(TenantUtils.getTenantCode());
    Page<DictTypeEntity> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return this.baseMapper.findByConditions(page, dto);
  }

  /**
   * 根据物料编码获取详情
   *
   * @param tenantCode 租户编码
   * @return
   */
  public List<DictTypeEntity> findAllDictType(@Param("tenantCode") String tenantCode) {
    LambdaQueryWrapper<DictTypeEntity> wrapper = Wrappers.<DictTypeEntity>lambdaQuery();
    wrapper.eq(DictTypeEntity::getTenantCode, tenantCode);
    wrapper.eq(DictTypeEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode());
    return this.baseMapper.selectList(wrapper);
  }

  /**
   * 根据物料编码获取详情
   *
   * @param tenantCode    租户编码
   * @param dictTypeCodes 物料编码集合
   * @return
   */
  public List<DictTypeEntity> findByDictTypeCodes(@Param("tenantCode") String tenantCode, @Param("dictTypeCodes") Collection<String> dictTypeCodes) {
    LambdaQueryWrapper<DictTypeEntity> wrapper = Wrappers.<DictTypeEntity>lambdaQuery();
    wrapper.eq(DictTypeEntity::getTenantCode, tenantCode);
    wrapper.in(DictTypeEntity::getDictTypeCode, dictTypeCodes);
    wrapper.eq(DictTypeEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode());
    return this.baseMapper.selectList(wrapper);
  }

  /**
   * 根据物料编码获取详情
   *
   * @param tenantCode   租户编码
   * @param dictTypeCode 物料编码
   * @return
   */
  public DictTypeEntity findByDictTypeCode(@Param("tenantCode") String tenantCode, @Param("dictTypeCode") String dictTypeCode) {
    LambdaQueryWrapper<DictTypeEntity> wrapper = Wrappers.<DictTypeEntity>lambdaQuery();
    wrapper.eq(DictTypeEntity::getTenantCode, tenantCode);
    wrapper.eq(DictTypeEntity::getDictTypeCode, dictTypeCode);
    return this.baseMapper.selectOne(wrapper);
  }

  /**
   * 根据主键集合，修改 enable_status
   *
   * @param enable 启禁用状态
   * @param ids    ID集合
   */
  public void updateEnableStatusByIdIn(EnableStatusEnum enable, Collection<String> ids) {
    if (CollectionUtils.isEmpty(ids)) {
      return;
    }
    LambdaUpdateWrapper<DictTypeEntity> updateWrapper = new LambdaUpdateWrapper<>();
    updateWrapper.set(DictTypeEntity::getEnableStatus, enable.getCode())
        .in(DictTypeEntity::getId, ids);
    this.baseMapper.update(null, updateWrapper);
  }

  /**
   * 根据主键集合，修改 del_flag
   *
   * @param delFlag 删除状态
   * @param ids    ID集合
   */
  public void updateDelFlagByIdIn(DelFlagStatusEnum delFlag, Collection<String> ids) {
    if (CollectionUtils.isEmpty(ids)) {
      return;
    }
    LambdaUpdateWrapper<DictTypeEntity> updateWrapper = new LambdaUpdateWrapper<>();
    updateWrapper.set(DictTypeEntity::getDelFlag, delFlag.getCode())
        .in(DictTypeEntity::getId, ids);
    this.baseMapper.update(null, updateWrapper);
  }

}
