package com.biz.crm.mdm.business.dictionary.local.repository;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.dictionary.local.entity.DictAttrConfEntity;
import com.biz.crm.mdm.business.dictionary.local.entity.DictDataEntity;
import com.biz.crm.mdm.business.dictionary.local.mapper.DictAttrConfMapper;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.ibatis.annotations.Param;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.Collection;
import java.util.List;

/**
 * 物料数据库操作层实现
 *
 * @author pengxi
 */
@Component
public class DictAttrConfRepository extends ServiceImpl<DictAttrConfMapper, DictAttrConfEntity> {

  /**
   * 根据物料编码获取详情
   *
   * @param tenantCode   租户编码
   * @param dictTypeCode 物料编码
   * @return
   */
  public List<DictAttrConfEntity> findByDictAttrConfCode(@Param("tenantCode") String tenantCode, @Param("dictTypeCode") String dictTypeCode) {
    LambdaQueryWrapper<DictAttrConfEntity> wrapper = Wrappers.lambdaQuery();
    wrapper.eq(DictAttrConfEntity::getTenantCode, tenantCode);
    wrapper.eq(DictAttrConfEntity::getDictTypeCode, dictTypeCode);
    wrapper.eq(DictAttrConfEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode());
    wrapper.orderByAsc(DictAttrConfEntity::getShowOrder);
    return this.baseMapper.selectList(wrapper);
  }

  /**
   * 根据物料编码获取详情
   *
   * @param tenantCode    租户编码
   * @param dictTypeCodes 物料编码集合
   * @return
   */
  public List<DictAttrConfEntity> findByDictTypeCodes(@Param("tenantCode") String tenantCode, @Param("dictTypeCodes") Collection<String> dictTypeCodes) {
    LambdaQueryWrapper<DictAttrConfEntity> wrapper = Wrappers.lambdaQuery();
    wrapper.eq(DictAttrConfEntity::getTenantCode, tenantCode);
    wrapper.in(DictAttrConfEntity::getDictTypeCode, dictTypeCodes);
    wrapper.eq(DictAttrConfEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode());
    wrapper.orderByAsc(DictAttrConfEntity::getShowOrder);
    return this.baseMapper.selectList(wrapper);
  }

  /**
   * 根据主键集合，修改 enable_status
   *
   * @param enable 启禁用状态
   * @param ids    ID集合
   */
  public void updateEnableStatusByIdIn(EnableStatusEnum enable, Collection<String> ids) {
    if (CollectionUtils.isEmpty(ids)) {
      return;
    }
    LambdaUpdateWrapper<DictAttrConfEntity> updateWrapper = new LambdaUpdateWrapper<>();
    updateWrapper.set(DictAttrConfEntity::getEnableStatus, enable.getCode())
        .in(DictAttrConfEntity::getId, ids)
        .eq(DictAttrConfEntity::getTenantCode, TenantUtils.getTenantCode());//设置租户编号信息
    this.baseMapper.update(null, updateWrapper);
  }

  /**
   * 根据主键集合，修改 del_flag
   *
   * @param delFlag 删除状态
   * @param ids     ID集合
   */
  public void updateDelFlagByIdIn(DelFlagStatusEnum delFlag, Collection<String> ids) {
    if (CollectionUtils.isEmpty(ids)) {
      return;
    }
    LambdaUpdateWrapper<DictAttrConfEntity> updateWrapper = new LambdaUpdateWrapper<>();
    updateWrapper.set(DictAttrConfEntity::getDelFlag, delFlag.getCode())
        .in(DictAttrConfEntity::getId, ids)
        .eq(DictAttrConfEntity::getTenantCode, TenantUtils.getTenantCode());//设置租户编号信息
    this.baseMapper.update(null, updateWrapper);
  }

  /**
   * 重构修改方法
   * @param entity
   * @param tenantCode
   */
  public void updateByIdAndTenantCode(DictAttrConfEntity entity, String tenantCode) {
    LambdaUpdateWrapper<DictAttrConfEntity>lambdaUpdateWrapper=Wrappers.lambdaUpdate();
    lambdaUpdateWrapper.eq(DictAttrConfEntity::getTenantCode,tenantCode);
    lambdaUpdateWrapper.in(DictAttrConfEntity::getId,entity.getId());
    this.baseMapper.update(entity,lambdaUpdateWrapper);
  }

  /**
   * 重构查询方法
   * @param id
   * @param tenantCode
   * @return
   */
  public DictAttrConfEntity findByIdAndTenantCode(String id, String tenantCode) {
    return this.lambdaQuery()
        .eq(DictAttrConfEntity::getTenantCode,tenantCode)
        .in(DictAttrConfEntity::getId,id)
        .one();
  }
}
