package com.biz.crm.mdm.business.dictionary.local.repository;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.dictionary.local.entity.DictAttrConfEntity;
import com.biz.crm.mdm.business.dictionary.local.entity.DictDataEntity;
import com.biz.crm.mdm.business.dictionary.local.mapper.DictDataMapper;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.ibatis.annotations.Param;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.Collection;
import java.util.List;

/**
 * 物料数据库操作层实现
 *
 * @author pengxi
 */
@Component
public class DictDataRepository extends ServiceImpl<DictDataMapper, DictDataEntity> {

  /**
   * 根据物料编码获取详情
   *
   * @param tenantCode      租户编码
   * @param parentDictCodes 父级字典编码集合
   * @return
   */
  public List<DictDataEntity> findByParentDictCodes(@Param("tenantCode") String tenantCode, @Param("parentDictCodes") Collection<String> parentDictCodes) {
    LambdaQueryWrapper<DictDataEntity> wrapper = Wrappers.lambdaQuery();
    wrapper.eq(DictDataEntity::getTenantCode, tenantCode);
    wrapper.in(DictDataEntity::getParentDictCode, parentDictCodes);
    wrapper.eq(DictDataEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode());
    wrapper.orderByAsc(DictDataEntity::getDictSort);
    return this.baseMapper.selectList(wrapper);
  }

  /**
   * 根据物料编码获取详情
   *
   * @param tenantCode    租户编码
   * @param dictTypeCodes 物料编码集合
   * @return
   */
  public List<DictDataEntity> findByDictTypeCodes(@Param("tenantCode") String tenantCode, @Param("dictTypeCodes") Collection<String> dictTypeCodes) {
    LambdaQueryWrapper<DictDataEntity> wrapper = Wrappers.lambdaQuery();
    wrapper.eq(DictDataEntity::getTenantCode, tenantCode);
    wrapper.in(DictDataEntity::getDictTypeCode, dictTypeCodes);
    wrapper.eq(DictDataEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode());
    wrapper.orderByAsc(DictDataEntity::getDictSort);
    return this.baseMapper.selectList(wrapper);
  }

  /**
   * 根据物料编码获取详情
   *
   * @param tenantCode   租户编码
   * @param dictTypeCode 字典编码类型
   * @param dictCode     字典编码
   * @return
   */
  public DictDataEntity findByDictTypeCodeAndDictCode(@Param("tenantCode") String tenantCode, @Param("dictTypeCode") String dictTypeCode, @Param("dictCode") String dictCode) {
    LambdaQueryWrapper<DictDataEntity> wrapper = Wrappers.lambdaQuery();
    wrapper.eq(DictDataEntity::getTenantCode, tenantCode);
    wrapper.eq(DictDataEntity::getDictTypeCode, dictTypeCode);
    wrapper.eq(DictDataEntity::getDictCode, dictCode);
    wrapper.eq(DictDataEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode());
    return this.baseMapper.selectOne(wrapper);
  }

  /**
   * 根据物料编码获取详情
   *
   * @param tenantCode   租户编码
   * @param dictTypeCode 字典编码类型
   * @return
   */
  public List<DictDataEntity> findByDictTypeCode(@Param("tenantCode") String tenantCode, @Param("dictTypeCode") String dictTypeCode) {
    LambdaQueryWrapper<DictDataEntity> wrapper = Wrappers.lambdaQuery();
    wrapper.eq(DictDataEntity::getTenantCode, tenantCode);
    wrapper.eq(DictDataEntity::getDictTypeCode, dictTypeCode);
    wrapper.eq(DictDataEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode());
    wrapper.orderByAsc(DictDataEntity::getDictSort);
    return this.baseMapper.selectList(wrapper);
  }

  /**
   * 根据主键集合，修改 enable_status
   *
   * @param enable 启禁用状态
   * @param ids    ID集合
   */
  public void updateEnableStatusByIdIn(EnableStatusEnum enable, Collection<String> ids) {
    if (CollectionUtils.isEmpty(ids)) {
      return;
    }
    LambdaUpdateWrapper<DictDataEntity> updateWrapper = new LambdaUpdateWrapper<>();
    updateWrapper.set(DictDataEntity::getEnableStatus, enable.getCode())
        .in(DictDataEntity::getId, ids)
        .eq(DictDataEntity::getTenantCode, TenantUtils.getTenantCode());//设置租户编号信息
    this.baseMapper.update(null, updateWrapper);
  }

  /**
   * 根据主键集合，修改 del_flag
   *
   * @param delFlag 删除标记状态
   * @param ids     ID集合
   */
  public void updateDelFlagByIdIn(DelFlagStatusEnum delFlag, Collection<String> ids) {
    if (CollectionUtils.isEmpty(ids)) {
      return;
    }
    LambdaUpdateWrapper<DictDataEntity> updateWrapper = new LambdaUpdateWrapper<>();
    updateWrapper.set(DictDataEntity::getDelFlag, delFlag.getCode())
        .in(DictDataEntity::getId, ids)
        .eq(DictDataEntity::getTenantCode, TenantUtils.getTenantCode());//设置租户编号信息
    this.baseMapper.update(null, updateWrapper);
  }

  /**
   * 重构修改方法
   * @param entity
   * @param tenantCode
   */
  public void updateByIdAndTenantCode(DictDataEntity entity, String tenantCode) {
    LambdaUpdateWrapper<DictDataEntity>lambdaUpdateWrapper=Wrappers.lambdaUpdate();
    lambdaUpdateWrapper.eq(DictDataEntity::getTenantCode,tenantCode);
    lambdaUpdateWrapper.in(DictDataEntity::getId,entity.getId());
    this.baseMapper.update(entity,lambdaUpdateWrapper);
  }

  /**
   * 重构修改方法
   * @param list
   * @param tenantCode
   */
  public void updateBatchByIdAndTenantCode(List<DictDataEntity> list, String tenantCode) {
    LambdaUpdateWrapper<DictDataEntity>lambdaUpdateWrapper=Wrappers.lambdaUpdate();
    list.stream().forEach(dictDataEntity -> {
      lambdaUpdateWrapper.eq(DictDataEntity::getTenantCode,tenantCode);
      lambdaUpdateWrapper.in(DictDataEntity::getId,dictDataEntity.getId());
      this.baseMapper.update(dictDataEntity,lambdaUpdateWrapper);
    });
  }

  /**
   * 重构查询方法
   * @param id
   * @param tenantCode
   * @return
   */
  public DictDataEntity findByIdAndTenantCode(String id, String tenantCode) {
    return this.lambdaQuery()
        .eq(DictDataEntity::getTenantCode,tenantCode)
        .in(DictDataEntity::getId,id)
        .one();
  }
}
