package com.biz.crm.mdm.business.dictionary.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.dictionary.local.entity.DictTypeEntity;
import com.biz.crm.mdm.business.dictionary.local.repository.DictTypeRepository;
import com.biz.crm.mdm.business.dictionary.sdk.constant.DictConstant;
import com.biz.crm.mdm.business.dictionary.sdk.event.DictTypeEventListener;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictAttrConfVoService;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictDataVoService;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictTypeVoService;
import com.biz.crm.mdm.business.dictionary.sdk.dto.DictTypeDto;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictTypeVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;

/**
 * 数据字典类型表接口实现
 * @author pengxi
 */
@Service
public class DictTypeVoServiceImpl implements DictTypeVoService {
  @Autowired(required = false)
  private DictTypeRepository dictTypeRepository;
  @Autowired(required = false)
  private DictAttrConfVoService dictAttrConfVoService;
  @Autowired(required = false)
  @Lazy
  private DictDataVoService dictDataVoService;
  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;
  /**
   * 字段事件监听器
   */
  @Autowired(required = false)
  @Lazy
  private List<DictTypeEventListener> dictTypeEventListeners;

  @Override
  public Page<DictTypeVo> findByConditions(Pageable pageable, DictTypeDto reqVo) {
    return dictTypeRepository.findByConditions(pageable, reqVo);
  }

  @Override
  public DictTypeVo findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    //重构查询方法
    DictTypeEntity entity = this.dictTypeRepository.findByIdAndTenantCode(id,TenantUtils.getTenantCode());
    if (entity == null) {
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(entity, DictTypeVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public DictTypeVo findByDictTypeCode(String dictTypeCode) {
    if (StringUtils.isBlank(dictTypeCode)) {
      return null;
    }
    DictTypeEntity entity = this.dictTypeRepository.findByDictTypeCode(TenantUtils.getTenantCode(), dictTypeCode);
    if (entity == null) {
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(entity, DictTypeVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  @Transactional
  public void save(DictTypeDto dto) {
    Validate.notNull(dto, "字典类型对象不能为空");
    dto.setId(null);
    Validate.notBlank(dto.getDictTypeCode(), "字典类型编码不能为空");
    Validate.isTrue(!dto.getDictTypeCode().contains(DictConstant.PATH_SPLIT), "字典类型编码不能包含字符“" + DictConstant.PATH_SPLIT + "”");
    Validate.isTrue(!dto.getDictTypeCode().contains(","), "字典类型编码不能包含字符“,”");
    Validate.notBlank(dto.getDictTypeName(), "字典类型名称不能为空");
    Validate.notBlank(dto.getDictTypeModule(), "模块不能为空");
    DictTypeEntity oldEntity = this.dictTypeRepository.findByDictTypeCode(TenantUtils.getTenantCode(), dto.getDictTypeCode());
    Validate.isTrue(Objects.isNull(oldEntity), "字典类型编码已经存在");
    DictTypeEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, DictTypeEntity.class, HashSet.class, ArrayList.class);
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setTenantCode(TenantUtils.getTenantCode());
    this.dictTypeRepository.save(entity);
  }

  @Override
  @Transactional
  public void update(DictTypeDto reqVo) {
    Validate.notNull(reqVo, "字典类型对象不能为空");
    Validate.notBlank(reqVo.getId(), "id不能为空");
    Validate.notBlank(reqVo.getDictTypeCode(), "字典类型编码不能为空");
    Validate.notBlank(reqVo.getDictTypeName(), "字典类型名称不能为空");
    Validate.notBlank(reqVo.getDictTypeModule(), "模块编码不能为空");
    DictTypeEntity oldEntity = this.dictTypeRepository.findByIdAndTenantCode(reqVo.getId(),TenantUtils.getTenantCode());
    Validate.notNull(oldEntity, "id无效");
    Validate.isTrue(oldEntity.getDictTypeCode().equals(reqVo.getDictTypeCode()), "字典类型编码不能修改");
    DictTypeEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(reqVo, DictTypeEntity.class, HashSet.class, ArrayList.class);
    //重构修改方法
    this.dictTypeRepository.updateByIdAndTenantCode(entity,TenantUtils.getTenantCode());
  }

  @Override
  @Transactional
  public void deleteBatch(List<String> ids) {
    Validate.notEmpty(ids, "请选择一条数据");
    List<DictTypeEntity> entities = this.dictTypeRepository.listByIdsAndTenantCode(ids,TenantUtils.getTenantCode());
    Validate.isTrue(CollectionUtils.isNotEmpty(entities), "已删除或不存在");
    entities.forEach(item -> {
      //删除字典扩展
      dictAttrConfVoService.deleteByDictTypeCode(item.getDictTypeCode());
      // 删除字典值
      dictDataVoService.deleteByDictTypeCode(item.getDictTypeCode());
      // TODO 删除缓存
    });
    this.dictTypeRepository.updateDelFlagByIdIn(DelFlagStatusEnum.DELETE, ids);
    // 删除时事件通知
    if (CollectionUtils.isNotEmpty(dictTypeEventListeners)) {
      List<DictTypeVo> voList = (List<DictTypeVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, DictTypeEntity.class, DictTypeVo.class, HashSet.class, ArrayList.class);
      dictTypeEventListeners.forEach(event -> event.onDelete(voList));
    }
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.notEmpty(ids, "请选择一条数据");
    List<DictTypeEntity> entities = this.dictTypeRepository.listByIdsAndTenantCode(ids,TenantUtils.getTenantCode());
    Validate.isTrue(CollectionUtils.isNotEmpty(entities), "已删除或不存在");
    this.dictTypeRepository.updateEnableStatusByIdIn(EnableStatusEnum.ENABLE, ids);
    // 启用时事件通知
    if (CollectionUtils.isNotEmpty(dictTypeEventListeners)) {
      List<DictTypeVo> voList = (List<DictTypeVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, DictTypeEntity.class, DictTypeVo.class, HashSet.class, ArrayList.class);
      dictTypeEventListeners.forEach(event -> event.onEnable(voList));
    }
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.notEmpty(ids, "请选择一条数据");
    List<DictTypeEntity> entities = this.dictTypeRepository.listByIdsAndTenantCode(ids,TenantUtils.getTenantCode());
    Validate.isTrue(CollectionUtils.isNotEmpty(entities), "已删除或不存在");
    this.dictTypeRepository.updateEnableStatusByIdIn(EnableStatusEnum.DISABLE, ids);
    // 禁用时事件通知
    if (CollectionUtils.isNotEmpty(dictTypeEventListeners)) {
      List<DictTypeVo> voList = (List<DictTypeVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, DictTypeEntity.class, DictTypeVo.class, HashSet.class, ArrayList.class);
      dictTypeEventListeners.forEach(event -> event.onDisable(voList));
    }
  }

}
