package com.biz.crm.mdm.business.fiscal.year.local.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.mdm.business.fiscal.year.local.entity.FiscalYearEntity;
import com.biz.crm.mdm.business.fiscal.year.local.service.FiscalYearService;
import com.biz.crm.mdm.business.fiscal.year.sdk.dto.FiscalYearDto;
import com.biz.crm.mdm.business.fiscal.year.sdk.dto.FiscalYearPageDto;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;

/**
 * 财年信息: FiscalYear: 财年设置
 *
 * @author ning.zhang
 * @date 2021/11/16
 */
@Slf4j
@RestController
@RequestMapping("/v1/fiscalYear/fiscalYear")
@Api(tags = "财年信息: FiscalYear: 财年设置")
public class FiscalYearController {

  @Autowired(required = false)
  private FiscalYearService fiscalYearService;

  /**
   * 新增财年信息
   * @param dto 参数dto
   * @return 新增财年数据
   */
  @ApiOperation(value = "新增财年")
  @PostMapping("")
  public Result create(@RequestBody FiscalYearDto dto) {
    try {
      return Result.ok(this.fiscalYearService.create(dto));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 更新财年信息
   * @param dto 参数dto
   * @return 更新后财年数据
   */
  @ApiOperation(value = "更新财年信息")
  @PatchMapping("")
  public Result update(@RequestBody FiscalYearDto dto) {
    try {
      return Result.ok(this.fiscalYearService.update(dto));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 删除财年
   * @param ids 财年id集合
   * @return 操作结果
   */
  @ApiOperation(value = "删除")
  @DeleteMapping("/deleteBatch")
  public Result deleteBatch(@RequestParam("ids") List<String> ids) {
    try {
      this.fiscalYearService.deleteBatch(ids);
      return Result.ok("删除成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 启用财年
   * @param ids 财年id集合
   * @return 操作结果
   */
  @ApiOperation(value = "启用")
  @PatchMapping("/enableBatch")
  public Result enableBatch(@RequestBody List<String> ids) {
    try {
      this.fiscalYearService.enableBatch(ids);
      return Result.ok("启用成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 禁用财年
   * @param ids 财年id集合
   * @return 操作结果
   */
  @ApiOperation(value = "禁用")
  @PatchMapping("/disableBatch")
  public Result disableBatch(@RequestBody List<String> ids) {
    try {
      this.fiscalYearService.disableBatch(ids);
      return Result.ok("禁用成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 财年分页查询
   * @param pageable 分页信息
   * @param dto 参数Dto
   * @return 财年分页数据
   */
  @ApiOperation(value = "财年分页查询")
  @GetMapping("/findByConditions")
  public Result<Page<FiscalYearEntity>> findByConditions(@PageableDefault(50) Pageable pageable,
                                                         @ApiParam(name = "FiscalYearPageDto", value = "分页Dto") FiscalYearPageDto dto) {
    try {
      return Result.ok(this.fiscalYearService.findByConditions(pageable, dto));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 通过ID获取财年信息
   * @param id 财年ID
   * @return 操作结果
   */
  @ApiOperation(value = "通过ID获取财年信息")
  @GetMapping("/findById")
  public Result<FiscalYearEntity> findById(@RequestParam(value = "id", required = false) @ApiParam(name = "id", value = "主键ID") String id) {
    try {
      return Result.ok(this.fiscalYearService.findById(id));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

}
