package com.biz.crm.mdm.business.material.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictToolkitService;
import com.biz.crm.mdm.business.material.local.entity.MaterialEntity;
import com.biz.crm.mdm.business.material.local.entity.MaterialMediaEntity;
import com.biz.crm.mdm.business.material.local.entity.MaterialRelationWarehouseEntity;
import com.biz.crm.mdm.business.material.local.repository.MaterialMediaRepository;
import com.biz.crm.mdm.business.material.local.repository.MaterialRelationWarehouseRepository;
import com.biz.crm.mdm.business.material.local.repository.MaterialRepository;
import com.biz.crm.mdm.business.material.local.service.MaterialMediaService;
import com.biz.crm.mdm.business.material.sdk.constant.MaterialConstant;
import com.biz.crm.mdm.business.material.sdk.dto.MaterialDto;
import com.biz.crm.mdm.business.material.sdk.dto.MaterialPageDto;
import com.biz.crm.mdm.business.material.sdk.dto.MaterialRelationWarehouseDto;
import com.biz.crm.mdm.business.material.sdk.dto.MaterialSearchDto;
import com.biz.crm.mdm.business.material.sdk.event.MaterialEventListener;
import com.biz.crm.mdm.business.material.sdk.service.MaterialVoService;
import com.biz.crm.mdm.business.material.sdk.vo.MaterialEventVo;
import com.biz.crm.mdm.business.material.sdk.vo.MaterialMediaVo;
import com.biz.crm.mdm.business.material.sdk.vo.MaterialRelationWarehouseVo;
import com.biz.crm.mdm.business.material.sdk.vo.MaterialVo;
import com.biz.crm.mdm.business.material.unit.dto.MaterialUnitTypeDto;
import com.biz.crm.mdm.business.material.unit.service.MaterialUnitTypeVoService;
import com.biz.crm.mdm.business.material.unit.vo.MaterialUnitTypeVo;
import com.biz.crm.mdm.business.product.level.sdk.service.ProductLevelVoSdkService;
import com.biz.crm.mdm.business.product.level.sdk.vo.ProductLevelVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 物料接口实现
 *
 * @author pengxi
 * @date 2021-09-27 14:44:10
 */
@Service
public class MaterialVoServiceImpl implements MaterialVoService {
  @Autowired(required = false)
  private MaterialRepository materialRepository;
  @Autowired(required = false)
  private GenerateCodeService generateCodeService;
  @Autowired(required = false)
  private ProductLevelVoSdkService productLevelVoSdkService;
  @Autowired(required = false)
  private MaterialUnitTypeVoService materialUnitTypeVoService;
  @Autowired(required = false)
  private DictToolkitService dictToolkitService;
  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private MaterialMediaService materialMediaService;
  @Autowired(required = false)
  private MaterialMediaRepository materialMediaRepository;
  @Autowired(required = false)
  private MaterialRelationWarehouseRepository materialRelationWarehouseRepository;

  /**
   * 物料事件监听器
   */
  @Autowired(required = false)
  @Lazy
  private List<MaterialEventListener> materialEventListeners;

  @Override
  public List<MaterialVo> findAll() {
    List<MaterialEntity> entityList = materialRepository.findAll();
    return (List<MaterialVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entityList, MaterialEntity.class,
      MaterialVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public Page<MaterialVo> findByConditions(Pageable pageable, MaterialDto dto) {
    dto = Optional.ofNullable(dto).orElse(new MaterialDto());
    // 根据产品层级编码获取产品层级子级编码集合做物料范围过滤
    if (StringUtils.isNotBlank(dto.getProductLevelCode())) {
      List<String> productLevelCodes =
          this.productLevelVoSdkService.findCurAndChildrenCodesByCodes(Sets.newHashSet(dto.getProductLevelCode()));
      dto.setProductLevelCodes(productLevelCodes);
    }
    Page<MaterialVo> materialVoPage = this.materialRepository.findByConditions(pageable, dto);
    if (materialVoPage.getCurrent()==0) {
      return materialVoPage;
    }
    // 返回值需要获取产品层级信息
    List<String> productLevelCodes = materialVoPage.getRecords().stream().map(MaterialVo::getProductLevelCode).distinct().collect(Collectors.toList());
    List<ProductLevelVo> productLevelVos = Lists.newArrayList();
    if (CollectionUtils.isNotEmpty(productLevelCodes)) {
      productLevelVos = this.productLevelVoSdkService.findListByCodes(productLevelCodes);
    }
    // 获取物料图片
    List<String> materialIds = materialVoPage.getRecords().stream().map(MaterialVo::getId).collect(Collectors.toList());
    List<MaterialMediaEntity> mediaEntities = Lists.newArrayList();
    if(CollectionUtils.isNotEmpty(materialIds)) {
      mediaEntities = this.materialMediaRepository.findByMaterialIds(materialIds);
    }
    Map<String, List<MaterialMediaVo>> mediaMap = Maps.newHashMap();
    if (CollectionUtils.isNotEmpty(mediaEntities)) {
      mediaMap = mediaEntities.stream().collect(Collectors.groupingBy(MaterialMediaEntity::getMaterialId, Collectors.mapping(o ->
          this.nebulaToolkitService.copyObjectByWhiteList(o, MaterialMediaVo.class, HashSet.class, ArrayList.class),Collectors.toList())));
    }
    // 按产品编码作为key转成产品层级map
    Map<String, ProductLevelVo> productLevelMap = productLevelVos.stream().collect(Collectors.toMap(ProductLevelVo::getProductLevelCode, Function.identity()));
    Map<String, String> materialType = this.dictToolkitService.findMapByDictTypeCode(MaterialConstant.MATERIAL_TYPE);
    Map<String, List<MaterialMediaVo>> finalMediaMap = mediaMap;
    materialVoPage.getRecords().forEach(vo->{
      // 物料类型名称
      if (MapUtils.isNotEmpty(materialType) && StringUtils.isNotBlank(vo.getMaterialType()) &&  materialType.containsKey(vo.getMaterialType())) {
        vo.setMaterialTypeName(materialType.get(vo.getMaterialType()));
      }
      EnableStatusEnum enableStatusEnum = EnableStatusEnum.codeToEnum(vo.getEnableStatus());
      if (enableStatusEnum != null) {
        vo.setEnableStatusName(enableStatusEnum.getDes());
      }
      // 解析销售公司
      this.analysisSaleCompanyName(vo);
      // 解析物料文件信息
      vo.setPicList(finalMediaMap.get(vo.getId()));
      // 获取产品层级名称
      if (StringUtils.isNotBlank(vo.getProductLevelCode()) && productLevelMap.containsKey(vo.getProductLevelCode())) {
        vo.setProductLevelName(productLevelMap.get(vo.getProductLevelCode()).getProductLevelName());
      }
    });
    return materialVoPage;
  }

  /**
   * 根据销售公司编码解析成集合
   * 注意saleCompany的值格式为“a,b,c”
   * */
  private void analysisSaleCompanyName(MaterialVo vo) {
    if (StringUtils.isBlank(vo.getSaleCompany())) {
      return;
    }
    //解析封装成集合
    vo.setSaleCompanyList(Lists.newArrayList(vo.getSaleCompany().split(",")));
  }

  /**
   * 根据产品层级编码获取产品层级名称
   * */
  private void analysisProductLevel(MaterialVo vo) {
    if (StringUtils.isBlank(vo.getProductLevelCode())) {
      return;
    }
    ProductLevelVo productLevelVo = this.productLevelVoSdkService.findDetailsByCode(vo.getProductLevelCode());
    if (productLevelVo == null) {
      return;
    }
    vo.setProductLevelName(productLevelVo.getProductLevelName());
  }

  /**
   * 解析物料文件信息
   * */
  private void analysisMaterialMedia(MaterialVo vo) {
    if (StringUtils.isBlank(vo.getId())) {
      return;
    }
    List<MaterialMediaEntity> materialMediaEntities = this.materialMediaRepository.findByMaterialIds(Collections.singletonList(vo.getId()));
    if (CollectionUtils.isNotEmpty(materialMediaEntities)) {
      List<MaterialMediaVo> materialMediaVoList = Lists.newArrayList(this.nebulaToolkitService.copyCollectionByWhiteList(materialMediaEntities, MaterialMediaEntity.class, MaterialMediaVo.class, HashSet.class, ArrayList.class));
      vo.setPicList(materialMediaVoList);
    }
  }

  private void analysisMaterialRelationWarehouse(MaterialVo vo){
    if (StringUtils.isBlank(vo.getId())) {
      return;
    }
    List<MaterialRelationWarehouseEntity> warehouseEntities = this.materialRelationWarehouseRepository.listByMaterialsId(vo.getId());
    if (CollectionUtils.isNotEmpty(warehouseEntities)){
      Collection<MaterialRelationWarehouseVo> vos = this.nebulaToolkitService.copyCollectionByBlankList(warehouseEntities, MaterialRelationWarehouseEntity.class, MaterialRelationWarehouseVo.class, HashSet.class, ArrayList.class);
      vo.setWarehouseVos(new ArrayList<>(vos));
    }
  }

  @Override
  public MaterialVo findDetailById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    MaterialEntity entity = this.materialRepository.findById(id);
    if (entity == null) {
      return null;
    }
    MaterialVo materialVo = this.nebulaToolkitService.copyObjectByBlankList(entity, MaterialVo.class, HashSet.class, ArrayList.class);
    // 解析销售公司
    this.analysisSaleCompanyName(materialVo);
    // 解析产品层级
    this.analysisProductLevel(materialVo);
    // 解析物料文件信息
    this.analysisMaterialMedia(materialVo);
    //解析物料关联仓库信息
    this.analysisMaterialRelationWarehouse(materialVo);
    //物料单位类型
    MaterialUnitTypeDto materialUnitTypeDto = new MaterialUnitTypeDto();
    materialUnitTypeDto.setUnitTypeCodes(Collections.singletonList(materialVo.getUnitTypeCode()));
    List<MaterialUnitTypeVo> materialUnitTypeVos = materialUnitTypeVoService.findMaterialUnitTypeByConditions(materialUnitTypeDto);
    if (CollectionUtils.isEmpty(materialUnitTypeVos)) {
      return materialVo;
    }
    materialVo.setMaterialUnitTypeVo(materialUnitTypeVos.get(0));
    return materialVo;
  }

  @Override
  public MaterialVo findDetailByMaterialCode(String materialCode) {
    if (StringUtils.isBlank(materialCode)) {
      return null;
    }
    MaterialEntity entity = this.materialRepository.findDetailByMaterialCode(TenantUtils.getTenantCode(), materialCode);
    if (entity == null) {
      return null;
    }
    MaterialVo materialVo = this.nebulaToolkitService.copyObjectByWhiteList(entity, MaterialVo.class, HashSet.class, ArrayList.class);
    // 解析销售公司
    this.analysisSaleCompanyName(materialVo);
    // 解析产品层级
    this.analysisProductLevel(materialVo);
    // 解析物料文件信息
    this.analysisMaterialMedia(materialVo);
    //解析物料关联仓库信息
    this.analysisMaterialRelationWarehouse(materialVo);
    //物料单位类型
    MaterialUnitTypeDto materialUnitTypeDto = new MaterialUnitTypeDto();
    materialUnitTypeDto.setUnitTypeCodes(Collections.singletonList(materialVo.getUnitTypeCode()));
    List<MaterialUnitTypeVo> materialUnitTypeVos = materialUnitTypeVoService.findMaterialUnitTypeByConditions(materialUnitTypeDto);
    if (CollectionUtils.isEmpty(materialUnitTypeVos)) {
      return materialVo;
    }
    materialVo.setMaterialUnitTypeVo(materialUnitTypeVos.get(0));
    return materialVo;
  }

  @Override
  public List<MaterialVo> findDetailByMaterialCodes(Set<String> materialCodes) {
    if (CollectionUtils.isEmpty(materialCodes)) {
      return Lists.newArrayList();
    }
    List<MaterialEntity> entities = this.materialRepository.findDetailByMaterialCodes(TenantUtils.getTenantCode(), materialCodes);
    if (CollectionUtils.isEmpty(entities)) {
      return null;
    }
    Map<String, String> materialType = this.dictToolkitService.findMapByDictTypeCode(MaterialConstant.MATERIAL_TYPE);
    // k-productLevelCode,v-productLevelName
    Map<String, String> mapLevel = Maps.newHashMap();
    Set<String> productLevelCodeSet =
        entities.stream()
            .filter(a -> StringUtils.isNotBlank(a.getProductLevelCode()))
            .map(MaterialEntity::getProductLevelCode)
            .collect(Collectors.toSet());
    if (CollectionUtils.isNotEmpty(productLevelCodeSet)) {
      List<ProductLevelVo> levelVoList =
          this.productLevelVoSdkService.findListByCodes(Lists.newArrayList(productLevelCodeSet));
      if (CollectionUtils.isNotEmpty(levelVoList)) {
        mapLevel =
            levelVoList.stream()
                .filter(
                    a -> StringUtils.isNoneBlank(a.getProductLevelCode(), a.getProductLevelName()))
                .collect(
                    Collectors.toMap(
                        ProductLevelVo::getProductLevelCode,
                        ProductLevelVo::getProductLevelName,
                        (a, b) -> a));
      }
    }
    List<MaterialVo> voList = (List<MaterialVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, MaterialEntity.class, MaterialVo.class, HashSet.class, ArrayList.class);
    // 完善物料单位信息
    List<String> unitTypeCodes = voList.stream().filter(o->StringUtils.isNotBlank(o.getUnitTypeCode())).map(MaterialVo::getUnitTypeCode).distinct().collect(Collectors.toList());
    MaterialUnitTypeDto materialUnitTypeDto = new MaterialUnitTypeDto();
    materialUnitTypeDto.setUnitTypeCodes(unitTypeCodes);
    List<MaterialUnitTypeVo> materialUnitTypeVos = materialUnitTypeVoService.findMaterialUnitTypeByConditions(materialUnitTypeDto);
    if (CollectionUtils.isEmpty(materialUnitTypeVos)) {
      return voList;
    }
    // unitTypeCode 能确保唯一
    Map<String, MaterialUnitTypeVo> materialUnitTypeVoMap = materialUnitTypeVos.stream().collect(Collectors.toMap(MaterialUnitTypeVo::getUnitTypeCode, Function.identity()));
    for (MaterialVo o : voList) {
      if (StringUtils.isNotBlank(o.getUnitTypeCode()) && materialUnitTypeVoMap.containsKey(o.getUnitTypeCode())) {
        o.setMaterialUnitTypeVo(materialUnitTypeVoMap.get(o.getUnitTypeCode()));
      }
      o.setProductLevelName(mapLevel.get(o.getProductLevelCode()));
      o.setMaterialTypeName(materialType.get(o.getMaterialType()));
    }
    return voList;
  }

  /**
   * 把销售公司编码集合转成英文逗号拼接的字符串保存
   * 注意拼接后saleCompany值格式为“a,b,c”
   * */
  private void analysisSaleCompanyCode(MaterialDto dto) {
    if (CollectionUtils.isEmpty(dto.getSaleCompanyList())) {
      return;
    }
    // 销售公司（前端页面多选）
    String connector = ",";
    dto.setSaleCompany(String.join(connector, dto.getSaleCompanyList()));
  }

  @Override
  @Transactional
  public void save(MaterialDto materialDto) {
    Validate.notNull(materialDto, "进行当前操作时，信息对象必须传入!!");
    Validate.notEmpty(materialDto.getMaterialName(), "添加信息时，物料名称不能为空！");
    //Validate.notEmpty(materialDto.getUnitTypeCode(), "添加信息时，物料单位类型编码不能为空！");
    // TODO 成本价格式不正确：请输入正数，保留两位小数
    if (StringUtils.isBlank(materialDto.getMaterialCode())) {
      // redis生物料编码code
      List<String> codeList = this.generateCodeService.generateCode(MaterialConstant.MATERIAL_CODE, 1);
      Validate.isTrue(CollectionUtils.isNotEmpty(codeList), "添加信息时，生成物料编码失败！");
      materialDto.setMaterialCode(codeList.get(0));
      materialDto.setTenantCode(TenantUtils.getTenantCode());
    } else {
      MaterialEntity dbEntity = this.materialRepository.findDetailByMaterialCode(TenantUtils.getTenantCode(), materialDto.getMaterialCode());
      Validate.isTrue(Objects.isNull(dbEntity), "添加信息时，物料编码已存在！");
    }
    // 把销售公司编码集合转成英文逗号拼接的字符串保存
    this.analysisSaleCompanyCode(materialDto);
    MaterialEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(materialDto, MaterialEntity.class, HashSet.class, ArrayList.class);
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    //新增租户编号
    entity.setTenantCode(TenantUtils.getTenantCode());
    this.materialRepository.save(entity);

    //关联仓库信息
    List<MaterialRelationWarehouseDto> warehouseDtos = materialDto.getWarehouseDtos();
    if (CollectionUtils.isNotEmpty(warehouseDtos)){
      warehouseDtos.forEach(o -> {
        o.setMaterialId(entity.getId());
        o.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        o.setTenantCode(TenantUtils.getTenantCode());
      });

      Collection<MaterialRelationWarehouseEntity> warehouseEntities = this.nebulaToolkitService.copyCollectionByWhiteList(warehouseDtos, MaterialRelationWarehouseDto.class, MaterialRelationWarehouseEntity.class, HashSet.class, ArrayList.class);
      this.materialRelationWarehouseRepository.saveBatch(warehouseEntities);
    }

    //更新物料图片信息
    this.materialMediaService.update(materialDto.getPicList(),entity.getId());
    //创建日志
    MaterialVo materialVo = this.nebulaToolkitService.copyObjectByWhiteList(entity, MaterialVo.class, HashSet.class, ArrayList.class);
    MaterialEventVo materialEventVo = new MaterialEventVo();
    materialEventVo.setNewVo(materialVo);
    materialEventVo.setOldVo(null);
    this.materialEventListeners.forEach(event -> event.onCreate(materialEventVo));
  }

  @Override
  @Transactional
  public void update(MaterialDto materialDto) {
    Validate.notNull(materialDto, "进行当前操作时，信息对象必须传入!!");
    Validate.notEmpty(materialDto.getId(), "修改信息时，id不能为空！");
    Validate.notEmpty(materialDto.getMaterialName(), "修改信息时，物料名称不能为空！");
    Validate.notEmpty(materialDto.getUnitTypeCode(), "添加信息时，物料单位类型编码不能为空！");
    final MaterialEntity oldEntity = this.materialRepository.findByIdAndTenantCode(materialDto.getId(),TenantUtils.getTenantCode());   //重构查询方法
    Validate.notNull(oldEntity, "不存在或已删除");
    // 把销售公司编码集合转成英文逗号拼接的字符串保存
    this.analysisSaleCompanyCode(materialDto);
    MaterialEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(materialDto, MaterialEntity.class, HashSet.class, ArrayList.class);
    //重构修改的方法
    this.materialRepository.updateByIdAndTenantCode(entity,TenantUtils.getTenantCode());
    //更新物料图片信息
    this.materialMediaService.update(materialDto.getPicList(),entity.getId());

    //关联仓库信息
    List<MaterialRelationWarehouseDto> warehouseDtos = materialDto.getWarehouseDtos();
    if (CollectionUtils.isNotEmpty(warehouseDtos)){

      this.materialRelationWarehouseRepository.removeByMaterialsId(entity.getId());

      warehouseDtos.forEach(o -> {
        o.setMaterialId(entity.getId());
        o.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        o.setTenantCode(TenantUtils.getTenantCode());
      });

      Collection<MaterialRelationWarehouseEntity> warehouseEntities = this.nebulaToolkitService.copyCollectionByWhiteList(warehouseDtos, MaterialRelationWarehouseDto.class, MaterialRelationWarehouseEntity.class, HashSet.class, ArrayList.class);
      this.materialRelationWarehouseRepository.saveBatch(warehouseEntities);
    }

    // 更新物料时通知商品及商品的所有关联主数据信息做相应逻辑处理
    if (CollectionUtils.isNotEmpty(materialEventListeners)) {
      MaterialVo oldVo = this.nebulaToolkitService.copyObjectByWhiteList(oldEntity, MaterialVo.class, HashSet.class, ArrayList.class);
      MaterialVo materialVo = this.nebulaToolkitService.copyObjectByWhiteList(entity, MaterialVo.class, HashSet.class, ArrayList.class);
      MaterialEventVo materialEventVo = new MaterialEventVo();
      materialEventVo.setNewVo(materialVo);
      materialEventVo.setOldVo(oldVo);
      this.materialEventListeners.forEach(event -> event.onChange(materialEventVo));
    }
  }

  @Override
  @Transactional
  public void deleteBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "请选中要操作的数据");
    List<MaterialEntity> entities = this.materialRepository.findByIds(ids);
    Validate.isTrue(CollectionUtils.isNotEmpty(entities), "不存在或已删除！");
    this.materialRepository.updateDelFlagByIdIn(DelFlagStatusEnum.DELETE, ids);
    this.materialRelationWarehouseRepository.removeByMaterialsIds(ids);
    // 删除物料时通知商品及商品的所有关联主数据信息做相应逻辑处理
    if (CollectionUtils.isNotEmpty(materialEventListeners)) {
      List<MaterialVo> voList = (List<MaterialVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, MaterialEntity.class, MaterialVo.class, HashSet.class, ArrayList.class);
      MaterialEventVo materialEventVo = new MaterialEventVo();
      materialEventVo.setMaterialVos(voList);
      this.materialEventListeners.forEach(event -> event.onDelete(materialEventVo));
    }
  }

  @Autowired(required = false)
  private NebulaNetEventClient nebulaNetEventClient;

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "请选中要操作的数据");
    List<MaterialEntity> entities = this.materialRepository.findByIds(ids);
    Validate.isTrue(CollectionUtils.isNotEmpty(entities), "不存在或已删除！");
    this.materialRepository.updateEnableStatusByIdIn(EnableStatusEnum.ENABLE, ids);
    // 启用物料时通知商品及商品的所有关联主数据信息做相应逻辑处理
    List<MaterialVo> voList = (List<MaterialVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, MaterialEntity.class, MaterialVo.class, HashSet.class, ArrayList.class);
    MaterialEventVo materialEventVo = new MaterialEventVo();
    materialEventVo.setMaterialVos(voList);
    SerializableBiConsumer<MaterialEventListener, MaterialEventVo> sf = MaterialEventListener::onEnable;
    nebulaNetEventClient.publish(materialEventVo, MaterialEventListener.class, sf);
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "请选中要操作的数据");
    List<MaterialEntity> entities = this.materialRepository.findByIds(ids);
    Validate.isTrue(CollectionUtils.isNotEmpty(entities), "不存在或已删除！");
    this.materialRepository.updateEnableStatusByIdIn(EnableStatusEnum.DISABLE, ids);
    // 禁用物料时通知商品及商品的所有关联主数据信息做相应逻辑处理
    if (CollectionUtils.isNotEmpty(materialEventListeners)) {
      List<MaterialVo> voList = (List<MaterialVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, MaterialEntity.class, MaterialVo.class, HashSet.class, ArrayList.class);
      MaterialEventVo materialEventVo = new MaterialEventVo();
      materialEventVo.setMaterialVos(voList);
      SerializableBiConsumer<MaterialEventListener, MaterialEventVo> disable = MaterialEventListener::onDisable;
      nebulaNetEventClient.publish(materialEventVo, MaterialEventListener.class, disable);
    }
  }

  @Override
  public Page<MaterialVo> findByMaterialPageDto(Pageable pageable, MaterialPageDto dto) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    dto = ObjectUtils.defaultIfNull(dto, new MaterialPageDto());
    dto.setTenantCode(TenantUtils.getTenantCode());
    List<String> productLevelCodes = Lists.newArrayList();
    if (CollectionUtils.isNotEmpty(dto.getProductLevelCodes())) {
      productLevelCodes = dto.getProductLevelCodes();
    }
    //封装商品层级类类型对应的产品层级编码
    if (StringUtils.isNotBlank(dto.getProductLevelType())) {
      List<ProductLevelVo> levelVoList = this.productLevelVoSdkService.findByProductLevelType(dto.getProductLevelType());
      if (CollectionUtils.isNotEmpty(levelVoList)) {
        productLevelCodes.addAll(levelVoList.stream().map(ProductLevelVo::getProductLevelCode).collect(Collectors.toList()));
      }
    }
    dto.setProductLevelCodes(productLevelCodes);
    Page<MaterialEntity> entityPage = this.materialRepository.findByMaterialPageDto(pageable, dto);
    Page<MaterialVo> pageResult = new Page<>(entityPage.getCurrent(), entityPage.getSize(), entityPage.getTotal());
    List<MaterialEntity> entities = entityPage.getRecords();
    if (CollectionUtils.isEmpty(entityPage.getRecords())) {
      return pageResult;
    }
    pageResult.setRecords(this.covertEntityToVo(entities));
    return pageResult;
  }

  @Override
  public Set<String> findCodeByProductLevelCodes(Set<String> productLevelCodeSet) {
    if(CollectionUtils.isEmpty(productLevelCodeSet)) {
      return Sets.newHashSet();
    }
    return this.materialRepository.findCodeByProductLevelCodes(productLevelCodeSet);
  }

  @Override
  public List<MaterialVo> findMaterialByIds(MaterialSearchDto dto) {
    List<String> ids = dto.getIds();
    Validate.isTrue(CollectionUtils.isNotEmpty(ids),"传入的id集合不能为空");
    List<MaterialEntity> list = this.materialRepository.lambdaQuery()
            .in(MaterialEntity::getId, ids)
            .eq(MaterialEntity::getTenantCode, TenantUtils.getTenantCode())
            .eq(MaterialEntity::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
            .list();
    return this.covertEntityToVo(list);
  }

  /**
   * 转换物料实体为vo
   *
   * @param entities 物料实体
   * @return 物料vo
   */
  private List<MaterialVo> covertEntityToVo(List<MaterialEntity> entities) {
    List<String> levelCodes = Lists.newArrayList();
    List<String> materialIds = Lists.newArrayList();
    entities.forEach(materialEntity -> {
      if (StringUtils.isNotBlank(materialEntity.getProductLevelCode())) {
        levelCodes.add(materialEntity.getProductLevelCode());
      }
      materialIds.add(materialEntity.getId());
    });
    List<MaterialMediaEntity> mediaEntities = this.materialMediaRepository.findByMaterialIds(materialIds);
    Map<String, List<MaterialMediaVo>> mediaMap = Maps.newHashMap();
    if (CollectionUtils.isNotEmpty(mediaEntities)) {
      mediaMap = mediaEntities.stream().collect(Collectors.groupingBy(MaterialMediaEntity::getMaterialId, Collectors.mapping(o ->
              this.nebulaToolkitService.copyObjectByWhiteList(o, MaterialMediaVo.class, HashSet.class, ArrayList.class),Collectors.toList())));
    }
    Map<String, List<ProductLevelVo>> levelMap = this.productLevelVoSdkService.findCurAndParentByCodes(levelCodes);
    List<MaterialVo> list = Lists.newArrayList();
    for (MaterialEntity materialEntity : entities) {
      MaterialVo vo = this.nebulaToolkitService.copyObjectByWhiteList(materialEntity, MaterialVo.class, HashSet.class, ArrayList.class);
      // 解析销售公司
      this.analysisSaleCompanyName(vo);
      // 解析产品层级信息
      List<ProductLevelVo> productLevelVos = levelMap.get(materialEntity.getProductLevelCode());
      if (!CollectionUtils.isEmpty(productLevelVos)) {
        vo.setProductLevels(productLevelVos);
        productLevelVos.forEach(productLevelVo -> {
          if (productLevelVo.getProductLevelCode().equals(materialEntity.getProductLevelCode())) {
            vo.setProductLevelName(productLevelVo.getProductLevelName());
          }
        });
      }
      //解析物料文件信息
      vo.setPicList(mediaMap.get(materialEntity.getId()));
      list.add(vo);
    }
    return list;
  }

}
