package com.biz.crm.mdm.business.material.unit.service.internal;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mdm.business.dictionary.sdk.constant.DictConstant;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictToolkitService;
import com.biz.crm.mdm.business.material.unit.constant.MaterialUnitConstant;
import com.biz.crm.mdm.business.material.unit.dto.MaterialUnitDto;
import com.biz.crm.mdm.business.material.unit.entity.MaterialUnit;
import com.biz.crm.mdm.business.material.unit.entity.MaterialUnitType;
import com.biz.crm.mdm.business.material.unit.repository.MaterialUnitRepository;
import com.biz.crm.mdm.business.material.unit.repository.MaterialUnitTypeRepository;
import com.biz.crm.mdm.business.material.unit.service.MaterialUnitService;
import com.biz.crm.mn.common.base.constant.CommonConstant;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.constants.MasterDataMdgConstants;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.dto.MasterDataMdgBaseDto;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.service.MasterDataMdgService;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.vo.MasterDataMdgMaterialUnitVo;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author jerry7 物料单位service实现
 */
@Slf4j
@Service
public class MaterialUnitServiceImpl implements MaterialUnitService {

    @Autowired(required = false)
    private MaterialUnitRepository materialUnitRepository;

    @Autowired(required = false)
    private MaterialUnitTypeRepository materialUnitTypeRepository;

    @Autowired(required = false)
    private LoginUserService loginUserService;

    @Autowired(required = false)
    private RedisMutexService redisMutexService;

    @Autowired(required = false)
    private MasterDataMdgService masterDataMdgService;

    @Autowired(required = false)
    private DictToolkitService dictToolkitService;

    @Override
    public Page<MaterialUnit> findByConditions(Pageable pageable, MaterialUnitDto dto) {
        pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
        dto = Optional.ofNullable(dto).orElse(new MaterialUnitDto());
        dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        Page<MaterialUnit> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return materialUnitRepository.findByConditions(page, dto);
    }

    @Override
    @Transactional
    public void enableBatch(List<String> ids) {
        Validate.isTrue(!CollectionUtils.isEmpty(ids), "待修正的数据主键不能为空");
        materialUnitRepository.updateEnableStatusByIdIn(EnableStatusEnum.ENABLE, ids);
    }

    @Override
    @Transactional
    public void disableBatch(List<String> ids) {
        Validate.isTrue(!CollectionUtils.isEmpty(ids), "待修正的数据主键不能为空");
        materialUnitRepository.updateEnableStatusByIdIn(EnableStatusEnum.DISABLE, ids);
    }

    @Override
    @Transactional
    public void deleteBatch(List<String> ids) {
        Validate.isTrue(!CollectionUtils.isEmpty(ids), "待修正的数据主键不能为空");
        materialUnitRepository.updateDelStatusByIdIn(DelFlagStatusEnum.DELETE, ids);
    }

    @Override
    public void deleteByUnitTypeCode(String unitTypeCode, List<String> unitCodes) {
        Validate.notNull(unitTypeCode, "待修正的物料单位类别不能为空");
        materialUnitRepository.updateDelStatusByUnitType(DelFlagStatusEnum.DELETE, unitTypeCode, unitCodes);
    }

    @Override
    public MaterialUnit findById(String id) {
        if (StringUtils.isBlank(id)) {
            return null;
        }
        // 重构查询方法
        return materialUnitRepository.findByIdAndTenantCode(id, TenantUtils.getTenantCode());
    }

    @Override
    public List<MaterialUnit> findByUnitTypeCode(String unitTypeCode) {
        return this.materialUnitRepository.findByUnitTypeCode(unitTypeCode);
    }

    @Override
    @Transactional
    public MaterialUnit create(MaterialUnit materialUnit) {
        Validate.notNull(materialUnit, "创建登录配置实体不能为空！");
        this.createValidate(materialUnit);
        // 设置基本数据
        materialUnit.setCreateTime(new Date());
        if (StringUtils.isBlank(materialUnit.getEnableStatus())) {
            materialUnit.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        }
        materialUnit.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        materialUnit.setTenantCode(TenantUtils.getTenantCode());
        materialUnit.setCreateAccount(loginUserService.getLoginAccountName());
        this.materialUnitRepository.saveOrUpdate(materialUnit);
        // 如果扩展配置不为空，添加配置扩展
        return materialUnit;
    }

    @Override
    @Transactional
    public MaterialUnit update(MaterialUnit materialUnit) {
        Validate.notNull(materialUnit, "更新登录配置实体不能为空！");
        Validate.notBlank(materialUnit.getId(), "登录登录配置ID不能为空！");
        this.createValidate(materialUnit);
        materialUnit.setModifyTime(new Date());
        materialUnit.setModifyAccount(loginUserService.getLoginAccountName());
        // 新增租户编号
        materialUnit.setTenantCode(TenantUtils.getTenantCode());
        this.materialUnitRepository.saveOrUpdate(materialUnit);
        return materialUnit;
    }

    @Override
    public List<MaterialUnit> listByUnitTypeCodes(List<String> unitTypeCodeList) {
        if (CollectionUtils.isEmpty(unitTypeCodeList)) {
            return null;
        }
        return this.materialUnitRepository.listByUnitTypeCodes(unitTypeCodeList);
    }

    /**
     * 批量拉取 MDG 物料单位数据
     *
     * @param dto
     * @author huojia
     * @date 2022/12/5 21:37
     **/
    @Override
    public void pullMaterialUnitList(MasterDataMdgBaseDto dto) {
        if (ObjectUtils.isEmpty(dto)) {
            dto = new MasterDataMdgBaseDto();
        }
        if (StringUtils.isEmpty(dto.getPageNum())) {
            dto.setPageNum("1");
        }
        if (StringUtils.isEmpty(dto.getPageSize())) {
            dto.setPageSize(CommonConstant.MAX_PAGE_SIZE_STR);
        }
        String lockKey = DateUtil.format(new Date(), DateUtil.DEFAULT_YEAR_MONTH_DAY_NO_CH);
        if (StringUtils.isEmpty(lockKey)) {
            dto.setDs(lockKey);
        }
        boolean lock = true;
        try {
            lock = this.lock(lockKey);
            if (!lock) {
                return;
            }
            List<MaterialUnit> materialUnitList = new ArrayList<>();
            List<MaterialUnitType> materialUnitTypeList = new ArrayList<>();
            List<MaterialUnit> saveListMaterialUnit = new ArrayList<>();
            List<MaterialUnit> updateListMaterialUnit = new ArrayList<>();
            // 区分更新、新增
            List<MaterialUnitType> saveList = new ArrayList<>();
            List<MaterialUnitType> updateList = new ArrayList<>();
            List<String> unitTypeCode = new ArrayList<>();
            List<MasterDataMdgMaterialUnitVo> masterDataMdgMaterialUnitVos = new ArrayList<>();
            List<MaterialUnitType> byUnitTypeCodes = new ArrayList<>();
            List<MaterialUnit> byUnitCode = new ArrayList<>();
            Map<String, MaterialUnitType> materialUnitTypeMap = new HashMap<>();
            Map<String, MaterialUnit> materialUnitMap = new HashMap<>();
            List<MasterDataMdgMaterialUnitVo> failList = new ArrayList<>();
            Map<String, String> codeMap = new HashMap<>(10000);
            AtomicReference<MaterialUnitType> materialUnitTypeEntity = new AtomicReference<>(new MaterialUnitType());
            int maxPage = 0;
            if (BooleanEnum.TRUE.getCapital().equals(dto.getFullPullFlag())) {
                maxPage = this.countPageMax(dto);
            } else {
                maxPage = Integer.parseInt(dto.getPageNum());
            }
            //物料基本单位翻转
            Map<String, String> unitConvertMap = dictToolkitService.findConvertMapByDictTypeCode(DictConstant.MATERIAL_BASE_UNIT);
            for (int pageNum = Integer.parseInt(dto.getPageNum()); pageNum <= maxPage; pageNum++) {
                dto.setPageNum(Integer.toString(pageNum));
                masterDataMdgMaterialUnitVos = masterDataMdgService.pullMaterialUnitList(dto);
                if (CollectionUtils.isEmpty(masterDataMdgMaterialUnitVos)) {
                    return;
                }
                // 数据校验
                this.unitCodeValidate(masterDataMdgMaterialUnitVos, materialUnitList, materialUnitTypeList, codeMap, unitConvertMap);
                log.info("当前拉取进度：" + pageNum + "/" + maxPage + "，每页" + dto.getPageSize() + " 条数据，拉取失败的数据共"
                        + failList.size() + "条，---------------------------->" + JSON.toJSONString(failList));
                failList.clear();
                unitTypeCode =
                        materialUnitTypeList.stream().map(MaterialUnitType::getUnitTypeCode).collect(Collectors.toList());
                // 区分更新、新增操作
                byUnitTypeCodes = materialUnitTypeRepository.findByUnitTypeCodes(unitTypeCode);
                // 区分更新、新增操作
                byUnitCode = materialUnitRepository.findByUnitCodes(unitTypeCode);
                if (CollectionUtils.isEmpty(byUnitTypeCodes)) {
                    this.saveOrUpdateBatchMaterialUnitType(materialUnitTypeList, null);
                    materialUnitTypeList.clear();
                }
                if (CollectionUtils.isEmpty(byUnitCode)) {
                    this.saveOrUpdateBatchMaterialUnit(materialUnitList, null);
                    materialUnitList.clear();
                }

                if (org.apache.commons.collections.CollectionUtils.isEmpty(materialUnitTypeList)
                        && org.apache.commons.collections.CollectionUtils.isEmpty(materialUnitList)) {
                    return;
                }

                // 客户表头数据 区分更新、新增
                if (org.apache.commons.collections.CollectionUtils.isNotEmpty(byUnitTypeCodes)) {
                    materialUnitTypeMap.putAll(byUnitTypeCodes.stream()
                            .collect(Collectors.toMap(MaterialUnitType::getUnitTypeCode, Function.identity())));
                    materialUnitTypeList.forEach(materialUnitType -> {
                        if (materialUnitTypeMap.containsKey(materialUnitType.getUnitTypeCode())) {
                            materialUnitTypeEntity.set(materialUnitTypeMap.get(materialUnitType.getUnitTypeCode()));
                            materialUnitType.setId(materialUnitTypeEntity.get().getId());
                            materialUnitType.setEnableStatus(materialUnitTypeEntity.get().getEnableStatus());
                            updateList.add(materialUnitType);
                        } else {
                            saveList.add(materialUnitType);
                        }
                    });
                    this.saveOrUpdateBatchMaterialUnitType(saveList, updateList);
                    saveList.clear();
                    updateList.clear();
                    materialUnitTypeList.clear();
                    materialUnitTypeMap.clear();
                }

                // 客户表头数据 区分更新、新增
                this.saveOrUpdateBatchMaterialUnit(materialUnitList, null);
                saveListMaterialUnit.clear();
                updateListMaterialUnit.clear();
                materialUnitList.clear();
                materialUnitMap.clear();
            }
        } finally {
            if (lock) {
                this.unLock(lockKey);
            }
        }
    }

    /**
     * 最大分页数
     *
     * @param dto
     * @return
     * @date 2023/1/8 1:33
     */
    private int countPageMax(MasterDataMdgBaseDto dto) {
        Integer totalNum = masterDataMdgService.countList(MasterDataMdgConstants.MDG_MATERIAL_UNIT_TPM);
        int maxPage = 1;
        int sizeInt = Integer.parseInt(dto.getPageSize());
        if (totalNum > sizeInt) {
            maxPage = totalNum % sizeInt > 0 ? (totalNum / sizeInt) + 1 : totalNum / sizeInt;
        }
        return maxPage;
    }

    /**
     * 批量新增数据（物料单位类型MDG用）
     *
     * @param saveList
     * @param updateList
     * @author dutaotao
     * @date 2023/1/3 17:38
     **/
    @Transactional(rollbackFor = Exception.class)
    void saveOrUpdateBatchMaterialUnitType(List<MaterialUnitType> saveList, List<MaterialUnitType> updateList) {

        // 新增操作
        if (!org.springframework.util.CollectionUtils.isEmpty(saveList)) {
            this.materialUnitTypeRepository.saveBatch(saveList);
        }

        // 更新操作
        if (!org.springframework.util.CollectionUtils.isEmpty(updateList)) {
            this.materialUnitTypeRepository.updateBatchById(updateList);
        }

    }

    /**
     * 批量新增数据（物料单位MDG用）
     *
     * @param saveListMaterialUnit
     * @param updateListMaterialUnit
     * @author dutaotao
     * @date 2023/1/3 17:38
     **/
    @Transactional(rollbackFor = Exception.class)
    public void saveOrUpdateBatchMaterialUnit(List<MaterialUnit> saveListMaterialUnit,
                                              List<MaterialUnit> updateListMaterialUnit) {
        List<String> collectUnitTypeCode = saveListMaterialUnit
                .stream()
                .map(MaterialUnit::getUnitTypeCode)
                .collect(Collectors.toList());
        this.materialUnitRepository.deleteBatchMaterialUnit(collectUnitTypeCode);
        // 新增操作
        if (!org.springframework.util.CollectionUtils.isEmpty(saveListMaterialUnit)) {
            this.materialUnitRepository.saveBatch(saveListMaterialUnit);
        }
    }

    /**
     * 按年月日解锁
     *
     * @param yearMonthDay
     * @author huojia
     * @date 2022/12/6 21:52
     **/
    private void unLock(String yearMonthDay) {
        if (StringUtils.isEmpty(yearMonthDay)) {
            throw new RuntimeException("拉取供应商解锁失败，日期不能为空！");
        }
        redisMutexService.unlock(MaterialUnitConstant.MATERIAL_UNIT_LOCK + yearMonthDay);
    }

    /**
     * 数据校验
     *
     * @param masterDataMdgUnitTypeVos
     * @author huojia
     * @date 2022/12/6 21:37
     **/
    private void unitCodeValidate(List<MasterDataMdgMaterialUnitVo> masterDataMdgUnitTypeVos,
                                  List<MaterialUnit> materialUnitList,
                                  List<MaterialUnitType> materialUnitTypeList,
                                  Map<String, String> codeMa,
                                  Map<String, String> unitConvertMap) {
        String tenantCode = TenantUtils.getTenantCode();
        masterDataMdgUnitTypeVos.forEach(mdgVo -> {
            // 主键为空直接跳过
            if (StringUtils.isEmpty(mdgVo.getMatnr())) {
                return;
            }
            // 主键为空直接跳过
            if (StringUtils.isEmpty(mdgVo.getMeinh())) {
                return;
            }
            // 物料单位换算系数
            MaterialUnit materialUnit = new MaterialUnit();
            BigDecimal businessScale = new BigDecimal(mdgVo.getUmren().trim());
            materialUnit.setConvertScale(businessScale);

            BigDecimal baseScale = new BigDecimal(mdgVo.getUmrez().trim());
            materialUnit.setMeasureConvertScale(baseScale);

            // 物料单位编码
            if (MaterialUnitConstant.UNIT_CODE_ST.equalsIgnoreCase(mdgVo.getMeinh())) {
                mdgVo.setMeinh(MaterialUnitConstant.UNIT_CODE_PC);
            }
            materialUnit.setUnitCode(mdgVo.getMeinh());
            // 物料单位名称
            materialUnit.setUnitName(mdgVo.getMeinhdesc());
            // 物料单位类型编码
            materialUnit.setUnitTypeCode(mdgVo.getMatnr());
            materialUnit.setStandardUnitCode(unitConvertMap.getOrDefault(mdgVo.getMeinhdesc1(), ""));
            materialUnit.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            materialUnit.setTenantCode(tenantCode);
            materialUnit.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            BooleanEnum booleanEnum = Objects.equals(materialUnit.getUnitCode(), materialUnit.getStandardUnitCode())
                    ? BooleanEnum.TRUE : BooleanEnum.FALSE;
            materialUnit.setStandardUnitFlag(booleanEnum.getCapital());
            materialUnitList.add(materialUnit);

            // 主键重复，跳过
            if (codeMa.containsKey(mdgVo.getMatnr())) {
                return;
            }
            codeMa.put(mdgVo.getMatnr(), "");
            MaterialUnitType materialUnitType = new MaterialUnitType();
            materialUnitType.setUnitTypeCode(mdgVo.getMatnr());
            // 根据产品编码查询数据
            materialUnitType.setMeasureTypeName(mdgVo.getMeinhdesc1());
            materialUnitType.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            materialUnitType.setTenantCode(tenantCode);
            materialUnitType.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            materialUnitTypeList.add(materialUnitType);

        });
    }

    /**
     * 按年月日加锁
     *
     * @param yearMonthDay
     * @return boolean
     * @author huojia
     * @date 2022/12/6 21:52
     **/
    private boolean lock(String yearMonthDay) {
        if (StringUtils.isEmpty(yearMonthDay)) {
            throw new RuntimeException("拉取物料单位加锁失败，日期不能为空！");
        }
        return this.redisMutexService.tryLock(MaterialUnitConstant.MATERIAL_UNIT_LOCK + yearMonthDay, TimeUnit.HOURS,
                12);
    }

    /**
     * 创建/编辑 数据校验
     *
     * @param materialUnit 物料单位实体
     */
    private void createValidate(MaterialUnit materialUnit) {
        Validate.notBlank(materialUnit.getUnitName(), "物料单位名称不能为空");
        Validate.notBlank(materialUnit.getUnitTypeCode(), "物料单位类别编码不能为空");
        Validate.notNull(materialUnit.getConvertScale(), "物料单位转换系数不能为空");
        Validate.notBlank(materialUnit.getStandardUnitFlag(), "物料单位标准单位标记不能为空");
        Validate.isTrue(materialUnit.getConvertScale().compareTo(new BigDecimal(BigInteger.ZERO)) > 0, "物料单位转换系数不能小于0");
        Validate.notNull(materialUnit.getMeasureConvertScale(), "EA单位转换系数不能为空");
        Validate.isTrue(materialUnit.getMeasureConvertScale().compareTo(new BigDecimal(BigInteger.ZERO)) > 0,
                "EA单位转换系数不能小于0");
        // 校验物料单位类型
        MaterialUnitType materialUnitType =
                materialUnitTypeRepository.findByUnitTypeCode(materialUnit.getUnitTypeCode());
        Validate.notNull(materialUnitType, "物料单位类别不存在！");
    }
}
