package com.biz.crm.mdm.business.material.unit.service.internal;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Objects;

import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.biz.crm.mdm.business.material.unit.entity.MaterialUnit;
import com.biz.crm.mdm.business.material.unit.entity.MaterialUnitType;
import com.biz.crm.mdm.business.material.unit.repository.MaterialUnitRepository;
import com.biz.crm.mdm.business.material.unit.repository.MaterialUnitTypeRepository;
import com.biz.crm.mdm.business.material.unit.service.MaterialUnitVoService;
import com.biz.crm.mdm.business.material.unit.vo.MaterialUnitTypeVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;

/**
 * @author jerry7
 * 物料单位vo service实现
 */
@Service
public class MaterialUnitVoServiceImpl implements MaterialUnitVoService {

  @Autowired(required = false)
  private MaterialUnitRepository materialUnitRepository;

  @Autowired(required = false)
  private MaterialUnitTypeServiceImpl materialUnitTypeService;

  @Autowired(required = false)
  private MaterialUnitTypeRepository materialUnitTypeRepository;

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;


  @Override
  public BigDecimal findScaleByStandUnit(String unitCode) {
    Validate.notBlank(unitCode, "需要转换的物料单位编码不能为空！");
    MaterialUnit materialUnit = materialUnitRepository.findByUnitCode(unitCode);
    Validate.notNull(materialUnit, "物料单位编码" + unitCode + "找不到对应的物料单位！");
    return materialUnit.getConvertScale();
  }

  @Override
  public BigDecimal findScaleByGoalUnit(String fromUnitCode, String goalUnitCode, int decimalScale, RoundingMode roundingMode) {
    Validate.isTrue(ObjectUtils.allNotNull(fromUnitCode, goalUnitCode), "需要转换的物料单位编码不能为空！");
    if (ObjectUtils.isEmpty(decimalScale)) {
      decimalScale = 2;
    }
    if (ObjectUtils.isEmpty(roundingMode)) {
      roundingMode = RoundingMode.HALF_UP;
    }
    MaterialUnit fromMaterialUnit = materialUnitRepository.findByUnitCode(fromUnitCode);
    MaterialUnit goalMaterialUnit = materialUnitRepository.findByUnitCode(goalUnitCode);
    Validate.isTrue(ObjectUtils.allNotNull(fromMaterialUnit, goalMaterialUnit), "需要转换的物料单位编码找不到对应的物料单位！");
    Validate.isTrue(fromMaterialUnit.getUnitTypeCode().equals(goalMaterialUnit.getUnitTypeCode()), "需要转换的物料单位必须是同一个物料单位类型！");
    return fromMaterialUnit.getConvertScale().divide(goalMaterialUnit.getConvertScale(), decimalScale, roundingMode);
  }

  /**
   * 根据物料类别编码查价格详情详情
   *
   * @param productCode 物料类别编码
   * @return 配置实体
   */
  @Override
  public MaterialUnitTypeVo findByUnitTypeCodeProduct(String productCode) {
    if (StringUtils.isBlank(productCode)) {
      return null;
    }
    MaterialUnitType byUnitTypeCodeProduct = materialUnitTypeRepository.findByUnitTypeCodeProduct(productCode, TenantUtils.getTenantCode());
    if (Objects.isNull(byUnitTypeCodeProduct)){
      throw new RuntimeException("数据来源为空") ;
    }
    MaterialUnitType materialUnitType = this.materialUnitTypeService.findById(byUnitTypeCodeProduct.getId());
    if (Objects.isNull(materialUnitType)){
      throw new RuntimeException("物料单位数据来源为空") ;
    }
    MaterialUnitTypeVo vo = this.nebulaToolkitService.copyObjectByWhiteList(materialUnitType, MaterialUnitTypeVo.class, HashSet.class, ArrayList.class);
    return vo;
  }
}
