package com.biz.crm.mdm.business.material.unit.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mdm.business.material.unit.constant.MaterialUnitConstant;
import com.biz.crm.mdm.business.material.unit.dto.MaterialUnitTypeDto;
import com.biz.crm.mdm.business.material.unit.entity.MaterialUnit;
import com.biz.crm.mdm.business.material.unit.entity.MaterialUnitType;
import com.biz.crm.mdm.business.material.unit.event.MaterialUnitChangeListener;
import com.biz.crm.mdm.business.material.unit.repository.MaterialUnitTypeRepository;
import com.biz.crm.mdm.business.material.unit.service.MaterialUnitService;
import com.biz.crm.mdm.business.material.unit.service.MaterialUnitTypeService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import java.math.BigDecimal;

import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.RoundingMode;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * @author jerry7
 * 物料单位类型 service实现
 */
@Service
public class MaterialUnitTypeServiceImpl implements MaterialUnitTypeService {

  @Autowired(required = false)
  private MaterialUnitTypeRepository materialUnitTypeRepository;

  @Autowired(required = false)
  private LoginUserService loginUserService;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired(required = false)
  private MaterialUnitService materialUnitService;

  @Autowired(required = false)
  @Lazy
  private List<MaterialUnitChangeListener> materialUnitChangeListeners;



  @Override
  public Page<MaterialUnitType> findByConditions(Pageable pageable, MaterialUnitTypeDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new MaterialUnitTypeDto());
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    Page<MaterialUnitType> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return materialUnitTypeRepository.findByConditions(page, dto);
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "待修正的数据主键不能为空");
    materialUnitTypeRepository.updateEnableStatusByIdIn(EnableStatusEnum.ENABLE, ids);
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "待修正的数据主键不能为空");
    materialUnitTypeRepository.updateEnableStatusByIdIn(EnableStatusEnum.DISABLE, ids);
  }

  @Override
  @Transactional
  public void deleteBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "待修正的数据主键不能为空");
    if (!CollectionUtils.isEmpty(materialUnitChangeListeners) && !CollectionUtils.isEmpty(ids)) {
      ids.forEach(oldUnit -> {
        MaterialUnitType type = this.findById(oldUnit);
        if (ObjectUtils.isNotEmpty(type)) {
          for (MaterialUnitChangeListener listener : materialUnitChangeListeners) {
            listener.onDeleteUnitType(type.getUnitTypeCode());
          }
        }
      });
    }
    materialUnitTypeRepository.updateDelStatusByIdIn(DelFlagStatusEnum.DELETE, ids);
  }

  @Override
  public MaterialUnitType findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    MaterialUnitType detailById = materialUnitTypeRepository.findDetailById(id);
    if (Objects.isNull(detailById)){
      return null;
    }
    List<MaterialUnit> byUnitTypeCode = materialUnitService.findByUnitTypeCode(detailById.getUnitTypeCode());
    detailById.setMaterialUnits(byUnitTypeCode);
    return detailById;
  }

  @Override
  public MaterialUnitType findByUnitTypeCode(String materialTypeCode) {
    if (StringUtils.isBlank(materialTypeCode)) {
      return null;
    }
    return materialUnitTypeRepository.findDetailByUnitTypeCode(materialTypeCode);
  }

  @Override
  @Transactional
  public MaterialUnitType create(MaterialUnitType materialUnitType) {
    this.createValidate(materialUnitType);
    Validate.isTrue(StringUtils.isBlank(materialUnitType.getId()), "创建登录配置ID不能有值！");
    //设置基础属性值
    materialUnitType.setCreateTime(new Date());
    materialUnitType.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    materialUnitType.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    materialUnitType.setTenantCode(TenantUtils.getTenantCode());
    materialUnitType.setCreateAccount(loginUserService.getLoginAccountName());
    //生成物料类型编码
    List<String> codes = generateCodeService.generateCode(MaterialUnitConstant.MATERIAL_UNIT_TYPE_CODE, 1);
    Validate.isTrue(!CollectionUtils.isEmpty(codes), "生成物料单位类别编码失败！");
    materialUnitType.setUnitTypeCode(codes.stream().findFirst().get());
    this.materialUnitTypeRepository.save(materialUnitType);
    //保存物料单位
    List<MaterialUnit> materialUnits = materialUnitType.getMaterialUnits();
    materialUnits.forEach(materialUnit -> {
      materialUnit.setUnitTypeCode(materialUnitType.getUnitTypeCode());
      materialUnit.setTenantCode(materialUnitType.getTenantCode());   //新增租户编号
      this.materialUnitService.create(materialUnit);
    });
    return materialUnitType;
  }

  @Override
  @Transactional
  public MaterialUnitType update(MaterialUnitType materialUnitType) {
    this.createValidate(materialUnitType);
    Validate.notBlank(materialUnitType.getId(), "登录登录配置ID不能为空！");
    materialUnitType.setModifyTime(new Date());
    materialUnitType.setModifyAccount(loginUserService.getLoginAccountName());
    materialUnitType.setTenantCode(TenantUtils.getTenantCode());   //新增租户编号
    this.materialUnitTypeRepository.saveOrUpdate(materialUnitType);
    //保存物料单位
    List<MaterialUnit> materialUnits = materialUnitType.getMaterialUnits();
    List<String> unitCodes = materialUnits.stream().map(MaterialUnit::getUnitCode).collect(Collectors.toList());
    //发送物料删除事件
    List<MaterialUnit> oldUnits = this.materialUnitService.findByUnitTypeCode(materialUnitType.getUnitTypeCode());
    if (!CollectionUtils.isEmpty(materialUnitChangeListeners) && !CollectionUtils.isEmpty(oldUnits)) {
      oldUnits.forEach(oldUnit -> {
        if (!unitCodes.contains(oldUnit.getUnitCode())) {
          for (MaterialUnitChangeListener listener : materialUnitChangeListeners) {
            listener.onDelete(oldUnit.getUnitCode());
          }
        }
      });
    }
    //删除物料
    this.materialUnitService.deleteByUnitTypeCode(materialUnitType.getUnitTypeCode(), unitCodes);
    materialUnits.forEach(materialUnit -> {
      materialUnit.setUnitTypeCode(materialUnitType.getUnitTypeCode());
      materialUnit.setTenantCode(materialUnitType.getTenantCode());   //新增租户编号
      this.materialUnitService.create(materialUnit);
    });
    return materialUnitType;
  }

  private void createValidate(MaterialUnitType materialUnitType) {
    Validate.notNull(materialUnitType, "更新登录配置实体不能为空！");
    Validate.notBlank(materialUnitType.getMeasureTypeName(), "EA单位名称不能为空");
    Validate.notNull(materialUnitType.getMaterialUnits(), "物料单位列表不能为空");
    //根据物料单位转换系数倒转
    List<MaterialUnit> materialUnits = materialUnitType.getMaterialUnits();
    Validate.isTrue(!CollectionUtils.isEmpty(materialUnits), "请至少添加一个单位的信息");
    materialUnits.sort(Comparator.comparing(MaterialUnit::getConvertScale).reversed());
    MaterialUnit firstUnit = materialUnits.get(0);
    StringBuilder unitTypeName = new StringBuilder(1 + firstUnit.getUnitName());
    for (int i = 1; i < materialUnits.size(); i++) {
      BigDecimal v = firstUnit.getConvertScale()
              .divide(materialUnits.get(i).getConvertScale(), 2, RoundingMode.HALF_UP);
      unitTypeName.append("=").append(this.hasDecimal(v) ? v : v.intValue()).append(materialUnits.get(i).getUnitName());
    }
    materialUnitType.setUnitTypeName(unitTypeName.toString());
    //校验基本单位
    List<String> codes = generateCodeService.generateCode(MaterialUnitConstant.MATERIAL_UNIT_TYPE_CODE, materialUnits.size());
    Validate.isTrue(!CollectionUtils.isEmpty(codes), "生成物料单位类别编码失败！");
    Map<String, List<MaterialUnit>> unitMap = Maps.newHashMap();
    for (int i = 0; i < materialUnits.size(); i++) {
      MaterialUnit unit = materialUnits.get(i);
      List<MaterialUnit> unitList = Lists.newArrayList();
      if (unitMap.containsKey(unit.getStandardUnitFlag())) {
        unitList.addAll(unitMap.get(unit.getStandardUnitFlag()));
      }
      unitList.add(unit);
      unitMap.put(unit.getStandardUnitFlag(), unitList);
      //未物料单位设置物料编码
      if (StringUtils.isBlank(unit.getUnitCode())) {
        unit.setUnitCode(codes.get(i));
      }
    }
    Validate.isTrue(unitMap.containsKey(BooleanEnum.TRUE.getCapital()) && unitMap.get(BooleanEnum.TRUE.getCapital()).size() == 1, "一个物料单位组内基本单位有且只能有一个");
    //设置标准单位编码
    materialUnits.forEach(materialUnit -> materialUnit.setStandardUnitCode(unitMap.get(BooleanEnum.TRUE.getCapital()).get(0).getUnitCode()));
  }

  /**
   * 是否有小数
   *
   * @param v
   * @return
   */
  private Boolean hasDecimal(BigDecimal v) {
    String split = ".";
    if (Objects.isNull(v)) {
      return false;
    }
    String str = Optional.of(v).orElse(BigDecimal.ZERO).toString();
    if (str.contains(split)) {
      String s = StringUtils.split(str, split)[1];
      return BigDecimal.ZERO.compareTo(new BigDecimal(s)) < 0;
    } else {
      return false;
    }
  }
}
