package com.biz.crm.mdm.business.news.notice.local.service.internal;

import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.model.AbstractCrmUserIdentity;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mdm.business.news.notice.local.entity.NewsNoticeEntity;
import com.biz.crm.mdm.business.news.notice.local.repository.NewsNoticeRepository;
import com.biz.crm.mdm.business.news.notice.local.service.NewsNoticeFileService;
import com.biz.crm.mdm.business.news.notice.local.service.NewsNoticeScopeService;
import com.biz.crm.mdm.business.news.notice.local.service.NewsNoticeService;
import com.biz.crm.mdm.business.news.notice.sdk.dto.NewsNoticeDto;
import com.biz.crm.mdm.business.news.notice.sdk.enums.ScopeType;
import com.biz.crm.mdm.business.news.notice.sdk.event.NewsNoticeEventListener;
import com.biz.crm.mdm.business.news.notice.sdk.vo.NewsNoticeEventVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 公告表服务实现类
 *
 * @author lww
 * @date 2022/5/13
 */
@Service("noticeService")
public class NewsNoticeServiceImpl implements NewsNoticeService {

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private NewsNoticeRepository newsNoticeRepository;
  @Autowired(required = false)
  private NewsNoticeFileService newsNoticeFileService;
  @Autowired(required = false)
  private NewsNoticeScopeService newsNoticeScopeService;
  @Autowired(required = false)
  private List<NewsNoticeEventListener> listeners;
  @Autowired(required = false)
  private LoginUserService loginUserService;

  @Override
  @Transactional
  public NewsNoticeEntity create(NewsNoticeDto dto) {
    this.createValidation(dto);
    AbstractCrmUserIdentity loginDetails = loginUserService.getAbstractLoginUser();
    NewsNoticeEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, NewsNoticeEntity.class, HashSet.class, ArrayList.class);
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    // 确认组织机构信息
    Object orgCodeObject = loginDetails.invokeFieldValue("orgCode");
    Object orgNameObject = loginDetails.invokeFieldValue("orgName");
    entity.setPublishOrgCode(orgCodeObject == null?null:orgCodeObject.toString());
    entity.setPublishOrgName(orgNameObject == null?null:orgNameObject.toString());
    this.newsNoticeRepository.save(entity);
    //绑定关联数据
    this.newsNoticeFileService.update(dto.getFileList(), entity.getId());
    this.newsNoticeScopeService.update(dto.getScopeList(), entity.getId());
    //新增公告事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      NewsNoticeEventVo vo = this.nebulaToolkitService.copyObjectByBlankList(entity, NewsNoticeEventVo.class, HashSet.class, ArrayList.class);
      listeners.forEach(listener -> {
        listener.onCreate(vo);
      });
    }
    return entity;
  }


  @Override
  @Transactional
  public NewsNoticeEntity update(NewsNoticeDto dto) {
    this.updateValidation(dto);
    NewsNoticeEntity entity = this.newsNoticeRepository.getById(dto.getId());
    Validate.notNull(entity, "公告信息不存在");
    NewsNoticeEntity updateEntity = this.nebulaToolkitService.copyObjectByWhiteList(dto, NewsNoticeEntity.class, HashSet.class, ArrayList.class);
    this.newsNoticeRepository.updateById(updateEntity);
    //绑定关联数据
    this.newsNoticeFileService.update(dto.getFileList(), entity.getId());
    this.newsNoticeScopeService.update(dto.getScopeList(), entity.getId());
    //更新公告事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      NewsNoticeEventVo oldVo = this.nebulaToolkitService.copyObjectByWhiteList(entity, NewsNoticeEventVo.class, HashSet.class, ArrayList.class);
      NewsNoticeEventVo newVo = this.nebulaToolkitService.copyObjectByBlankList(dto, NewsNoticeEventVo.class, HashSet.class, ArrayList.class);
      listeners.forEach(listener -> {
        listener.onUpdate(oldVo, newVo);
      });
    }
    return updateEntity;
  }

  @Override
  @Transactional
  public void deleteBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "缺失id");
    List<NewsNoticeEntity> entities = this.newsNoticeRepository.listByIds(ids);
    Validate.isTrue(!CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据删除个数不匹配");
    this.newsNoticeRepository.updateDelFlagByIds(ids);
    //删除公告事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      List<NewsNoticeEventVo> voList = Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(entities, NewsNoticeEntity.class
          , NewsNoticeEventVo.class, HashSet.class, ArrayList.class));
      listeners.forEach(listener -> {
        listener.onDelete(voList);
      });
    }
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "缺失id");
    List<NewsNoticeEntity> entities = this.newsNoticeRepository.listByIds(ids);
    Validate.isTrue(!CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据启用个数不匹配");
    this.newsNoticeRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
    //启用公告事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      List<NewsNoticeEventVo> voList = Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(entities, NewsNoticeEntity.class
          , NewsNoticeEventVo.class, HashSet.class, ArrayList.class));
      listeners.forEach(listener -> {
        listener.onEnable(voList);
      });
    }
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "缺失id");
    List<NewsNoticeEntity> entities = this.newsNoticeRepository.listByIds(ids);
    Validate.isTrue(!CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据禁用个数不匹配");
    this.newsNoticeRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
    //禁用公告事件通知
    if (!CollectionUtils.isEmpty(listeners)) {
      List<NewsNoticeEventVo> voList = Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(entities, NewsNoticeEntity.class
          , NewsNoticeEventVo.class, HashSet.class, ArrayList.class));
      listeners.forEach(listener -> {
        listener.onDisable(voList);
      });
    }
  }

  /**
   * 在创建newsnotice模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void createValidation(NewsNoticeDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setId(null);
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getTitle(), "缺失标题");
    Validate.notBlank(dto.getType(), "缺失类型");
    Validate.notNull(dto.getStartTime(), "缺失生效开始时间");
    Validate.notNull(dto.getEndTime(), "缺失生效结束时间");
    Validate.isTrue(dto.getStartTime().before(dto.getEndTime()), "生效开始时间必须小于生效结束时间");
    if (!CollectionUtils.isEmpty(dto.getScopeList())) {
      dto.getScopeList().forEach(noticeScopeDto -> {
        List<String> scopeTypeList = Arrays.stream(ScopeType.values()).map(ScopeType::name).collect(Collectors.toList());
        Validate.isTrue(scopeTypeList.contains(noticeScopeDto.getScopeType()), "不支持的范围类型!");
      });
    }
    Validate.isTrue(dto.getTitle().length() < 128, "标题，在进行编辑时填入值超过了限定长度(128)，请检查!");
  }

  /**
   * 在修改newsnotice模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void updateValidation(NewsNoticeDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getId(), "修改信息时，id不能为空！");
    Validate.notBlank(dto.getTitle(), "缺失标题");
    Validate.notBlank(dto.getType(), "缺失类型");
    Validate.notNull(dto.getStartTime(), "缺失生效开始时间");
    Validate.notNull(dto.getEndTime(), "缺失生效结束时间");
    Validate.isTrue(dto.getStartTime().before(dto.getEndTime()), "生效开始时间必须小于生效结束时间");
    if (!CollectionUtils.isEmpty(dto.getScopeList())) {
      dto.getScopeList().forEach(noticeScopeDto -> {
        List<String> scopeTypeList = Arrays.stream(ScopeType.values()).map(ScopeType::name).collect(Collectors.toList());
        Validate.isTrue(scopeTypeList.contains(noticeScopeDto.getScopeType()), "不支持的范围类型!");
      });
    }
    Validate.isTrue(dto.getTitle().length() < 128, "标题，在进行编辑时填入值超过了限定长度(128)，请检查!");
  }
}
