package com.biz.crm.mdm.business.org.local.service.internal;

import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mdm.business.org.local.entity.Org;
import com.biz.crm.mdm.business.org.local.entity.OrgRegion;
import com.biz.crm.mdm.business.org.local.repository.OrgRegionRepository;
import com.biz.crm.mdm.business.org.local.service.OrgRegionService;
import com.biz.crm.mdm.business.org.local.service.OrgService;
import com.biz.crm.mdm.business.org.sdk.dto.OrgRegionDto;
import com.biz.crm.mdm.business.region.sdk.service.RegionVoService;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * @author hecheng
 * @description: 组织区域 关系操作
 * @date 2021/10/15 上午10:46
 */
@Service
public class OrgRegionServiceImpl implements OrgRegionService {

  @Autowired(required = false) private OrgRegionRepository orgRegionRepository;
  @Autowired(required = false) private OrgService orgService;

  @Autowired(required = false)
  private LoginUserService loginUserService;

  @Autowired(required = false)
  private RegionVoService regionVoService;

  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public List<OrgRegion> findByOrgCode(String orgCode) {
    if (StringUtils.isBlank(orgCode)) {
      return null;
    }
    return this.orgRegionRepository.findByOrgCode(orgCode, TenantUtils.getTenantCode());
  }

  @Override
  @Transactional
  public List<OrgRegion> createBatch(List<OrgRegion> orgRegions) {
    if (CollectionUtils.isEmpty(orgRegions)) {
      return null;
    }
    List<OrgRegion> result = Lists.newArrayList();
    Map<String, List<OrgRegion>> relMap =
        orgRegions.stream().collect(Collectors.groupingBy(OrgRegion::getOrgCode));
    for (Map.Entry<String, List<OrgRegion>> item : relMap.entrySet()) {
      String orgCode = item.getKey();
      List<OrgRegion> tempOrgRegions = item.getValue();
      if (StringUtils.isBlank(orgCode) || CollectionUtils.isEmpty(tempOrgRegions)) {
        continue;
      }
      Org org = this.orgService.findByOrgCode(orgCode);
      Validate.notNull(org, "[%s]组织不存在", orgCode);
      this.orgRegionRepository.deleteByOrgCode(orgCode, TenantUtils.getTenantCode());
      tempOrgRegions =
          tempOrgRegions.stream()
              .filter(a -> StringUtils.isNotBlank(a.getCountryCode()))
              .map(
                  orgRegion -> {
                    Date now = new Date();
                    String account = this.loginUserService.getLoginAccountName();
                    orgRegion.setCreateAccount(account);
                    orgRegion.setModifyAccount(account);
                    orgRegion.setCreateTime(now);
                    orgRegion.setModifyTime(now);
                    orgRegion.setTenantCode(TenantUtils.getTenantCode());
                    return orgRegion;
                  })
              .collect(Collectors.toList());
      if (CollectionUtils.isNotEmpty(tempOrgRegions)) {
        this.createValidation(tempOrgRegions);
        this.orgRegionRepository.saveBatch(tempOrgRegions);
        result.addAll(tempOrgRegions);
      }
    }
    return result;
  }

  @Override
  public List<OrgRegion> findByOrgCodes(List<String> orgCodes) {
    if (CollectionUtils.isEmpty(orgCodes)) {
      return null;
    }
    return this.orgRegionRepository.findByOrgCodes(orgCodes, TenantUtils.getTenantCode());
  }

  /**
   * 校验数据
   *
   * @param orgRegions
   */
  private void createValidation(List<OrgRegion> orgRegions) {
    /** 1.验证是否存在区域 2.验证区域是否重复或者 区域存在包含关系 3.验证是否已被其他组织绑定 */
    Stream<String> countryCodeStream =
        orgRegions.stream()
            .filter(item -> StringUtils.isNotBlank(item.getCountryCode()))
            .map(OrgRegion::getCountryCode);
    Stream<String> provinceCodeStream =
        orgRegions.stream()
            .filter(item -> StringUtils.isNotBlank(item.getProvinceCode()))
            .map(OrgRegion::getProvinceCode);
    Stream<String> cityCodeStream =
        orgRegions.stream()
            .filter(item -> StringUtils.isNotBlank(item.getCityCode()))
            .map(OrgRegion::getCityCode);
    Stream<String> districtCodeStream =
        orgRegions.stream()
            .filter(item -> StringUtils.isNotBlank(item.getDistrictCode()))
            .map(OrgRegion::getDistrictCode);
    Stream<String> townshipCodeStream =
        orgRegions.stream()
            .filter(item -> StringUtils.isNotBlank(item.getTownshipCode()))
            .map(OrgRegion::getTownshipCode);
    Set<String> regionCodes =
        Stream.of(
                countryCodeStream,
                provinceCodeStream,
                cityCodeStream,
                districtCodeStream,
                townshipCodeStream)
            .flatMap(itemStream -> itemStream)
            .collect(Collectors.toSet());
    Map<String, String> keyValue =
        this.regionVoService.findRegionCodeNameMap(Lists.newArrayList(regionCodes));
    Validate.notNull(keyValue, "系统中不存在区域");
    List<String> uniqueKeys = Lists.newArrayList();
    for (OrgRegion orgRegion : orgRegions) {
      Validate.isTrue(keyValue.containsKey(orgRegion.getCountryCode()), "国家编码有误");
      Validate.isTrue(keyValue.containsKey(orgRegion.getProvinceCode()), "省份编码有误");
      Validate.isTrue(keyValue.containsKey(orgRegion.getCityCode()), "市编码有误");
      Validate.isTrue(keyValue.containsKey(orgRegion.getDistrictCode()), "区县编码有误");
      Validate.isTrue(keyValue.containsKey(orgRegion.getTownshipCode()), "乡镇编码有误");
      // 判断是否重复，或者存在包含的情况
      String key =
          generateUniqueKey(
              orgRegion.getCountryCode(),
              orgRegion.getProvinceCode(),
              orgRegion.getCityCode(),
              orgRegion.getDistrictCode(),
              orgRegion.getTownshipCode());
      for (String exitsKey : uniqueKeys) {
        Validate.isTrue(!Objects.equals(exitsKey, key), "存在重复区域数据：【%s】", key);
        Validate.isTrue(
            !(exitsKey.startsWith(key) || key.startsWith(exitsKey)),
            "区域存在行存在包含关系数据：【%s,%s】",
            key,
            exitsKey);
      }
      uniqueKeys.add(key);
      // 判断是否区域已经被绑定
      OrgRegionDto dto =
          this.nebulaToolkitService.copyObjectByWhiteList(
              orgRegion, OrgRegionDto.class, HashSet.class, ArrayList.class);
      dto.setOrgCode(null);
      dto.setTenantCode(TenantUtils.getTenantCode());
      List<OrgRegion> list = this.orgRegionRepository.findByConditions(dto);
      if (CollectionUtils.isEmpty(list)) {
        continue;
      }
      List<OrgRegion> listExcludeSelf =
          list.stream()
              .filter(x -> !Objects.equals(orgRegion.getOrgCode(), x.getOrgCode()))
              .collect(Collectors.toList());
      if (CollectionUtils.isEmpty(listExcludeSelf)) {
        continue;
      }
      Map<String, OrgRegion> uniqueKeyMap =
          list.stream()
              .collect(
                  Collectors.toMap(
                      k -> {
                        return generateUniqueKey(
                            k.getCountryCode(),
                            k.getProvinceCode(),
                            k.getCityCode(),
                            k.getDistrictCode(),
                            k.getTownshipCode());
                      },
                      v -> v));
      if (uniqueKeyMap.containsKey(key)) {
        OrgRegion useOrg = uniqueKeyMap.get(key);
        String useOrgName = useOrg.getOrgCode();
        String name =
            (StringUtils.isBlank(orgRegion.getCountryCode())
                    ? ""
                    : keyValue.get(orgRegion.getCountryCode()))
                + (StringUtils.isBlank(orgRegion.getProvinceCode())
                    ? ""
                    : keyValue.get(orgRegion.getProvinceCode()))
                + (StringUtils.isBlank(orgRegion.getCityCode())
                    ? ""
                    : keyValue.get(orgRegion.getCityCode()))
                + (StringUtils.isBlank(orgRegion.getDistrictCode())
                    ? ""
                    : keyValue.get(orgRegion.getDistrictCode()))
                + (StringUtils.isBlank(orgRegion.getTownshipCode())
                    ? ""
                    : keyValue.get(orgRegion.getTownshipCode()));
        Validate.isTrue(false, name + "已经被[" + useOrgName + "]关联");
      }
    }
  }

  /**
   * 拼接字段作为唯一验证的凭据
   *
   * @param countryCode
   * @param provinceCode
   * @param cityCode
   * @param districtCode
   * @param townshipCode
   * @return
   */
  private String generateUniqueKey(
      String countryCode,
      String provinceCode,
      String cityCode,
      String districtCode,
      String townshipCode) {
    return "_"
        + (StringUtils.isEmpty(countryCode) ? "" : countryCode + "_")
        + (StringUtils.isEmpty(provinceCode) ? "" : provinceCode + "_")
        + (StringUtils.isEmpty(cityCode) ? "" : cityCode + "_")
        + (StringUtils.isEmpty(districtCode) ? "" : districtCode + "_")
        + (StringUtils.isEmpty(townshipCode) ? "" : townshipCode + "_");
  }
}
