package com.biz.crm.mdm.business.org.local.service.notifier;

import com.biz.crm.mdm.business.org.local.entity.OrgPosition;
import com.biz.crm.mdm.business.org.local.service.OrgPositionService;
import com.biz.crm.mdm.business.org.sdk.common.constant.OrgCodeConstant;
import com.biz.crm.mdm.business.org.sdk.vo.OrgPositionRelationViewVo;
import com.biz.crm.mdm.business.position.sdk.dto.RelationDataDto;
import com.biz.crm.mdm.business.position.sdk.event.PositionEventListener;
import com.biz.crm.mdm.business.position.sdk.vo.PositionEventVo;
import com.biz.crm.mdm.business.position.sdk.vo.PositionRelationEventVo;
import com.biz.crm.mdm.business.position.sdk.vo.PositionRelationVo;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author hecheng
 * @description: 组织实现 职位信息变更事件通知接口
 * @date 2021/10/14 下午4:36
 */
@Component
public class PositionEventOrgListenerImpl implements PositionEventListener {

  @Autowired(required = false)
  private OrgPositionService orgPositionService;

  @Override
  @Transactional
  public void onCreate(PositionEventVo vo) {
    /**
     * 当特定的岗位被创建时，组织机构模块收到事件后，会根据DTO中关联的key信息，来判定是否要创建 组织机构-岗位的关联信息。
     * 1 判断 组织Code是否存在 如果不存在那么跳过创建关系
     * 2.验证 组织是否存在
     * 3.验证 职位是否存在
     * 4.查询职位是否已存在对应的组织关系（这里是一个组织对应多个职位的关系）
     * 4.1如果已存在那么删除旧的关系，新建新的关系
     * 4.2如果不存在那么直接创建
     */
    if (Objects.isNull(vo)) {
      return;
    }
    List<PositionRelationEventVo> relationData = vo.getRelationData();
    if (CollectionUtils.isEmpty(relationData)) {
      return;
    }
    for (PositionRelationEventVo relation : relationData) {
      if (Objects.equals(relation.getRelationKey(), OrgCodeConstant.KEY)) {
        List<RelationDataDto> relationCodes = relation.getRelationData();
        if (CollectionUtils.isEmpty(relationCodes)) {
          return;
        }
        List<OrgPosition> orgPositions = relationCodes.stream().map(item -> {
          OrgPosition orgPosition = new OrgPosition();
          orgPosition.setOrgCode(item.getCode());
          orgPosition.setPositionCode(vo.getPositionCode());
          return orgPosition;
        }).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(orgPositions)) {
          return;
        }
        this.orgPositionService.create(orgPositions);
      }
    }

  }

  @Override
  @Transactional
  public void onUpdate(PositionEventVo oldVo, PositionEventVo newVo) {
    /**
     * 当特定的岗位被创建时，组织机构模块收到事件后，会根据DTO中关联的key信息，来判定是否要创建 组织机构-岗位的关联信息。
     *
     * 1 判断 组织Code是否存在 如果不存在那么删除已存在的关系
     * 2验证 组织是否存在
     * 3验证 职位是否存在
     * 4查询职位是否已存在对应的组织关系（这里是一个组织对应多个职位的关系）
     * 4.1如果已存在那么删除旧的关系，新建新的关系
     * 4.2如果不存在那么直接创建
     */
    if (Objects.isNull(oldVo) || Objects.isNull(newVo)) {
      return;
    }
    this.orgPositionService.deleteByPositionCodes(Lists.newArrayList(oldVo.getPositionCode()));
    List<OrgPosition> newOrgPositions = getOrgPositions(newVo);
    if (CollectionUtils.isEmpty(newOrgPositions)) {
      return;
    }
    this.orgPositionService.create(newOrgPositions);
  }

  private List<OrgPosition> getOrgPositions(PositionEventVo newVo) {
    List<PositionRelationEventVo> relationData = newVo.getRelationData();
    if (CollectionUtils.isEmpty(relationData)) {
      return null;
    }
    List<OrgPosition> allOrgPositions = new ArrayList<>();
    for (PositionRelationEventVo relation : relationData) {
      if (Objects.equals(relation.getRelationKey(), OrgCodeConstant.KEY)) {
        List<RelationDataDto> relationCodes = relation.getRelationData();
        if (CollectionUtils.isEmpty(relationCodes)) {
          return null;
        }
        List<OrgPosition> orgPositions = relationCodes.stream().map(item -> {
          OrgPosition orgPosition = new OrgPosition();
          orgPosition.setOrgCode(item.getCode());
          orgPosition.setPositionCode(newVo.getPositionCode());
          return orgPosition;
        }).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(orgPositions)) {
          return null;
        } else {
          allOrgPositions.addAll(orgPositions);
        }
      }
    }
    return allOrgPositions;
  }

  /**
   * 不需要实现
   *
   * @param voList 禁用vo信息
   */
  @Override
  @Transactional
  public void onDisable(List<PositionEventVo> voList) {

  }

  /**
   * 不需要实现
   *
   * @param voList 禁用vo信息
   */
  @Override
  @Transactional
  public void onEnable(List<PositionEventVo> voList) {

  }

  /**
   * 业务逻辑-当职位删除的时候直接删除职位和职位和组织的关联关系,不需要阻止职位的删除
   *
   * @param voList 删除vo信息
   */
  @Override
  public void onDelete(List<PositionEventVo> voList) {
    if (CollectionUtils.isEmpty(voList)) {
      return;
    }
    List<String> positionCodes = voList.stream().map(PositionEventVo::getPositionCode).collect(Collectors.toList());
    this.orgPositionService.deleteByPositionCodes(positionCodes);
  }

  @Override
  public List<PositionRelationVo> onRequestByPositionCodes(List<String> positionCodes) {
    if (CollectionUtils.isEmpty(positionCodes)) {
      return null;
    }
    List<OrgPosition> orgPositions = this.orgPositionService.findByPositionCodes(positionCodes);
    if (CollectionUtils.isEmpty(orgPositions)) {
      return null;
    }
    //按职位分组
    Map<String, List<OrgPosition>> relMap = orgPositions.stream().collect(Collectors.groupingBy(OrgPosition::getPositionCode));
    List<PositionRelationVo> list = Lists.newArrayList();
    for (Map.Entry<String, List<OrgPosition>> item : relMap.entrySet()) {
      PositionRelationVo vo = new PositionRelationVo();
      vo.setPositionCode(item.getKey());
      List<OrgPosition> relations = item.getValue();
      if (!CollectionUtils.isEmpty(relations)) {
        List<OrgPositionRelationViewVo> relationVos = relations.stream().map(temp -> {
          OrgPositionRelationViewVo viewVo = new OrgPositionRelationViewVo();
          viewVo.setCode(temp.getOrgCode());
          return viewVo;
        }).collect(Collectors.toList());
        vo.setRelationData(Lists.newArrayList(relationVos));
      }
      vo.setRelationKey(OrgCodeConstant.KEY);
      vo.setRelationName(OrgCodeConstant.NAME);
      list.add(vo);
    }
    return list;
  }
}
