package com.biz.crm.mdm.business.org.local.service.internal;

import com.biz.crm.mdm.business.org.local.entity.Org;
import com.biz.crm.mdm.business.org.local.entity.OrgPosition;
import com.biz.crm.mdm.business.org.local.repository.OrgPositionRepository;
import com.biz.crm.mdm.business.org.local.service.OrgPositionService;
import com.biz.crm.mdm.business.org.local.service.OrgService;
import com.biz.crm.mdm.business.org.sdk.dto.OrgPositionBindDto;
import com.biz.crm.mdm.business.position.sdk.service.PositionVoService;
import com.biz.crm.mdm.business.position.sdk.vo.PositionVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author hecheng
 * @description: 组织职位接口实现
 * @date 2021/10/14 下午3:54
 */
@Service
public class OrgPositionServiceImpl implements OrgPositionService {

  @Autowired(required = false)
  private OrgPositionRepository orgPositionRepository;
  @Autowired(required = false)
  private OrgService orgService;
  @Autowired(required = false)
  private PositionVoService positionVoService;

  /**
   * 基于数据库执行的数据视图执行内容缓存（最多500毫秒）
   */
  private static volatile Cache<String, List<OrgPosition>> cache = null;

  public OrgPositionServiceImpl(){
    if(cache == null) {
      synchronized (OrgPositionServiceImpl.class) {
        while(cache == null) {
          cache = CacheBuilder.newBuilder()
                  .initialCapacity(10000)
                  .expireAfterWrite(500, TimeUnit.MILLISECONDS)
                  .maximumSize(100000)
                  .build();
        }
      }
    }
  }

  @Override
  @Transactional
  public List<OrgPosition> create(List<OrgPosition> orgPositions) {
    if (CollectionUtils.isEmpty(orgPositions)) {
      return null;
    }
    List<OrgPosition> result = Lists.newArrayList();
    for (OrgPosition orgPosition : orgPositions) {
      OrgPosition current = this.createForm(orgPosition);
      OrgPosition oldRel = this.orgPositionRepository.findByPositionCodeAndOrgCode(orgPosition.getPositionCode(), orgPosition.getOrgCode(), TenantUtils.getTenantCode());
      if (Objects.nonNull(oldRel)) {
        result.add(oldRel);
      } else {
        this.orgPositionRepository.save(current);
        result.add(current);
      }
    }
    return result;
  }

  @Override
  public List<OrgPosition> findByPositionCodes(List<String> positionCodes) {
    if (CollectionUtils.isEmpty(positionCodes)) {
      return null;
    }
    String cacheKey = StringUtils.join(TenantUtils.getTenantCode(), positionCodes);
    List<OrgPosition> graph = cache.getIfPresent(cacheKey);
    if (graph == null) {
      graph = this.orgPositionRepository.findByPositionCodes(positionCodes, TenantUtils.getTenantCode());
      cache.put(cacheKey, graph);
    }
    return graph;
  }

  @Override
  public List<OrgPosition> findByOrgCodes(List<String> orgCodes) {
    if (CollectionUtils.isEmpty(orgCodes)) {
      return null;
    }
    String cacheKey = StringUtils.join(TenantUtils.getTenantCode(), orgCodes);
    List<OrgPosition> graph = cache.getIfPresent(cacheKey);
    if (graph == null) {
      graph = this.orgPositionRepository.findByOrgCodes(orgCodes, TenantUtils.getTenantCode());
      cache.put(cacheKey, graph);
    }
    return graph;
  }

  @Override
  @Transactional
  public void deleteByPositionCodes(List<String> positionCodes) {
    if (CollectionUtils.isEmpty(positionCodes)) {
      return;
    }
    this.orgPositionRepository.deleteByPositionCodes(positionCodes, TenantUtils.getTenantCode());
  }

  @Override
  public void bindPosition(OrgPositionBindDto orgPositionBindDto) {
    //1.验证关联参数
    validateRelation(orgPositionBindDto);
    //2.删除旧关联关系
    if (StringUtils.isNotBlank(orgPositionBindDto.getOldOrgCode())) {
      orgPositionRepository.removeRelationByOrgCodeAndPositionCodes(
          orgPositionBindDto.getOldOrgCode(),
          orgPositionBindDto.getPositionCodeList(), TenantUtils.getTenantCode());
    }
    //3.设置新关联关系
    List<OrgPosition> customerOrgPositions =
        orgPositionBindDto.getPositionCodeList().stream().map(positionCode -> {
          OrgPosition orgPosition = new OrgPosition();
          orgPosition.setTenantCode(TenantUtils.getTenantCode());
          orgPosition.setPositionCode(positionCode);
          orgPosition.setOrgCode(orgPositionBindDto.getOrgCode());
          return orgPosition;
        }).collect(Collectors.toList());
    orgPositionRepository.saveOrUpdateBatch(customerOrgPositions);
  }

  private void validateRelation(OrgPositionBindDto orgPositionBindDto) {
    Validate.notEmpty(orgPositionBindDto.getOrgCode(), "组织编码必须传入！");
    Validate.isTrue(!CollectionUtils.isEmpty(orgPositionBindDto.getPositionCodeList()),
        "需要关联的职位编码不能为空！");
  }

  @Override
  public void unbindPosition(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "当进行解绑操作时，业务技术编号信息必须传入");
    //删除数据
    orgPositionRepository.removeByIds(ids);
  }

  /**
   * 创建
   *
   * @param orgPosition
   * @return
   */
  private OrgPosition createForm(OrgPosition orgPosition) {
    /*
     * 对静态模型的保存操作过程为：
     * 1、如果当前模型对象不是主模型
     *  1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     *  1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     *  1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *   2.3.2、以及验证每个分组的OneToMany明细信息
     * */
    orgPosition.setTenantCode(TenantUtils.getTenantCode());
    orgPosition.setTenantCode(TenantUtils.getTenantCode());
    this.createValidation(orgPosition);
    return orgPosition;
  }

  /**
   * 校验数据
   *
   * @param orgPosition
   */
  private void createValidation(OrgPosition orgPosition) {
    Validate.notNull(orgPosition, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(orgPosition.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    orgPosition.setId(null);
    Validate.notBlank(orgPosition.getOrgCode(), "添加信息时，组织编号不能为空！");
    Validate.notBlank(orgPosition.getPositionCode(), "添加信息时，职位编号不能为空！");
    String orgCode = orgPosition.getOrgCode();
    Org org = this.orgService.findByOrgCode(orgCode);
    Validate.notNull(org, "[%s]组织不存在!", orgCode);
    String positionCode = orgPosition.getPositionCode();
    Validate.notNull(this.positionVoService, "未发现职位模块实现, 请检查");
    List<PositionVo> positionVos = this.positionVoService.findByIdsOrCodes(null, Lists.newArrayList(positionCode));
    Validate.isTrue(CollectionUtils.isNotEmpty(positionVos), "[%s]职位不存在!", positionCode);
  }
}
