package com.biz.crm.mdm.business.org.local.repository;

import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.mdm.business.org.local.entity.OrgPosition;
import com.biz.crm.mdm.business.org.local.mapper.OrgPositionMapper;
import org.apache.commons.lang3.Validate;
import org.springframework.stereotype.Repository;
import org.springframework.util.CollectionUtils;

import java.util.List;

/**
 * @author hecheng
 * @description: 组织职位关系 数据层
 * @date 2021/9/26 下午4:32
 */
@Repository
public class OrgPositionRepository extends ServiceImpl<OrgPositionMapper, OrgPosition> {

  /**
   * 通过职位code查询 职位组织关系
   *
   * @param tenantCode
   * @param positionCode
   * @return
   */
  public List<OrgPosition> findByPositionCode(String positionCode, String tenantCode) {
    return this.lambdaQuery().eq(OrgPosition::getPositionCode, positionCode)
        .eq(OrgPosition::getTenantCode, tenantCode).list();
  }

  /**
   * 通过职位code集合查询 职位组织关系
   *
   * @param tenantCode
   * @param positionCodes
   * @return
   */
  public List<OrgPosition> findByPositionCodes(List<String> positionCodes, String tenantCode) {
    return this.lambdaQuery().in(OrgPosition::getPositionCode, positionCodes)
        .eq(OrgPosition::getTenantCode, tenantCode).list();
  }

  /**
   * 按职位code删除关系
   *
   * @param orgCode
   * @param positionCode
   * @param tenantCode
   */
  public void deleteByPositionCode(String orgCode, String positionCode, String tenantCode) {
    this.lambdaUpdate()
        .eq(OrgPosition::getTenantCode, tenantCode)
        .eq(OrgPosition::getOrgCode, orgCode)
        .eq(OrgPosition::getPositionCode, positionCode)
        .remove();
  }

  /**
   * 按职位code删除关系
   *
   * @param positionCodes
   * @param tenantCode
   */
  public void deleteByPositionCodes(List<String> positionCodes, String tenantCode) {
    this.lambdaUpdate()
        .eq(OrgPosition::getTenantCode, tenantCode)
        .in(OrgPosition::getPositionCode, positionCodes)
        .remove();
  }

  /**
   * 按职位code和组织code 查询
   *
   * @param positionCode
   * @param orgCode
   * @param tenantCode
   * @return
   */
  public OrgPosition findByPositionCodeAndOrgCode(String positionCode, String orgCode, String tenantCode) {
    return this.lambdaQuery().eq(OrgPosition::getPositionCode, positionCode)
        .eq(OrgPosition::getOrgCode, orgCode)
        .eq(OrgPosition::getTenantCode, tenantCode).one();
  }

  /**
   * 按组织code查询
   *
   * @param orgCodes
   * @param tenantCode
   * @return
   */
  public List<OrgPosition> findByOrgCodes(List<String> orgCodes, String tenantCode) {
    return this.lambdaQuery().in(OrgPosition::getOrgCode, orgCodes)
        .eq(OrgPosition::getTenantCode, tenantCode).list();
  }

  public void removeRelationByOrgCodeAndPositionCodes(String orgCode, List<String> positionCodeList, String tenantCode) {
    Validate.notEmpty(orgCode, "组织编码必须传入！");
    Validate.isTrue(!CollectionUtils.isEmpty(positionCodeList),
            "需要关联的职位编码不能为空！");
    this.lambdaUpdate()
            .eq(OrgPosition::getTenantCode, tenantCode)
            .eq(OrgPosition::getOrgCode, orgCode)
            .in(OrgPosition::getPositionCode, positionCodeList)
            .remove();
  }
}
