package com.biz.crm.mdm.business.position.level.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.position.level.local.service.PositionLevelRoleService;
import com.biz.crm.mdm.business.position.level.local.entity.PositionLevelEntity;
import com.biz.crm.mdm.business.position.level.local.entity.PositionLevelRoleEntity;
import com.biz.crm.mdm.business.position.level.local.service.PositionLevelService;
import com.biz.crm.mdm.business.position.level.sdk.dto.PositionLevelDto;
import com.biz.crm.mdm.business.position.level.sdk.service.PositionLevelVoService;
import com.biz.crm.mdm.business.position.level.sdk.vo.PositionLevelVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.rbac.sdk.service.RoleVoCacheService;
import com.bizunited.nebula.rbac.sdk.vo.RoleVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 职位级别 接口实现
 *
 * @author Tao.Chen
 * @date 2020-10-22 10:56:51
 */
@Service
public class PositionLevelVoServiceImpl implements PositionLevelVoService {

  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private PositionLevelRoleService positionLevelRoleService;
  @Autowired(required = false)
  private PositionLevelService positionLevelService;
  @Autowired(required = false)
  private RoleVoCacheService roleVoCacheService;

  @Override
  public Page<PositionLevelVo> findByConditions(Pageable pageable, PositionLevelDto dto) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    dto = ObjectUtils.defaultIfNull(dto, new PositionLevelDto());
    dto.setTenantCode(TenantUtils.getTenantCode());
    Page<PositionLevelEntity> entityPage = this.positionLevelService.findByConditions(pageable, dto);
    Page<PositionLevelVo> pageResult = new Page<>(entityPage.getCurrent(), entityPage.getSize(), entityPage.getTotal());
    if (CollectionUtils.isNotEmpty(entityPage.getRecords())) {
      pageResult.setRecords(convertEntityToVo(entityPage.getRecords()));
    }
    return pageResult;
  }

  @Override
  public PositionLevelVo findById(String id) {
    if (StringUtils.isEmpty(id)) {
      return null;
    }
    List<PositionLevelEntity> entities = this.positionLevelService.findByIdsOrCodes(Lists.newArrayList(id), null);
    if (CollectionUtils.isEmpty(entities)) {
      return null;
    }
    return convertEntityToVo(Lists.newArrayList(entities)).get(0);
  }

  @Override
  public List<PositionLevelVo> findByIdsOrCodes(List<String> ids, List<String> positionLevelCodes) {
    if (CollectionUtils.isEmpty(ids) && CollectionUtils.isEmpty(positionLevelCodes)) {
      return Lists.newArrayList();
    }
    List<PositionLevelEntity> entities = this.positionLevelService.findByIdsOrCodes(ids, positionLevelCodes);
    if (CollectionUtils.isEmpty(entities)) {
      return Lists.newArrayList();
    }
    return convertEntityToVo(entities);
  }

  @Override
  @Transactional
  public String createPositionNameByPositionLevelCode(String positionLevelCode, Set<String> positionNameSet, String tenantCode) {
    //TODO 分布式锁
    Validate.notBlank(positionLevelCode, "缺失职位级别编码");
    List<PositionLevelEntity> entities = this.positionLevelService.findByIdsOrCodes(null, Lists.newArrayList(positionLevelCode));
    Validate.isTrue(CollectionUtils.isNotEmpty(entities), "无效的职位级别");
    PositionLevelEntity positionLevel = entities.get(0);
    Validate.isTrue(EnableStatusEnum.ENABLE.getCode().equals(positionLevel.getEnableStatus()), "该职位级别已停用");
    Integer suffixSequence = positionLevel.getSuffixSequence();
    if (suffixSequence == null) {
      suffixSequence = 0;
    }
    String positionName = positionLevel.getPositionLevelName() + "_" + (++suffixSequence);
    while (true) {
      if (CollectionUtils.isEmpty(positionNameSet) || !positionNameSet.contains(positionName)) {
        break;
      }
      positionName = positionLevel.getPositionLevelName() + "_" + (++suffixSequence);
    }
    this.positionLevelService.updateSuffixSequenceById(suffixSequence, positionLevel.getId());
    return positionName;
  }

  /**
   * 职位级别实体列表转VO列表
   *
   * @param entities 职位实体列表
   * @return 职位VO列表
   */
  private List<PositionLevelVo> convertEntityToVo(List<PositionLevelEntity> entities) {
    List<PositionLevelVo> list = Lists.newArrayList(this.nebulaToolkitService.copyCollectionByWhiteList(entities, PositionLevelEntity.class
        , PositionLevelVo.class, HashSet.class, ArrayList.class));
    List<String> positionLevelCodes = list.stream().map(PositionLevelVo::getPositionLevelCode).collect(Collectors.toList());
    List<PositionLevelRoleEntity> positionRoles = this.positionLevelRoleService.findByPositionLevelCodes(positionLevelCodes);
    if (CollectionUtils.isNotEmpty(positionRoles)) {
      //封装职级关联角色编码映射
      Map<String, List<String>> positionRoleMap = positionRoles.stream().collect(Collectors
          .groupingBy(PositionLevelRoleEntity::getPositionLevelCode, Collectors.mapping(PositionLevelRoleEntity::getRoleCode, Collectors.toList())));
      //封装角色编码关联角色名称映射(合并所有角色编码)
      Set<String> roleCodes = positionRoleMap.values().stream().flatMap(Collection::stream).collect(Collectors.toSet());
      Set<RoleVo> roleVoSet = roleVoCacheService.findByTenantCodeAndRoleCodes(TenantUtils.getTenantCode(), roleCodes);
      Map<String, String> roleNameMap = Maps.newHashMap();
      if (CollectionUtils.isNotEmpty(roleVoSet)) {
        roleNameMap = roleVoSet.stream().collect(Collectors.toMap(RoleVo::getRoleCode, RoleVo::getRoleName, (key1, key2) -> key1));
      }
      //封装职级VO角色相关字段信息
      for (PositionLevelVo positionLevelVo : list) {
        List<String> roleCodeList = positionRoleMap.get(positionLevelVo.getPositionLevelCode());
        if (CollectionUtils.isEmpty(roleCodeList)) {
          continue;
        }
        List<String> roleNames = roleCodeList.stream().map(roleNameMap::get).filter(Objects::nonNull).collect(Collectors.toList());
        positionLevelVo.setRoleList(roleCodeList);
        positionLevelVo.setRoleCode(String.join(",", positionLevelVo.getRoleList()));
        if (CollectionUtils.isNotEmpty(roleNames)) {
          positionLevelVo.setRoleName(String.join(",", roleNames));
        }
      }
    }
    return list;
  }
}
