package com.biz.crm.mdm.business.position.level.local.service.internal;

import com.biz.crm.mdm.business.position.level.local.repository.PositionLevelRoleRepository;
import com.biz.crm.mdm.business.position.level.local.service.PositionLevelRoleService;
import com.biz.crm.mdm.business.position.level.local.entity.PositionLevelRoleEntity;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.rbac.sdk.service.RoleVoCacheService;
import com.google.common.collect.Lists;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.stream.Collectors;

/**
 * 职位级别-角色关联表接口实现
 *
 * @author Tao.Chen
 * @date 2020-11-17 20:23:13
 */
@Service
public class PositionLevelRoleServiceImpl implements PositionLevelRoleService {

  @Autowired(required = false)
  private PositionLevelRoleRepository positionLevelRoleRepository;
  @Autowired(required = false)
  private RoleVoCacheService roleVoCacheService;

  @Override
  @Transactional
  public void rebindByPositionLevelCodeAndRoleCodes(String positionLevelCode, List<String> roleCodes) {
    Validate.notBlank(positionLevelCode, "职位级别编码不能为空");
    String tenantCode = TenantUtils.getTenantCode();
    this.unbindByPositionLevelCode(positionLevelCode);
    //传入roleCodes为空,视作解除当前职位级别的角色关联信息
    if (CollectionUtils.isNotEmpty(roleCodes)) {
      return;
    }
    List<PositionLevelRoleEntity> roleEntities = roleCodes.stream().filter(StringUtils::isNotEmpty).map(roleCode -> {
      PositionLevelRoleEntity entity = new PositionLevelRoleEntity();
      entity.setPositionLevelCode(positionLevelCode);
      entity.setRoleCode(roleCode);
      entity.setTenantCode(tenantCode);
      return entity;
    }).collect(Collectors.toList());
    if (CollectionUtils.isNotEmpty(roleEntities)) {
      this.positionLevelRoleRepository.saveBatch(roleEntities);
      //角色关联职位之别信息发生改变,清理角色缓存
      roleCodes.forEach(roleCode -> {
        this.roleVoCacheService.clearCache(tenantCode, roleCode);
      });
    }
  }

  @Override
  @Transactional
  public void bindByPositionLevelCodeAndRoleCodes(String positionLevelCode, List<String> roleCodes) {
    Validate.notBlank(positionLevelCode, "职位级别编码不能为空");
    String tenantCode = TenantUtils.getTenantCode();
    //传入roleCodes为空,视为不绑定关系
    if (CollectionUtils.isEmpty(roleCodes)) {
      return;
    }
    List<PositionLevelRoleEntity> list = positionLevelRoleRepository
        .findByLevelCodeAndRoleCodeIn(positionLevelCode, roleCodes, tenantCode);
    List<String> existList = CollectionUtils.isNotEmpty(list)
        ? list.stream().map(PositionLevelRoleEntity::getRoleCode).collect(Collectors.toList()) : Lists.newArrayList();
    List<PositionLevelRoleEntity> roleEntities = roleCodes.stream().filter(x -> !existList.contains(x)).map(roleCode -> {
      PositionLevelRoleEntity entity = new PositionLevelRoleEntity();
      entity.setRoleCode(roleCode);
      entity.setPositionLevelCode(positionLevelCode);
      entity.setTenantCode(tenantCode);
      return entity;
    }).collect(Collectors.toList());
    if (CollectionUtils.isNotEmpty(roleEntities)) {
      this.positionLevelRoleRepository.saveBatch(roleEntities);
      //角色关联职位之别信息发生改变,清理角色缓存
      roleCodes.forEach(roleCode -> {
        this.roleVoCacheService.clearCache(tenantCode, roleCode);
      });
    }
  }

  @Override
  @Transactional
  public void unbindByPositionLevelCode(String positionLevelCode) {
    Validate.notBlank(positionLevelCode, "职位级别编码不能为空");
    this.unbindByPositionLevelCodeIn(Lists.newArrayList(positionLevelCode));
  }

  @Override
  @Transactional
  public void unbindByPositionLevelCodeIn(List<String> positionLevelCodes) {
    Validate.isTrue(CollectionUtils.isNotEmpty(positionLevelCodes), "职位级别编码不能为空");
    String tenantCode = TenantUtils.getTenantCode();
    List<PositionLevelRoleEntity> entities = this.positionLevelRoleRepository
        .findByPositionLevelCodeIn(positionLevelCodes, tenantCode);
    if (CollectionUtils.isEmpty(entities)) {
      return;
    }
    this.positionLevelRoleRepository.deleteByLevelCodeIn(positionLevelCodes, tenantCode);
    //角色关联职位之别信息发生改变,清理角色缓存
    entities.forEach(entity -> {
      this.roleVoCacheService.clearCache(tenantCode, entity.getRoleCode());
    });
  }

  @Override
  @Transactional
  public void unbindByRoleCode(String roleCode) {
    Validate.notBlank(roleCode, "角色编码不能为空");
    this.unbindByRoleCodeIn(Lists.newArrayList(roleCode));
  }

  @Override
  @Transactional
  public void unbindByRoleCodeIn(List<String> roleCodes) {
    Validate.isTrue(CollectionUtils.isNotEmpty(roleCodes), "角色编码不能为空");
    String tenantCode = TenantUtils.getTenantCode();
    this.positionLevelRoleRepository.deleteByRoleCodeIn(roleCodes, tenantCode);
    //角色关联职位之别信息发生改变,清理角色缓存
    roleCodes.forEach(roleCode -> {
      this.roleVoCacheService.clearCache(tenantCode, roleCode);
    });
  }

  @Override
  public List<PositionLevelRoleEntity> findByPositionLevelCodes(List<String> positionLevelCodes) {
    if (CollectionUtils.isEmpty(positionLevelCodes)) {
      return Lists.newArrayList();
    }
    String tenantCode = TenantUtils.getTenantCode();
    return this.positionLevelRoleRepository.findByPositionLevelCodeIn(positionLevelCodes, tenantCode);
  }

  @Override
  public List<PositionLevelRoleEntity> findByRoleCodes(List<String> roleCodes) {
    if (CollectionUtils.isEmpty(roleCodes)) {
      return Lists.newArrayList();
    }
    return this.positionLevelRoleRepository.findByRoleCodes(roleCodes, TenantUtils.getTenantCode());
  }
}
