package com.biz.crm.mdm.business.position.local.controller;

import java.util.List;

import com.biz.crm.mn.third.system.master.data.mdg.sdk.dto.MasterDataMdgBaseDto;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.mdm.business.position.local.entity.PositionEntity;
import com.biz.crm.mdm.business.position.local.service.PositionRoleService;
import com.biz.crm.mdm.business.position.local.service.PositionService;
import com.biz.crm.mdm.business.position.sdk.dto.BindPositionRoleDto;
import com.biz.crm.mdm.business.position.sdk.dto.PositionDto;
import com.biz.crm.mdm.business.position.sdk.dto.RebindParentPositionDto;
import com.biz.crm.mdm.business.position.sdk.dto.UnbindPositionRoleDto;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;

/**
 * 职位管理: Position: 职位信息
 *
 * @author ning.zhang
 * @date 2021/10/15
 */
@Slf4j
@RestController
@RequestMapping("/v1/position/position")
@Api(tags = "职位管理: Position: 职位信息")
public class PositionController {

  @Autowired(required = false)
  private PositionService positionService;
  @Autowired(required = false)
  private PositionRoleService positionRoleService;

  /**
   * 批量拉取 MDG 职位级别数据
   *
   * @param dto 请求参数
   * @return 供应商列表
   */
  @ApiOperation(value = "批量拉取 MDG 职位数据")
  @PostMapping(value = {"/pullPositionList"})
  public Result<?> pullPositionList(@RequestBody MasterDataMdgBaseDto dto) {
    try {
      this.positionService.pullPositionList(dto);
      this.positionService.updateRuleCode();
      return Result.ok("批量拉取 MDG 职位数据成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 创建职位
   *
   * @param dto 请求参数dto
   * @return 职位实体
   */
  @ApiOperation(value = "新增职位")
  @PostMapping("/create")
  public Result<PositionEntity> create(@RequestBody PositionDto dto) {
    try {
      return Result.ok(this.positionService.create(dto));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 更新职位
   *
   * @param dto 请求参数dto
   * @return 职位实体
   */
  @ApiOperation(value = "更新职位")
  @PatchMapping("/update")
  public Result<PositionEntity> update(@RequestBody PositionDto dto) {
    try {
      return Result.ok(this.positionService.update(dto));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 批量删除职位
   *
   * @param ids 职位id集合
   */
  @ApiOperation(value = "删除")
  @DeleteMapping("/deleteBatch")
  public Result<?> deleteBatch(@RequestParam("ids") List<String> ids) {
    try {
      this.positionService.deleteBatch(ids);
      return Result.ok("删除成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 批量启用职位
   *
   * @param ids 职位id集合
   */
  @ApiOperation(value = "启用")
  @PatchMapping("/enableBatch")
  public Result<?> enableBatch(@RequestBody List<String> ids) {
    try {
      this.positionService.enableBatch(ids);
      return Result.ok("启用成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 批量禁用职位
   *
   * @param ids 职位id集合
   */
  @ApiOperation(value = "禁用")
  @PatchMapping("/disableBatch")
  public Result<?> disableBatch(@RequestBody List<String> ids) {
    try {
      this.positionService.disableBatch(ids);
      return Result.ok("禁用成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 重置降维编码
   */
  @ApiOperation(value = "重置降维编码")
  @PatchMapping("/updateRuleCode")
  public Result<?> updateRuleCode() {
    try {
      this.positionService.updateRuleCode();
      return Result.ok();
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 通过职位编码集合和职位编码绑定职位角色关联关系
   *
   * @param dto 请求参数dto
   */
  @ApiOperation(value = "绑定角色")
  @PatchMapping("/bindRole")
  public Result<?> bindRole(@RequestBody BindPositionRoleDto dto) {
    try {
      this.positionRoleService.bindByPositionCodesAndRoleCode(dto);
      return Result.ok("绑定成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 通过职位编码集合和职位编码解绑职位角色关联关系
   *
   * @param dto 请求参数dto
   */
  @ApiOperation(value = "解绑角色")
  @PatchMapping("/unbindRole")
  public Result<?> unbindRole(@RequestBody UnbindPositionRoleDto dto) {
    try {
      this.positionRoleService.unbindByPositionCodesAndRoleCode(dto.getPositionCodeList(),dto.getRoleCode());
      return Result.ok("解绑成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 重新绑定上级职位
   * @param dto 请求参数dto
   */
  @ApiOperation(value = "重绑上级职位")
  @PatchMapping("/rebindParentPosition")
  public Result<?> rebindParentPosition(@RequestBody RebindParentPositionDto dto) {
    try {
      this.positionService.rebindParentPosition(dto);
      return Result.ok("重绑成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }
}
