package com.biz.crm.mdm.business.position.local.repository;

import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.local.entity.UuidFlagOpEntity;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.position.local.entity.PositionEntity;
import com.biz.crm.mdm.business.position.local.mapper.PositionMapper;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Objects;

/**
 * 职位的数据库访问类 {@link PositionEntity}
 *
 * @author ning.zhang
 * @date 2021/9/28
 */
@Component
public class PositionRepository extends ServiceImpl<PositionMapper, PositionEntity> {

  @Autowired(required = false)
  private PositionMapper positionMapper;

  /**
   * 通过删除标识和职位级别编码集合获取职位列表
   *
   * @param positionLevelCodes 职位级别编码集合
   * @param tenantCode         租户编码
   * @return 职位列表
   */
  public List<PositionEntity> findByPositionLevelCodes(List<String> positionLevelCodes, String tenantCode) {
    return this.lambdaQuery()
            .eq(PositionEntity::getTenantCode, tenantCode)
            .eq(PositionEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
            .in(PositionEntity::getPositionLevelCode, positionLevelCodes)
            .list();
  }

  /**
   * 根据职位编码获取职位
   *
   * @param positionCode 职位编码
   * @param tenantCode   租户编码
   * @return 职位
   */
  public PositionEntity findByPositionCode(String positionCode, String tenantCode) {
    return this.lambdaQuery()
            .eq(PositionEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
            .eq(PositionEntity::getTenantCode, tenantCode)
            .eq(PositionEntity::getPositionCode, positionCode)
            .one();
  }

  /**
   * 通过职位名称模糊查询职位列表
   *
   * @param positionName 职位名称
   * @param tenantCode   租户编码
   * @return 职位列表
   */
  public List<PositionEntity> findByPositionNameLike(String positionName, String tenantCode) {
    return this.lambdaQuery()
            .eq(PositionEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
            .eq(PositionEntity::getTenantCode, tenantCode)
            .like(PositionEntity::getPositionName, positionName)
            .select(PositionEntity::getPositionName)
            .list();
  }

  /**
   * 通过父职位编码查询职位列表
   *
   * @param parentCode 父职位编码
   * @param tenantCode 租户编码
   * @return 职位列表
   */
  public List<PositionEntity> findByParentCode(String parentCode, String tenantCode) {
    return this.lambdaQuery()
            .eq(PositionEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
            .eq(PositionEntity::getParentCode, parentCode)
            .eq(PositionEntity::getTenantCode, tenantCode)
            .list();
  }

  /**
   * 通过父职位编码集合查询职位列表
   *
   * @param parentCodes 父职位编码集合
   * @param tenantCode  租户编码
   * @return 职位列表
   */
  public List<PositionEntity> findByParentCodes(List<String> parentCodes, String tenantCode) {
    return this.lambdaQuery()
            .eq(PositionEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
            .in(PositionEntity::getParentCode, parentCodes)
            .eq(PositionEntity::getTenantCode, tenantCode)
            .list();
  }

  /**
   * 查询父职位编码是null的职位列表
   *
   * @param tenantCode 租户编码
   * @return 职位列表
   */
  public List<PositionEntity> findByParentCodeIsNull(String tenantCode) {
    return this.lambdaQuery()
            .isNull(PositionEntity::getParentCode).or().eq(PositionEntity::getParentCode, "")
            .eq(PositionEntity::getTenantCode, tenantCode)
            .eq(PositionEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
            .list();
  }

  /**
   * 根据ID更新规则code和职位等级
   *
   * @param positionCode 职位编码
   * @param ruleCode     规则code
   * @param levelNum     层级等级
   * @param tenantCode   租户编码
   */
  public void updateRuleCodeAndLevelNumByPositionCode(String positionCode, String ruleCode, Integer levelNum, String tenantCode) {
    this.lambdaUpdate()
            .eq(PositionEntity::getPositionCode, positionCode)
            .eq(PositionEntity::getTenantCode, tenantCode)
            .set(PositionEntity::getRuleCode, ruleCode)
            .set(PositionEntity::getLevelNum, levelNum)
            .update();
  }

  /**
   * 根据职位id集合或者编码集合获取职位列表
   *
   * @param ids           职位id集合
   * @param positionCodes 职位编码集合
   * @param tenantCode    租户编码
   * @return 职位列表
   */
  public List<PositionEntity> findByIdsOrCodes(List<String> ids, List<String> positionCodes, String tenantCode) {
    if(CollectionUtils.isNotEmpty(ids)) {
      ids.removeIf(Objects::isNull);
    }
    if(CollectionUtils.isNotEmpty(positionCodes)) {
      positionCodes.removeIf(Objects::isNull);
    }
    return this.lambdaQuery()
            .eq(PositionEntity::getTenantCode, tenantCode)
            .eq(PositionEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
            .in(CollectionUtils.isNotEmpty(positionCodes), PositionEntity::getPositionCode, positionCodes)
            .in(CollectionUtils.isNotEmpty(ids), PositionEntity::getId, ids)
            .list();
  }

  /**
   * 批量删除职位级别
   *
   * @param ids 职位ID集合
   */
  public void updateDelFlagByIds(List<String> ids) {
    this.lambdaUpdate()
            .in(PositionEntity::getId, ids)
            .eq(PositionEntity::getTenantCode, TenantUtils.getTenantCode())//设置租户编号信息
            .set(PositionEntity::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
            .update();
  }

  /**
   * 批量更新启用禁用状态
   *
   * @param ids          职位ID集合
   * @param enableStatus 启用禁用状态
   */
  public void updateEnableStatusByIds(List<String> ids, EnableStatusEnum enableStatus) {
    this.lambdaUpdate()
            .in(PositionEntity::getId, ids)
            .eq(PositionEntity::getTenantCode, TenantUtils.getTenantCode())//设置租户编号信息
            .set(PositionEntity::getEnableStatus, enableStatus.getCode())
            .update();
  }

  /**
   * 按降维码集合和状态查询职位列表
   *
   * @param ruleCodes    降维编码集合
   * @param enableStatus 启用状态
   * @param tenantCode   租户编码
   * @return 职位列表
   */
  public List<PositionEntity> findByRuleCodesAndEnableStatus(List<String> ruleCodes, String enableStatus, String tenantCode) {
    return this.lambdaQuery()
            .eq(StringUtils.isNotBlank(enableStatus), PositionEntity::getEnableStatus, enableStatus)
            .in(PositionEntity::getRuleCode, ruleCodes)
            .eq(PositionEntity::getTenantCode, tenantCode)
            .eq(PositionEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
            .list();
  }

  /**
   * 更具职位规则编码集合和启用状态查询当前职位和其所有子级职位列表
   *
   * @param ruleCodes    降维编码集合
   * @param enableStatus 启用状态
   * @param tenantCode   租户编码
   * @return 所有子级职位列表
   */
  public List<PositionEntity> findCurAndChildrenByRuleCodes(List<String> ruleCodes, String enableStatus, String tenantCode) {
    return positionMapper.findCurAndChildrenByRuleCodes(ruleCodes, enableStatus, tenantCode);
  }

  /**
   * 查找parentCode不为空但找不到对应上级的数据,设置parentCode为null
   *
   * @param tenantCode 租户编码
   */
  public void updateOrphanParentCodeNull(String tenantCode) {
    this.positionMapper.updateOrphanParentCodeNull(tenantCode);
  }

  /**
   * 按ruleCode查询所有子节点(包含自身)
   *
   * @param ruleCode   降维编码
   * @param tenantCode 租户编码
   * @return 所有子节点
   */
  public List<PositionEntity> findAllChildrenByRuleCode(String ruleCode, String tenantCode) {
    return this.lambdaQuery()
            .likeRight(PositionEntity::getRuleCode, ruleCode)
            .eq(PositionEntity::getTenantCode, tenantCode)
            .eq(PositionEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
            .list();
  }

  /**
   * 通过角色编码集合获取职位信息
   *
   * @param roleCodes  角色编码集合
   * @param tenantCode 租户编码
   * @return 职位实体集合
   */
  public List<PositionEntity> findByRoleCodes(List<String> roleCodes, String tenantCode) {
    return this.baseMapper.findByRoleCodes(roleCodes, tenantCode);
  }

  /**
   * 通过id和租户编号修改职位信息
   * @param updateEntity
   * @param tenantCode
   */
  public void updateByIdAndTenantCode(PositionEntity updateEntity, String tenantCode) {
    LambdaUpdateWrapper<PositionEntity>lambdaUpdateWrapper= Wrappers.lambdaUpdate();
    lambdaUpdateWrapper.eq(PositionEntity::getTenantCode,tenantCode);
    lambdaUpdateWrapper.in(PositionEntity::getId,updateEntity.getId());
    this.baseMapper.update(updateEntity,lambdaUpdateWrapper);
  }

  /**
   * 根据id和租户编号修改职位状态
   * @param disableEntities
   * @param tenantCode
   */
  public void updateBatchByIdAndTenantCode(List<PositionEntity> disableEntities, String tenantCode) {
    LambdaUpdateWrapper<PositionEntity>lambdaUpdateWrapper= Wrappers.lambdaUpdate();
    disableEntities.stream().forEach(positionEntity -> {
      lambdaUpdateWrapper.eq(PositionEntity::getTenantCode,tenantCode);
      lambdaUpdateWrapper.in(PositionEntity::getId,positionEntity.getId());
      this.baseMapper.update(positionEntity,lambdaUpdateWrapper);
    });
  }

  /**
   * 重构查询方法
   * @param id
   * @param tenantCode
   * @return
   */
  public PositionEntity findByIdAndTenantCode(String id, String tenantCode) {
    return this.lambdaQuery()
        .eq(PositionEntity::getTenantCode,tenantCode)
        .in(PositionEntity::getId,id)
        .one();

  }
    /**
     * 根据职位批量查询
     *
     * @author huojia
     * @date 2022/12/10 15:38
     * @param positionCodeList
     * @return java.util.List<com.biz.crm.mdm.business.position.local.entity.PositionEntity>
     **/
    public List<PositionEntity> findByPositionCodes(List<String> positionCodeList) {
      if (CollectionUtils.isEmpty(positionCodeList)) {
        return null;
      }
      return this.lambdaQuery()
              .in(PositionEntity::getPositionCode, positionCodeList)
              .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
              .list();
    }

    /**
     * 查询有上级用户的职位
     *
     * @author huojia
     * @date 2022/12/26 16:49
     * @return java.util.List<com.biz.crm.mdm.business.position.local.entity.PositionEntity>
     **/
    public List<PositionEntity> listParentUser() {
      return this.lambdaQuery()
              //.isNotNull(PositionEntity::getParentUserName)
              .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
              .list();
    }

    /**
     * 根据mdg职位编码查询
     *
     * @author huojia
     * @date 2023/1/3 11:32
     * @param mdgPositionCodeList
     * @return java.util.List<com.biz.crm.mdm.business.position.local.entity.PositionEntity>
     **/
    public List<PositionEntity> findByMdgPositionCodes(List<String> mdgPositionCodeList) {
      if (CollectionUtils.isEmpty(mdgPositionCodeList)) {
        return null;
      }
      return this.lambdaQuery()
              .in(PositionEntity::getMdgPositionCode, mdgPositionCodeList)
              .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
              .list();
    }
}
