package com.biz.crm.mdm.business.position.local.service.internal;

import com.biz.crm.mdm.business.position.local.entity.PositionRoleEntity;
import com.biz.crm.mdm.business.position.local.repository.PositionRoleRepository;
import com.biz.crm.mdm.business.position.local.service.PositionRoleService;
import com.biz.crm.mdm.business.position.sdk.dto.BindPositionRoleDto;
import com.biz.crm.mdm.business.position.sdk.dto.PositionDto;
import com.biz.crm.mdm.business.position.sdk.dto.PositionRoleDto;
import com.biz.crm.mdm.business.position.sdk.enums.IndateTypeEnum;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.rbac.sdk.service.RoleVoCacheService;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.stream.Collectors;

/**
 * 职位-角色关联表接口实现
 *
 * @author ning.zhang
 * @date 2021/10/13
 */
@Service
public class PositionRoleServiceImpl implements PositionRoleService {

  @Autowired(required = false)
  private PositionRoleRepository positionRoleRepository;
  @Autowired(required = false)
  private RoleVoCacheService roleVoCacheService;

  @Override
  public List<PositionRoleEntity> findByPositionCode(String positionCode) {
    if (StringUtils.isEmpty(positionCode)) {
      return Lists.newArrayList();
    }
    return positionRoleRepository.findByPositionCode(positionCode, TenantUtils.getTenantCode());
  }

  @Override
  public List<PositionRoleEntity> findByPositionCodes(List<String> positionCodes) {
    if (CollectionUtils.isEmpty(positionCodes)) {
      return Lists.newArrayList();
    }
    return positionRoleRepository.findByPositionCodes(positionCodes, TenantUtils.getTenantCode());
  }

  @Override
  public void unbindByPositionCodes(List<String> positionCodes) {
    Validate.isTrue(CollectionUtils.isNotEmpty(positionCodes), "职位编码不能为空");
    String tenantCode = TenantUtils.getTenantCode();
    List<PositionRoleEntity> entities = positionRoleRepository.findByPositionCodes(positionCodes, tenantCode);
    if (CollectionUtils.isEmpty(entities)) {
      return;
    }
    this.positionRoleRepository.deleteByPositionCodes(positionCodes, tenantCode);
    //角色关联职位信息发生改变,清理角色缓存
    entities.forEach(entity -> {
      this.roleVoCacheService.clearCache(tenantCode, entity.getRoleCode());
    });
  }

  @Override
  public void bindByPositionCodeAndRoleCodes(String positionCode, PositionDto dto) {
    Validate.notBlank(positionCode, "职位级别编码不能为空");
    Validate.notNull(dto, "职位信息不能为空");
    if (CollectionUtils.isEmpty(dto.getRoleCodeList()) && CollectionUtils.isEmpty(dto.getPositionRoles())) {
      return;
    }
    List<PositionRoleEntity> roleEntities = Lists.newArrayList();
    String tenantCode = TenantUtils.getTenantCode();
    if (!CollectionUtils.isEmpty(dto.getRoleCodeList())) {
      roleEntities.addAll(dto.getRoleCodeList().stream().map(roleCode -> {
        PositionRoleEntity entity = new PositionRoleEntity();
        entity.setRoleCode(roleCode);
        entity.setPositionCode(positionCode);
        entity.setTenantCode(tenantCode);
        return entity;
      }).collect(Collectors.toList()));
    } else {
      this.validationData(dto.getPositionRoles());
      roleEntities.addAll(dto.getPositionRoles().stream().map(roleDto -> {
        PositionRoleEntity entity = new PositionRoleEntity();
        entity.setEndTime(roleDto.getEndTime());
        entity.setStartTime(roleDto.getStartTime());
        entity.setIndateType(roleDto.getIndateType());
        entity.setRoleName(roleDto.getRoleName());
        entity.setPositionCode(positionCode);
        entity.setRoleCode(roleDto.getRoleCode());
        entity.setTenantCode(tenantCode);
        entity.setRemark(roleDto.getRemark());
        return entity;
      }).collect(Collectors.toList()));
    }
    this.positionRoleRepository.saveBatch(roleEntities);
    //角色关联职位之别信息发生改变,清理角色缓存
    roleEntities.forEach(roleEntity -> {
      this.roleVoCacheService.clearCache(tenantCode, roleEntity.getRoleCode());
    });
  }

  @Override
  public List<PositionRoleEntity> findByRoleCodes(List<String> roleCodes) {
    if (CollectionUtils.isEmpty(roleCodes)) {
      return Lists.newArrayList();
    }
    return this.positionRoleRepository.findByRoleCodes(roleCodes, TenantUtils.getTenantCode());
  }

  @Override
  public void bindByPositionCodesAndRoleCode(BindPositionRoleDto dto) {
    Validate.notNull(dto, "绑定信息不能为空");
    Validate.notNull(dto.getRoleCode(), "缺失角色编码");
    Validate.isTrue(!CollectionUtils.isEmpty(dto.getPositionRoles()) || !CollectionUtils.isEmpty(dto.getPositionCodeList()),"缺失职位编码");
    String roleCode = dto.getRoleCode();
    String tenantCode = TenantUtils.getTenantCode();
    List<PositionRoleEntity> entities = Lists.newArrayList();
    if(!CollectionUtils.isEmpty(dto.getPositionCodeList())){
      List<String> positionCodes = dto.getPositionCodeList();
      List<PositionRoleEntity> list = this.positionRoleRepository.findByPositionCodesAndRoleCode(positionCodes, roleCode, tenantCode);
      //过滤掉已经存在的关联
      if (CollectionUtils.isNotEmpty(list)) {
        List<String> existPositionCodes = list.stream().map(PositionRoleEntity::getPositionCode).collect(Collectors.toList());
        positionCodes = positionCodes.stream().filter(s -> !existPositionCodes.contains(s)).collect(Collectors.toList());
      }
      if (CollectionUtils.isEmpty(positionCodes)) {
        return;
      }
      //已经没有需要绑定的职位和角色关系,直接返回
      entities.addAll(positionCodes.stream().map(positionCode -> {
        PositionRoleEntity entity = new PositionRoleEntity();
        entity.setTenantCode(tenantCode);
        entity.setRoleCode(roleCode);
        entity.setPositionCode(positionCode);
        return entity;
      }).collect(Collectors.toList()));
    }else {
      List<PositionRoleDto> positionRoles = dto.getPositionRoles();
      this.validationData(positionRoles);
      List<String> positionCodes = positionRoles.stream().map(PositionRoleDto::getPositionCode).collect(Collectors.toList());
      List<PositionRoleEntity> list = this.positionRoleRepository.findByPositionCodesAndRoleCode(positionCodes, roleCode, tenantCode);
      //过滤掉已经存在的关联
      if (CollectionUtils.isNotEmpty(list)) {
        List<String> existPositionCodes = list.stream().map(PositionRoleEntity::getPositionCode).collect(Collectors.toList());
        positionCodes = positionCodes.stream().filter(s -> !existPositionCodes.contains(s)).collect(Collectors.toList());
      }
      if (CollectionUtils.isEmpty(positionCodes)) {
        return;
      }
      //已经没有需要绑定的职位和角色关系,直接返回
      for (PositionRoleDto roleDto : positionRoles) {
        if(positionCodes.contains(roleDto.getPositionCode())){
          PositionRoleEntity entity = new PositionRoleEntity();
          entity.setEndTime(roleDto.getEndTime());
          entity.setStartTime(roleDto.getStartTime());
          entity.setIndateType(roleDto.getIndateType());
          entity.setRoleName(roleDto.getRoleName());
          entity.setPositionCode(roleDto.getPositionCode());
          entity.setTenantCode(tenantCode);
          entity.setRoleCode(roleCode);
          entity.setRemark(roleDto.getRemark());
          entities.add(entity);
        }
      }
    }
    this.positionRoleRepository.saveBatch(entities);
  }

  @Override
  public void unbindByPositionCodesAndRoleCode(List<String> positionCodes, String roleCode) {
    Validate.isTrue(CollectionUtils.isNotEmpty(positionCodes), "缺失职位编码");
    Validate.notBlank(roleCode, "缺失角色编码");
    List<PositionRoleEntity> list = this.positionRoleRepository.findByPositionCodesAndRoleCode(positionCodes, roleCode, TenantUtils.getTenantCode());
    //没有需要解绑的数据,直接返回
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    List<String> positionRoleIds = list.stream().map(PositionRoleEntity::getId).collect(Collectors.toList());
    this.positionRoleRepository.removeByIdsAndTenantCode(positionRoleIds,TenantUtils.getTenantCode());
  }


  private void validationData(List<PositionRoleDto> positionRoles) {
    Validate.notNull(positionRoles, "进行当前操作时，信息对象必须传入!");
    for (PositionRoleDto dto : positionRoles) {
      Validate.notBlank(dto.getIndateType(), "有效期限不能为空！");
      if(IndateTypeEnum.CUSTOMIZE_TIME.getDictCode().equals(dto.getIndateType())){
        Validate.notNull(dto.getStartTime(), "开始时间不能为空！");
        Validate.notNull(dto.getEndTime(), "结束时间不能为空！");
        Validate.isTrue(dto.getStartTime().before(dto.getEndTime()), "生效开始时间必须小于生效结束时间");
      }
    }
  }
}
