package com.biz.crm.mdm.business.position.local.service.internal;

import com.alibaba.fastjson.JSON;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.TreeRuleCodeStrategy;
import com.biz.crm.business.common.sdk.service.TreeRuleCodeStrategyHolder;
import com.biz.crm.mdm.business.position.local.entity.PositionEntity;
import com.biz.crm.mdm.business.position.local.entity.PositionRoleEntity;
import com.biz.crm.mdm.business.position.local.repository.PositionRepository;
import com.biz.crm.mdm.business.position.local.service.PositionRoleService;
import com.biz.crm.mdm.business.position.local.service.PositionService;
import com.biz.crm.mdm.business.position.sdk.constant.PositionConstant;
import com.biz.crm.mdm.business.position.sdk.dto.PositionDto;
import com.biz.crm.mdm.business.position.sdk.dto.PositionRelationDto;
import com.biz.crm.mdm.business.position.sdk.event.PositionEventListener;
import com.biz.crm.mdm.business.position.sdk.service.PositionVoService;
import com.biz.crm.mdm.business.position.sdk.vo.*;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 职位VO接口实现
 *
 * @author ning.zhang
 * @date 2021/10/13
 */
@Slf4j
@Service
public class PositionVoServiceImpl implements PositionVoService {

  @Autowired(required = false)
  @Lazy
  private List<PositionEventListener> listeners;
  @Autowired(required = false)
  private PositionService positionService;
  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private PositionRoleService positionRoleService;
  @Autowired(required = false)
  private PositionRepository positionRepository;
  @Autowired(required = false)
  private TreeRuleCodeStrategyHolder treeRuleCodeStrategyHolder;

  @Override
  public List<PositionVo> findByIdsOrCodes(List<String> ids, List<String> positionCodes) {
    if (CollectionUtils.isEmpty(positionCodes) && CollectionUtils.isEmpty(ids)) {
      return Lists.newArrayList();
    }
    List<PositionEntity> list = this.positionService.findByIdsOrCodes(ids, positionCodes);
    if (CollectionUtils.isEmpty(list)) {
      return Lists.newArrayList();
    }
    return this.convertEntityToVo(list);
  }

  @Override
  public List<PositionVo> findByLotPositionCodes(List<String> positionCodes) {
    return this.findByIdsOrCodes(Lists.newArrayList(), positionCodes);
  }

  @Override
  public List<PositionVo> findDetailsByIdsOrCodes(List<String> ids, List<String> positionCodes) {
    if (CollectionUtils.isEmpty(positionCodes) && CollectionUtils.isEmpty(ids)) {
      return Lists.newArrayList();
    }
    List<PositionVo> list = this.findByIdsOrCodes(ids, positionCodes);
    if (!CollectionUtils.isEmpty(list)) {
      buildRelationData(list);
    }
    return list;
  }

  @Override
  public List<PositionVo> findAllChildrenByCode(String positionCode) {
    if (StringUtils.isBlank(positionCode)) {
      return Lists.newArrayList();
    }
    List<PositionEntity> entities = this.positionService.findByIdsOrCodes(null, Lists.newArrayList(positionCode));
    if (CollectionUtils.isEmpty(entities)) {
      return Lists.newArrayList();
    }
    List<PositionEntity> childrenList = this.positionService.findAllChildrenByRuleCode(entities.get(0).getRuleCode());
    if (CollectionUtils.isEmpty(childrenList)) {
      return Lists.newArrayList();
    }
    return Lists.newArrayList(this.nebulaToolkitService.copyCollectionByWhiteList(childrenList, PositionEntity.class
        , PositionVo.class, HashSet.class, ArrayList.class));
  }

  @Override
  public List<PositionVo> findByParentCode(String positionCode) {
    if (StringUtils.isBlank(positionCode)) {
      return Lists.newArrayList();
    }
    List<PositionEntity> entities = this.positionRepository.findByParentCode(positionCode, TenantUtils.getTenantCode());
    if (CollectionUtils.isEmpty(entities)) {
      return Lists.newArrayList();
    }
    return Lists.newArrayList(this.nebulaToolkitService.copyCollectionByWhiteList(entities, PositionEntity.class
        , PositionVo.class, HashSet.class, ArrayList.class));
  }

  @Override
  public PositionVo create(PositionDto dto) {
    PositionEntity entity = this.positionService.create(dto);
    if (entity == null) {
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(entity, PositionVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public PositionVo update(PositionDto dto) {
    PositionEntity entity = this.positionService.update(dto);
    if (entity == null) {
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(entity, PositionVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public PositionVo findByPositionCode(String positionCode) {
    PositionEntity entity = this.positionService.findByPositionCode(positionCode);
    if (entity == null) {
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(entity, PositionVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public List<PositionVo> findAllParentByRoleCodes(List<String> roleCodes) {
    if (CollectionUtils.isEmpty(roleCodes)) {
      return Lists.newLinkedList();
    }
    String tenantCode = TenantUtils.getTenantCode();
    List<PositionEntity> entities = this.positionRepository.findByRoleCodes(roleCodes, tenantCode);
    if (CollectionUtils.isEmpty(entities)) {
      return Lists.newLinkedList();
    }
    List<String> ruleCodes = entities.stream().map(PositionEntity::getRuleCode).filter(StringUtils::isNotBlank).collect(Collectors.toList());
    TreeRuleCodeStrategy treeRuleCodeStrategy = this.treeRuleCodeStrategyHolder.getStrategy(null);
    Set<String> parentRuleCodes = treeRuleCodeStrategy.findParentRuleCodeByRuleCodes(PositionConstant.POSITION_RULE_CODE_LENGTH, ruleCodes);
    List<PositionEntity> parentEntities = this.positionRepository
        .findByRuleCodesAndEnableStatus(Lists.newArrayList(parentRuleCodes), null, TenantUtils.getTenantCode());
    if (CollectionUtils.isEmpty(parentEntities)) {
      return Lists.newLinkedList();
    }
    return (List<PositionVo>) this.nebulaToolkitService.copyCollectionByWhiteList(parentEntities, PositionEntity.class
        , PositionVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public List<PositionVo> findPositionsByRoleCodes(List<String> roleCodes) {
    if (CollectionUtils.isEmpty(roleCodes)) {
      return Lists.newLinkedList();
    }
    String tenantCode = TenantUtils.getTenantCode();
    List<PositionEntity> entities = this.positionRepository.findByRoleCodes(roleCodes, tenantCode);
    if (CollectionUtils.isEmpty(entities)) {
      return Lists.newLinkedList();
    }
    return (List<PositionVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, PositionEntity.class
        , PositionVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public Set<String> findPositionCodesByPositionLevelCodes(List<String> positionLevelCodes) {
    if (CollectionUtils.isEmpty(positionLevelCodes)) {
      return new HashSet<>();
    }
    List<PositionEntity> byPositionLevelCodes = this.positionRepository.findByPositionLevelCodes(positionLevelCodes, TenantUtils.getTenantCode());
    if (CollectionUtils.isEmpty(byPositionLevelCodes)) {
      return new HashSet<>();
    }
    return byPositionLevelCodes.stream().map(PositionEntity::getPositionCode).collect(Collectors.toSet());
  }

  @Override
  public Set<String> findRolesByPositionCodes(List<String> positionCodes) {
    if (CollectionUtils.isEmpty(positionCodes)) {
      return new HashSet<>(0);
    }
    List<PositionRoleEntity> byPositionCodes = this.positionRoleService.findByPositionCodes(positionCodes);
    if (CollectionUtils.isEmpty(byPositionCodes)) {
      return new HashSet<>(0);
    }
    return byPositionCodes.stream().map(PositionRoleEntity::getRoleCode).collect(Collectors.toSet());
  }

  /**
   * 通过职位编码查询全部上级
   *
   * @param positionCode
   * @return
   */
  @Override
  public List<PositionVo> findAllParentByPositionCode(String positionCode) {
    return this.findParent(positionCode, true);
  }

  /**
   * 通过职位编码查询直属上级
   *
   * @param positionCode
   * @return
   */
  @Override
  public List<PositionVo> findParentByPositionCode(String positionCode) {
    return this.findParent(positionCode, false);
  }

  /**
   * 查询有上级用户的职位
   *
   * @author huojia
   * @date 2022/12/26 16:47
   * @return java.util.List<com.biz.crm.mdm.business.position.sdk.vo.PositionVo>
   **/
  @Override
  public List<PositionVo> listParentUser() {
    List<PositionEntity> positionEntityList = positionRepository.listParentUser();
    if (CollectionUtils.isEmpty(positionEntityList)) {
      return null;
    }
    return (List<PositionVo>) this.nebulaToolkitService.copyCollectionByWhiteList(
            positionEntityList, PositionEntity.class, PositionVo.class, LinkedHashSet.class, ArrayList.class
    );
  }

  /**
   * 只更新对应职位的上下级关系（MDG用，前面方法已做校验，此处不做校验）
   *
   * @param positionDtoList
   * @author huojia
   * @date 2022/12/31 0:28
   **/
  @Override
  public void updateMdgBatch(List<PositionDto> positionDtoList) {
    if (CollectionUtils.isEmpty(positionDtoList)) {
      return;
    }
    // 查询对应职位
    List<String> positionCodeList = positionDtoList.stream().map(PositionDto::getPositionCode).collect(Collectors.toList());
    List<PositionEntity> positionEntityList = positionRepository.findByIdsOrCodes(null, positionCodeList, TenantUtils.getTenantCode());
    Map<String, PositionEntity> positionMap = positionEntityList.stream().collect(Collectors.toMap(PositionEntity::getPositionCode, Function.identity()));
    // 查询上级职位
    List<String> parentCodeList = positionDtoList.stream().map(PositionDto::getParentCode).collect(Collectors.toList());
    List<PositionEntity> parentEntityList = positionRepository.findByIdsOrCodes(null, parentCodeList, TenantUtils.getTenantCode());
    Map<String, PositionEntity> parentEntityMap = parentEntityList.stream().collect(Collectors.toMap(PositionEntity::getPositionCode, Function.identity()));
    List<PositionEntity> saveOrUpdatePositionList = new ArrayList<>();
    List<PositionEventVo> positionEventVos = new ArrayList<>();
    String tenantCode = TenantUtils.getTenantCode();
    positionDtoList.forEach(positionDto -> {
      if (!positionMap.containsKey(positionDto.getPositionCode())) {
        PositionEntity positionEntity = this.nebulaToolkitService.copyObjectByWhiteList(positionDto, PositionEntity.class, LinkedHashSet.class, ArrayList.class);
        positionEntity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        positionEntity.setTenantCode(tenantCode);
        positionEntity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        positionEntity.setLevelNum(1);
        saveOrUpdatePositionList.add(positionEntity);
      } else {
        PositionEntity positionEntity = positionMap.get(positionDto.getPositionCode());
        positionEntity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        positionEntity.setParentCode(positionDto.getParentCode());
        saveOrUpdatePositionList.add(positionEntity);
      }
      PositionEventVo positionEventVo = this.nebulaToolkitService.copyObjectByWhiteList(positionDto, PositionEventVo.class, LinkedHashSet.class, ArrayList.class);
      List<PositionRelationDto> relationDataList = positionDto.getRelationData();
      List<PositionRelationEventVo> relationEventVoList = new ArrayList<>();
      relationDataList.forEach(relationData -> {
        PositionRelationEventVo positionRelationEventVo = this.nebulaToolkitService.copyObjectByWhiteList(relationData, PositionRelationEventVo.class, LinkedHashSet.class, ArrayList.class);
        positionRelationEventVo.setRelationData(relationData.getRelationData());
        relationEventVoList.add(positionRelationEventVo);
      });
      positionEventVo.setRelationData(relationEventVoList);
      positionEventVos.add(positionEventVo);
    });
    Map<String, PositionEntity> saveMap = saveOrUpdatePositionList.stream().collect(Collectors.toMap(PositionEntity::getPositionCode, Function.identity()));
    saveOrUpdatePositionList.forEach(save -> {
      if (StringUtils.isEmpty(save.getParentCode())) {
        save.setLevelNum(1);
      } else {
        if (saveMap.containsKey(save.getParentCode())) {
          save.setLevelNum(saveMap.get(save.getParentCode()).getLevelNum() + 1);
        } else if (parentEntityMap.containsKey(save.getParentCode())) {
          save.setLevelNum(parentEntityMap.get(save.getParentCode()).getLevelNum() + 1);
        } else {
          save.setLevelNum(1);
        }
      }
    });
    positionRepository.saveOrUpdateBatch(saveOrUpdatePositionList);
    // 保存职位组织关系
    log.info(">>>>>>>>>>-----------------" + "本次拉取到的职位事务数据：" + positionEventVos.size() + "条--------------------<<<<<<<<<<<" + JSON.toJSONString(positionEventVos));
    for (PositionEventListener positionEventListener: listeners) {
      positionEventListener.onCreate(positionEventVos);
    }
  }

  /**
   * 根据mdg编码批量查询
   *
   * @param mdgPositionCodeList
   * @return java.util.List<com.biz.crm.mdm.business.position.sdk.vo.PositionVo>
   * @author huojia
   * @date 2023/1/4 0:23
   **/
  @Override
  public List<PositionVo> findByMdgPositionCodeList(List<String> mdgPositionCodeList) {
    if (CollectionUtils.isEmpty(mdgPositionCodeList)) {
      return Lists.newArrayList();
    }
    List<PositionEntity> byMdgPositionCodes = positionRepository.findByMdgPositionCodes(mdgPositionCodeList);
    if (CollectionUtils.isEmpty(byMdgPositionCodes)) {
      return Lists.newArrayList();
    }
    return (List<PositionVo>) this.nebulaToolkitService.copyCollectionByWhiteList(
            byMdgPositionCodes, PositionEntity.class, PositionVo.class, LinkedHashSet.class, ArrayList.class
    );
  }

  /**
   * 通过职位编码   isAll为true 代表查询全部上级  isAll为false  代表查询直属上级
   *
   * @param positionCode
   * @param isAll
   * @return
   */
  private List<PositionVo> findParent(String positionCode, Boolean isAll) {
    if (StringUtils.isBlank(positionCode) || ObjectUtils.isEmpty(isAll)) {
      return new ArrayList<>();
    }
    List<String> allRuleCodeList = new ArrayList<>();
    PositionEntity byPositionCode = this.positionService.findByPositionCode(positionCode);
    if (ObjectUtils.isEmpty(byPositionCode)){
      return new ArrayList<>();
    }
    String ruleCode = byPositionCode.getRuleCode();
    int length = ruleCode.length();
    int positionRuleCodeLength = PositionConstant.POSITION_RULE_CODE_LENGTH;
    if (isAll){
      //获取分割次数
      int count = length / positionRuleCodeLength;
      for (int i = 1; i <= count; i++) {
        String substring = ruleCode.substring(0, (positionRuleCodeLength * i));
        allRuleCodeList.add(substring);
      }
    }else {
      String substring = ruleCode.substring(0, length - positionRuleCodeLength);
      allRuleCodeList.add(substring);
    }
    List<PositionEntity> byRuleCodesAndEnableStatus = this.positionRepository.findByRuleCodesAndEnableStatus(allRuleCodeList, EnableStatusEnum.ENABLE.getCode(), TenantUtils.getTenantCode());
    if (CollectionUtils.isEmpty(byRuleCodesAndEnableStatus)){
      return new ArrayList<>(0);
    }
    return (List<PositionVo>) this.nebulaToolkitService.copyCollectionByWhiteList(byRuleCodesAndEnableStatus,PositionEntity.class,PositionVo.class,HashSet.class,ArrayList.class);
  }

  /**
   * 职位实体列表转VO列表
   *
   * @param entities 职位实体列表
   * @return 职位VO列表
   */
  private List<PositionVo> convertEntityToVo(List<PositionEntity> entities) {
    List<PositionVo> list = Lists.newArrayList(this.nebulaToolkitService.copyCollectionByWhiteList(entities, PositionEntity.class
        , PositionVo.class, HashSet.class, ArrayList.class));
    List<PositionRoleEntity> positionRoles = this.positionRoleService
        .findByPositionCodes(list.stream().map(PositionVo::getPositionCode).collect(Collectors.toList()));
    if (!CollectionUtils.isEmpty(positionRoles)) {
      Map<String, List<String>> positionRoleMap = positionRoles.stream().collect(Collectors
          .groupingBy(PositionRoleEntity::getPositionCode, Collectors.mapping(PositionRoleEntity::getRoleCode, Collectors.toList())));
      list.forEach(positionVo -> {
        positionVo.setRoleList(positionRoleMap.get(positionVo.getPositionCode()));
        positionVo.setPositionRoles(Lists.newArrayList(this.nebulaToolkitService.copyCollectionByWhiteList(positionRoles, PositionRoleEntity.class
                , PositionRoleVo.class, HashSet.class, ArrayList.class)));
      });
    }
    return list;
  }

  /**
   * 封装职位上级模块关联数据
   *
   * @param list 待封装职位列表
   */
  private void buildRelationData(List<PositionVo> list) {
    if (CollectionUtils.isEmpty(list) || CollectionUtils.isEmpty(listeners)) {
      return;
    }
    List<String> positionCodes = list.stream().map(PositionVo::getPositionCode).collect(Collectors.toList());
    List<PositionRelationVo> relationList = Lists.newArrayList();
    for (PositionEventListener positionEventListener : listeners) {
      List<PositionRelationVo> dataList = positionEventListener.onRequestByPositionCodes(positionCodes);
      if (CollectionUtils.isEmpty(dataList)) {
        continue;
      }
      relationList.addAll(dataList);
    }
    if (!CollectionUtils.isEmpty(relationList)) {
      Map<String, List<PositionRelationVo>> relationMap = relationList.stream().collect(Collectors
          .groupingBy(PositionRelationVo::getPositionCode));
      list.forEach(positionVo -> {
        positionVo.setRelationData(relationMap.get(positionVo.getPositionCode()));
      });
    }
  }
}
