package com.biz.crm.mdm.business.position.local.repository;

import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.mdm.business.position.local.entity.PositionRoleEntity;
import com.biz.crm.mdm.business.position.local.mapper.PositionRoleMapper;
import org.springframework.stereotype.Component;

import java.util.List;

/**
 * 职位-角色关联表的数据库访问类 {@link PositionRoleEntity}
 *
 * @author ning.zhang
 * @date 2021/9/28
 */
@Component
public class PositionRoleRepository extends ServiceImpl<PositionRoleMapper, PositionRoleEntity> {

  /**
   * 通过职位编码获取职位角色列表
   *
   * @param positionCode 职位编码
   * @param tenantCode   租户编码
   * @return 职位角色列表
   */
  public List<PositionRoleEntity> findByPositionCode(String positionCode, String tenantCode) {
    return this.lambdaQuery()
            .eq(PositionRoleEntity::getTenantCode, tenantCode)
            .eq(PositionRoleEntity::getPositionCode, positionCode)
            .list();
  }

  /**
   * 通过职位编码集合获取职位角色列表
   *
   * @param positionCodes 职位编码集合
   * @param tenantCode    租户编码
   * @return 职位角色列表
   */
  public List<PositionRoleEntity> findByPositionCodes(List<String> positionCodes, String tenantCode) {
    return this.lambdaQuery()
            .eq(PositionRoleEntity::getTenantCode, tenantCode)
            .in(PositionRoleEntity::getPositionCode, positionCodes)
            .list();
  }

  /**
   * 通过职位编码集合删除角色关联关系
   *
   * @param positionCodes 职位编码集合
   * @param tenantCode    租户编码
   */
  public void deleteByPositionCodes(List<String> positionCodes, String tenantCode) {
    this.lambdaUpdate()
            .eq(PositionRoleEntity::getTenantCode, tenantCode)
            .in(PositionRoleEntity::getPositionCode, positionCodes)
            .remove();
  }

  /**
   * 通过职位编码和角色编码集合获取职位角色关联实体列表
   *
   * @param positionCode 职位编码
   * @param roleCodes    角色编码集合
   * @param tenantCode   租户编码
   * @return 职位角色关联列表
   */
  public List<PositionRoleEntity> findByPositionCodeAndRoleCodes(String positionCode, List<String> roleCodes
          , String tenantCode) {
    return super.lambdaQuery()
            .eq(PositionRoleEntity::getTenantCode, tenantCode)
            .eq(PositionRoleEntity::getPositionCode, positionCode)
            .in(PositionRoleEntity::getRoleCode, roleCodes)
            .list();
  }

  /**
   * 通过角色编码集合获取职位角色关联实体列表
   *
   * @param roleCodes  角色编码集合
   * @param tenantCode 租户编码
   * @return 职位角色关联列表
   */
  public List<PositionRoleEntity> findByRoleCodes(List<String> roleCodes, String tenantCode) {
    return super.lambdaQuery()
            .eq(PositionRoleEntity::getTenantCode, tenantCode)
            .in(PositionRoleEntity::getRoleCode, roleCodes)
            .list();
  }

  /**
   * 通过职位编码集合和角色编码获取职位角色关联列表
   *
   * @param positionCodes 职位编码集合
   * @param roleCode      角色编码
   * @param tenantCode    租户编码
   * @return 职位角色关联列表
   */
  public List<PositionRoleEntity> findByPositionCodesAndRoleCode(List<String> positionCodes, String roleCode, String tenantCode) {
    return this.lambdaQuery()
            .eq(PositionRoleEntity::getTenantCode, tenantCode)
            .in(PositionRoleEntity::getPositionCode, positionCodes)
            .eq(PositionRoleEntity::getRoleCode, roleCode)
            .list();
  }
}
